package followsrpcserver

import (
	"log"
	"strings"
	"time"

	"context"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	pb "code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/twitchtv/twirp"
)

// ListFollows that a user is following
func (s *Server) ListFollows(ctx context.Context, req *pb.FollowsReq) (*pb.FollowsResp, error) {
	req.UserId = strings.TrimSpace(req.UserId)
	if req.UserId == "" {
		return nil, twirp.RequiredArgumentError("user_id")
	}

	if req.Limit < 0 {
		return nil, twirp.InvalidArgumentError("limit", "can not be negative")
	}
	if req.Limit == 0 {
		req.Limit = followsReqDefaultLimit
		err := s.Stats.Inc("listfollows.length_0", 1, 0.1)
		if err != nil {
			log.Print(err)
		}
	}

	if req.Offset < 0 {
		return nil, twirp.InvalidArgumentError("offset", "can not be negative")
	}
	err := s.Stats.TimingDuration("listfollows.req_limit", time.Duration(req.Limit)*time.Millisecond, 0.1)
	if err != nil {
		log.Print(err)
	}

	direction := strings.ToLower(req.OrderByFollowedAt.String()) // "asc" (default) or "desc"

	gdbResp, err := s.Backend.ListFollows(ctx, req.UserId, req.Cursor, int(req.Limit), int(req.Offset), direction)
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	}

	return convertFollowsResp(gdbResp), nil
}

func convertFollowsResp(response *graphdbFulton.EdgeListResponse) *pb.FollowsResp {
	followers := []*pb.FollowResp{}
	for _, edge := range response.Edges {
		followers = append(followers, convertFollows(edge))
	}
	return &pb.FollowsResp{
		Follows: followers,
		Cursor:  response.Cursor,
	}
}

func convertFollows(edge *graphdbFulton.LoadedCursoredEdge) *pb.FollowResp {
	t := time.Unix(edge.Edge.Data.CreatedAt.Seconds, int64(edge.Edge.Data.CreatedAt.Nanos))

	return &pb.FollowResp{
		FromUserId:         edge.Edge.Edge.From.Id,
		TargetUserId:       edge.Edge.Edge.To.Id,
		FollowedAt:         t.Format(time.RFC3339),
		BlockNotifications: isBlockingNotification(edge.Edge),
		Cursor:             edge.Cursor,
	}
}
