package followsrpcserver_test

import (
	"context"
	"errors"
	"time"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/backend"
	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/mock"
)

func (s *FollowsSuite) TestListFollowsSuccess() {
	mockResp := buildBackendListFollowsResp("1")
	s.backendMock.On("ListFollows", mock.Anything, "1", "", defaultLimit, 0, defaultOrderByFollowedAt).
		Return(mockResp, nil) // followers of "1"

	resp, err := s.DoListFollows("1")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal(len(mockResp.Edges), len(resp.Follows), "expect Follows to be populated with data")
	s.Assert().Equal(mockResp.Cursor, resp.Cursor, "expect Cursor to have the same value as the backend response")
}

func (s *FollowsSuite) TestListFollowsBackendServerError() {
	s.backendMock.On("ListFollows", mock.Anything, "1", "", defaultLimit, 0, defaultOrderByFollowedAt).
		Return(nil, errors.New("trololol")) // oops, something went wrong

	resp, err := s.DoListFollows("1")

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error internal: trololol")
}

func (s *FollowsSuite) TestListFollowsWithPaginationSuccess() {
	cursor := "cursor-kakaka"
	limit := 33
	offset := 11
	direction := "desc"
	mockResp := buildBackendListFollowsResp("1")
	s.backendMock.On("ListFollows", mock.Anything, "1", cursor, limit, offset, direction).
		Return(mockResp, nil)

	req := &followsrpc.FollowsReq{UserId: "1", Cursor: cursor, Limit: int32(limit), Offset: int32(offset), OrderByFollowedAt: followsrpc.FollowsReq_DESC}
	resp, err := s.followsService.ListFollows(context.Background(), req)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().NotNil(resp, "expect resp to not be nil")
}

func (s *FollowsSuite) TestListFollowsNegativeLimitError() {
	req := &followsrpc.FollowsReq{UserId: "1", Limit: -1}
	resp, err := s.followsService.ListFollows(context.Background(), req)

	s.backendMock.AssertNotCalled(s.T(), "ListFollows") // validation should fail before making backend requests
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: limit can not be negative")
}

func (s *FollowsSuite) TestListFollowsNegativeOffsetError() {
	req := &followsrpc.FollowsReq{UserId: "1", Offset: -1}
	resp, err := s.followsService.ListFollows(context.Background(), req)

	s.backendMock.AssertNotCalled(s.T(), "ListFollows") // validation should fail before making backend requests
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: offset can not be negative")
}

func (s *FollowsSuite) TestListFollowsEmptyUserIDError() {
	resp, err := s.DoListFollows("")

	s.backendMock.AssertNotCalled(s.T(), "ListFollows") // validation should fail before making backend requests
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: user_id is required")
}

// Helpers

func buildBackendListFollowsResp(userID string) *graphdbFulton.EdgeListResponse {
	createdAt, err := ptypes.TimestampProto(time.Now())
	if err != nil {
		return nil
	}
	return &graphdbFulton.EdgeListResponse{
		Edges: []*graphdbFulton.LoadedCursoredEdge{
			{
				Edge: &graphdbFulton.LoadedEdge{
					Edge: &graphdbFulton.Edge{
						From: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   "222",
						},
						To: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   userID,
						},
						Type: backend.FollowsKind,
					},
					Data: &graphdbFulton.Data{
						CreatedAt: createdAt,
						Data: &graphdbFulton.DataBag{
							Bools: map[string]bool{
								"block_notifications": false,
							},
						},
					},
				},
				Cursor: "inner cursor1",
			},
			{
				Edge: &graphdbFulton.LoadedEdge{
					Edge: &graphdbFulton.Edge{
						From: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   "333",
						},
						To: &graphdbFulton.Node{
							Type: backend.UserKind,
							Id:   userID,
						},
						Type: backend.FollowsKind,
					},
					Data: &graphdbFulton.Data{
						CreatedAt: createdAt,
						Data: &graphdbFulton.DataBag{
							Bools: map[string]bool{
								"block_notifications": false,
							},
						},
					},
				},
				Cursor: "inner cursor2",
			},
		},
		Cursor: "A Cursor!",
	}
}

func (s *FollowsSuite) DoListFollows(userID string) (*followsrpc.FollowsResp, error) {
	req := &followsrpc.FollowsReq{UserId: userID} // other values are defaults
	return s.followsService.ListFollows(context.Background(), req)
}
