package followsrpcserver

import (
	"context"
	"errors"
	"strconv"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/discovery/experiments"
	"code.justin.tv/discovery/experiments/experiment"
	"code.justin.tv/feeds/following-service/backend"
	"github.com/twitchtv/twirp"
	"github.com/twitchtv/twirp/hooks/statsd"
)

// Server implements Follows service
type Server struct {
	Backend backend.Backender
	Stats   statsd.Statter
	Exps    experiments.Experiments
}

// FollowsReq.Limit default value (0 == default)
const followsReqDefaultLimit = 20
const blockNotificationsKey = "block_notifications"

func validateUserID(userID string) error {
	if id, err := strconv.ParseInt(userID, 10, 64); err != nil {
		return errors.New("UserID must be integer")
	} else if id <= 0 {
		return errors.New("UserID must be positive integer")
	}
	return nil
}

func (s *Server) doRollout(expName string, expID string, userID string) bool {
	group, err := s.Exps.Treat(experiment.UserType, expName, expID, userID)
	if err != nil {
		return false
	}

	switch group {
	case "variant":
		return true
	default:
		return false
	}
}

func isBlockingNotification(edge *graphdbFulton.LoadedEdge) bool {
	databag := edge.Data.Data
	isBlockingNotification := false

	if blockingNotificationVal, ok := databag.Bools[blockNotificationsKey]; ok {
		isBlockingNotification = blockingNotificationVal
	}
	return isBlockingNotification
}

// NewServerHooks initializes a ServerHooks with no-op callbacks. They
// pass through their context unchanged.
func NewServerHooks() *twirp.ServerHooks {
	return &twirp.ServerHooks{
		RequestReceived: func(ctx context.Context) (context.Context, error) {
			return ctx, nil
		},
		RequestRouted: func(ctx context.Context) (context.Context, error) {
			return ctx, nil
		},
		ResponsePrepared: func(ctx context.Context) context.Context {
			return ctx
		},
		ResponseSent: func(ctx context.Context) {},
		Error: func(ctx context.Context, _ twirp.Error) context.Context {
			return ctx
		},
	}
}

// NewInternalErrorsLoggerHook creates and returns a twirp server hook which, on error,
// logs the error to rollbar if it's an internal error.
func NewInternalErrorsLoggerHook(logger errorlogger.ErrorLogger) *twirp.ServerHooks {
	hooks := NewServerHooks()
	hooks.Error = func(ctx context.Context, twerr twirp.Error) context.Context {
		if twerr.Code() == twirp.Internal {
			logger.Error(twerr)
		}
		return ctx
	}
	return hooks
}
