package followsrpcserver_test

import (
	"context"
	"net/http"
	"testing"

	"code.justin.tv/feeds/following-service/mocks"
	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"code.justin.tv/feeds/following-service/rpc/followsrpcserver"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/suite"
	"github.com/twitchtv/twirp"
)

var defaultLimit = 20
var defaultOffset = 0
var defaultOrderByFollowedAt = "asc"

type FollowsSuite struct {
	suite.Suite
	backendMock    *mocks.Backender
	followsService followsrpc.Follows
}

func (s *FollowsSuite) SetupTest() {
	s.backendMock = &mocks.Backender{}
	noopStats, err := statsd.NewNoopClient()
	noopExp := mocks.NoopExperimentClient()
	s.Assert().NoError(err)

	s.followsService = &followsrpcserver.Server{Stats: noopStats, Exps: noopExp, Backend: s.backendMock}
}

// Testify's test runner
func TestFollowsSuite(t *testing.T) {
	suite.Run(t, new(FollowsSuite))
}

func (s *FollowsSuite) TestErrorHookInternalErrorLogged() {
	logger := &FakeErrorLogger{}
	hook := followsrpcserver.NewInternalErrorsLoggerHook(logger)
	err := twirp.InternalErrorWith(hystrix.ErrTimeout)

	hook.Error(context.Background(), err)

	s.Equal(1, logger.errorsLogged, "The internal twirp error should have been logged")
	s.EqualError(logger.lastErrorLogged, "twirp error internal: hystrix: timeout")
}

func (s *FollowsSuite) TestErrorHookValidationErrorNotLogged() {
	logger := &FakeErrorLogger{}
	hook := followsrpcserver.NewInternalErrorsLoggerHook(logger)
	err := twirp.RequiredArgumentError("from_user_id")

	hook.Error(context.Background(), err)

	s.Zero(logger.errorsLogged, "The client error should not have been logged")
	s.Nil(logger.lastErrorLogged, "No errors should have been logged.")
}

type FakeErrorLogger struct {
	errorsLogged    int
	lastErrorLogged error
}

func (l *FakeErrorLogger) Info(msg string) {}
func (l *FakeErrorLogger) Error(err error) {
	l.errorsLogged++
	l.lastErrorLogged = err
}
func (l *FakeErrorLogger) RequestError(*http.Request, error)       {}
func (l *FakeErrorLogger) RequestPanic(*http.Request, interface{}) {}
