package followsrpcserver

import (
	"context"
	"strings"
	"time"

	graphdbFulton "code.justin.tv/amzn/TwitchVXGraphDBECSTwirp"
	"code.justin.tv/feeds/following-service/backend"
	pb "code.justin.tv/feeds/following-service/rpc/followsrpc"
	usersservice "code.justin.tv/web/users-service/client"
	"github.com/twitchtv/twirp"
)

// UpsertFollow creates or updates a follow
func (s *Server) UpsertFollow(ctx context.Context, req *pb.UpsertFollowReq) (*pb.FollowResp, error) {
	req.FromUserId = strings.TrimSpace(req.FromUserId)
	req.TargetUserId = strings.TrimSpace(req.TargetUserId)

	// Validations
	if req.FromUserId == "" {
		return nil, twirp.RequiredArgumentError("from_user_id")
	}
	if req.TargetUserId == "" {
		return nil, twirp.RequiredArgumentError("target_user_id")
	}
	if err := validateUserID(req.FromUserId); err != nil {
		return nil, twirp.InvalidArgumentError("from_user_id", err.Error())
	}
	if err := validateUserID(req.TargetUserId); err != nil {
		return nil, twirp.InvalidArgumentError("target_user_id", err.Error())
	}
	if req.TargetUserId == req.FromUserId {
		return nil, twirp.InvalidArgumentError("target_user_id", "can not be from_user_id")
	}

	gdbResp, err := s.Backend.GetFollow(ctx, req.FromUserId, req.TargetUserId)
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	}

	var updateResp *graphdbFulton.EdgeGetResponse
	blockNotificationsNeedsUpdate := false
	followExists := (gdbResp != nil && gdbResp.Edge != nil)
	if followExists {
		blockNotificationsNeedsUpdate = isBlockingNotification(gdbResp.Edge) != req.BlockNotifications
	}

	if !followExists {
		err = s.Backend.FollowUser(ctx, req.FromUserId, req.TargetUserId, req.BlockNotifications)
	} else if blockNotificationsNeedsUpdate {
		updateResp, err = s.performUpdate(ctx, req, gdbResp)
	}
	if err != nil {
		if usersservice.IsUserNotFound(err) {
			return nil, twirp.InvalidArgumentError("from_user_id", err.Error())
		}
		if backend.IsFollowLimitExceeded(err) {
			return nil, twirp.NewError(twirp.ResourceExhausted, err.Error())
		}
		if backend.IsUserSuspended(err) || backend.IsUserDeleted(err) {
			return nil, twirp.NewError(twirp.FailedPrecondition, err.Error())
		}
		if backend.IsUserBlocked(err) || backend.IsUserAFollowBot(err) {
			return nil, twirp.NewError(twirp.PermissionDenied, err.Error())
		}
		return nil, twirp.InternalErrorWith(err)
	}

	// Follow was updated
	if updateResp != nil {
		t := time.Unix(gdbResp.Edge.Data.CreatedAt.Seconds, int64(gdbResp.Edge.Data.CreatedAt.Nanos))
		return &pb.FollowResp{
			FromUserId:         gdbResp.Edge.Edge.From.Id,
			TargetUserId:       gdbResp.Edge.Edge.To.Id,
			FollowedAt:         t.Format(time.RFC3339),
			BlockNotifications: isBlockingNotification(gdbResp.Edge),
		}, nil
	}

	// Either follow was unchanged or new follow was created
	return &pb.FollowResp{
		FromUserId:         req.FromUserId,
		TargetUserId:       req.TargetUserId,
		FollowedAt:         time.Now().Format(time.RFC3339),
		BlockNotifications: req.BlockNotifications,
		Cursor:             "",
	}, nil
}

func (s *Server) performUpdate(ctx context.Context, req *pb.UpsertFollowReq, gdbResp *graphdbFulton.EdgeGetResponse) (*graphdbFulton.EdgeGetResponse, error) {
	err := s.Backend.UpdateFollow(ctx, req.FromUserId, req.TargetUserId, req.BlockNotifications)
	if err != nil {
		return nil, err
	}

	// save a call to cohesion by updating the previous response in memory
	gdbResp.Edge.Data.Data.Bools[blockNotificationsKey] = req.BlockNotifications
	return gdbResp, nil
}
