package followsrpcserver_test

import (
	"context"
	"errors"

	"github.com/stretchr/testify/mock"

	"code.justin.tv/feeds/following-service/rpc/followsrpc"
)

func (s *FollowsSuite) TestUpsertFollowNewFollowWithoutNotificationsSuccess() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(nil, nil).Once()
	s.backendMock.On("FollowUser", mock.Anything, "1", "2", false).Return(nil)
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(buildBackendGetFollowResponse("1", "2", false), nil)

	resp, err := s.DoUpsertFollow("1", "2", false)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal("1", resp.FromUserId, "expect FromUserId to be 1")
	s.Assert().Equal("2", resp.TargetUserId, "expect TargetUserId to be 2")
	s.Assert().Equal(false, resp.BlockNotifications, "expect BlockNotifications to be false")
}

func (s *FollowsSuite) TestUpsertFollowNewFollowWithNotificationsSuccess() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(nil, nil).Once()
	s.backendMock.On("FollowUser", mock.Anything, "1", "2", true).Return(nil)
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(buildBackendGetFollowResponse("1", "2", true), nil)

	resp, err := s.DoUpsertFollow("1", "2", true)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal("1", resp.FromUserId, "expect FromUserId to be 1")
	s.Assert().Equal("2", resp.TargetUserId, "expect TargetUserId to be 2")
	s.Assert().Equal(true, resp.BlockNotifications, "expect BlockNotifications to be true")
}

func (s *FollowsSuite) TestUpsertFollowUpdateFollowSuccess() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(buildBackendGetFollowResponse("1", "2", true), nil).Once()
	s.backendMock.On("UpdateFollow", mock.Anything, "1", "2", false).Return(nil)

	resp, err := s.DoUpsertFollow("1", "2", false)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().NoError(err)
	s.Assert().Equal("1", resp.FromUserId, "expect FromUserId to be 1")
	s.Assert().Equal("2", resp.TargetUserId, "expect TargetUserId to be 2")
	s.Assert().Equal(false, resp.BlockNotifications, "expect BlockNotifications to be false")
}

func (s *FollowsSuite) TestUpsertFollowEmptyFromUserIDErrors() {
	resp, err := s.DoUpsertFollow("", "2", false)

	s.backendMock.AssertNotCalled(s.T(), "GetFollow") // validation fails even before making a backend request
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: from_user_id is required")
}

func (s *FollowsSuite) TestUpsertFollowEmptyTargetUserIDErrors() {
	resp, err := s.DoUpsertFollow("1", "", false)

	s.backendMock.AssertNotCalled(s.T(), "GetFollow") // validation fails even before making a backend request
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: target_user_id is required")
}

func (s *FollowsSuite) TestUpsertFollowSameUserAndTargetUserIDsErrors() {
	resp, err := s.DoUpsertFollow("1", "1", false)

	s.backendMock.AssertNotCalled(s.T(), "GetFollow") // validation fails even before making a backend request
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().EqualError(err, "twirp error invalid_argument: target_user_id can not be from_user_id")
}

func (s *FollowsSuite) TestUpsertFollowGetFollowFailure() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(nil, errors.New("an error occurred."))

	resp, err := s.DoUpsertFollow("1", "2", false)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().Error(err)
}

func (s *FollowsSuite) TestUpsertFollowFollowUserFailure() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(nil, nil)
	s.backendMock.On("FollowUser", mock.Anything, "1", "2", false).Return(errors.New("an error occurred."))

	resp, err := s.DoUpsertFollow("1", "2", false)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().Error(err)
}

func (s *FollowsSuite) TestUpsertFollowUpdateFollowFailure() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(buildBackendGetFollowResponse("1", "2", true), nil)
	s.backendMock.On("UpdateFollow", mock.Anything, "1", "2", false).Return(errors.New("an error occurred."))

	resp, err := s.DoUpsertFollow("1", "2", false)

	s.backendMock.AssertExpectations(s.T())
	s.Assert().Nil(resp, "expect resp to be nil")
	s.Assert().Error(err)
}

func (s *FollowsSuite) TestUpsertFollowNoopUpdate() {
	s.backendMock.On("GetFollow", mock.Anything, "1", "2").Return(buildBackendGetFollowResponse("1", "2", true), nil).Once()

	resp, err := s.DoUpsertFollow("1", "2", true)

	s.backendMock.AssertNotCalled(s.T(), "UpdateFollow") // no update is required
	s.Assert().NotNil(resp, "expect a response")
	s.Assert().NoError(err)
}

func (s *FollowsSuite) DoUpsertFollow(fromUserID string, targetUserID string, blockNotifications bool) (*followsrpc.FollowResp, error) {
	req := &followsrpc.UpsertFollowReq{FromUserId: fromUserID, TargetUserId: targetUserID, BlockNotifications: blockNotifications}
	return s.followsService.UpsertFollow(context.Background(), req)
}
