package sfxhystrix

import (
	"sync"
	"time"

	"sync/atomic"

	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"github.com/afex/hystrix-go/hystrix/metric_collector"
	"github.com/signalfx/golib/datapoint"
	"github.com/signalfx/golib/sfxclient"
)

type hystrixMetricCollector struct {
	Attempts          int64
	Errors            int64
	Successes         int64
	Failures          int64
	Rejects           int64
	ShortCircuits     int64
	Timeouts          int64
	FallbackSuccesses int64
	FallbackFailures  int64
	dims              map[string]string

	mu             sync.RWMutex
	totalDurations *sfxclient.RollingBucket
	runDurations   *sfxclient.RollingBucket
}

func (h *hystrixMetricCollector) IncrementAttempts() {
	atomic.AddInt64(&h.Attempts, 1)
}

func (h *hystrixMetricCollector) IncrementErrors() {
	atomic.AddInt64(&h.Errors, 1)
}

func (h *hystrixMetricCollector) IncrementSuccesses() {
	atomic.AddInt64(&h.Successes, 1)
}

func (h *hystrixMetricCollector) IncrementFailures() {
	atomic.AddInt64(&h.Failures, 1)
}

func (h *hystrixMetricCollector) IncrementRejects() {
	atomic.AddInt64(&h.Rejects, 1)
}

func (h *hystrixMetricCollector) IncrementShortCircuits() {
	atomic.AddInt64(&h.ShortCircuits, 1)
}

func (h *hystrixMetricCollector) IncrementTimeouts() {
	atomic.AddInt64(&h.Timeouts, 1)
}

func (h *hystrixMetricCollector) IncrementFallbackSuccesses() {
	atomic.AddInt64(&h.FallbackSuccesses, 1)
}

func (h *hystrixMetricCollector) IncrementFallbackFailures() {
	atomic.AddInt64(&h.FallbackFailures, 1)
}

var _ metricCollector.MetricCollector = &hystrixMetricCollector{}

func (h *hystrixMetricCollector) UpdateTotalDuration(timeSinceStart time.Duration) {
	h.mu.RLock()
	h.totalDurations.Add(timeSinceStart.Seconds())
	h.mu.RUnlock()
}

func (h *hystrixMetricCollector) UpdateRunDuration(runDuration time.Duration) {
	h.mu.RLock()
	h.runDurations.Add(runDuration.Seconds())
	h.mu.RUnlock()
}

func (h *hystrixMetricCollector) Reset() {
	atomic.StoreInt64(&h.Attempts, 0)
	atomic.StoreInt64(&h.Errors, 0)
	atomic.StoreInt64(&h.Successes, 0)
	atomic.StoreInt64(&h.Failures, 0)
	atomic.StoreInt64(&h.Rejects, 0)
	atomic.StoreInt64(&h.ShortCircuits, 0)
	atomic.StoreInt64(&h.Timeouts, 0)
	atomic.StoreInt64(&h.FallbackSuccesses, 0)
	atomic.StoreInt64(&h.FallbackFailures, 0)

	h.mu.Lock()
	h.totalDurations = sfxclient.NewRollingBucket("hystrix.total_durations", h.dims)
	h.runDurations = sfxclient.NewRollingBucket("hystrix.run_durations", h.dims)
	h.mu.Unlock()
}

func (h *hystrixMetricCollector) Datapoints() []*datapoint.Datapoint {
	h.mu.RLock()
	defer h.mu.RUnlock()
	ret := h.totalDurations.Datapoints()
	ret = append(ret, h.runDurations.Datapoints()...)
	ret = append(ret, []*datapoint.Datapoint{
		sfxclient.CumulativeP("hystrix.attempts", h.dims, &h.Attempts),
		sfxclient.CumulativeP("hystrix.errors", h.dims, &h.Errors),
		sfxclient.CumulativeP("hystrix.successes", h.dims, &h.Successes),
		sfxclient.CumulativeP("hystrix.failures", h.dims, &h.Failures),
		sfxclient.CumulativeP("hystrix.rejects", h.dims, &h.Rejects),
		sfxclient.CumulativeP("hystrix.short_circuits", h.dims, &h.ShortCircuits),
		sfxclient.CumulativeP("hystrix.timeouts", h.dims, &h.Timeouts),
		sfxclient.CumulativeP("hystrix.fallback_successes", h.dims, &h.FallbackSuccesses),
		sfxclient.CumulativeP("hystrix.fallback_failures", h.dims, &h.FallbackFailures),
	}...)
	return ret
}

// Setup adds a hystrix metric collector that also sends to SignalFx
func Setup(s *sfxstatsd.SetupResult) {
	metricCollector.Registry.Register(func(name string) metricCollector.MetricCollector {
		ret := &hystrixMetricCollector{
			dims: map[string]string{
				"hystrixcircuit": name,
			},
		}
		s.Scheduler.AddCallback(ret)
		ret.Reset()
		return ret
	})
}
