package models

import (
	"fmt"
	"time"

	"code.justin.tv/common/yimg"
)

type ChannelProperties struct {
	ID                  uint64     `json:"id"`
	Name                string     `json:"name"`
	DirectoryHidden     bool       `json:"directory_hidden"`
	Broadcaster         *string    `json:"broadcaster"`
	BroadcasterLanguage *string    `json:"broadcaster_language"`
	BroadcasterSoftware *string    `json:"broadcaster_software"`
	CreatedOn           *time.Time `json:"created_on"`
	Game                *string    `json:"game"`
	GameID              *uint64    `json:"game_id"`
	Mature              *bool      `json:"mature"`
	Status              *string    `json:"status"`
	Title               *string    `json:"title"`
	UpdatedOn           *time.Time `json:"updated_on"`
	ViewsCount          *uint64    `json:"views_count"`

	BlockNonPublicAds *bool `json:"block_non_public_ads"`
	PrerollsDisabled  *bool `json:"prerolls_disabled"`
	PostrollsDisabled *bool `json:"postrolls_disabled"`
	FightAdBlock      *bool `json:"fight_ad_block"`

	LastBroadcastTime        *time.Time `json:"last_broadcast_time"`
	LastBroadcastID          *string    `json:"last_broadcast_id"`
	LastLiveNotificationSent *time.Time `json:"last_live_notification_sent"`

	About           *string `json:"about"`
	RedirectChannel *string `json:"redirect_channel"`
	PrimaryTeamID   *uint64 `json:"primary_team_id"`
	DisableChat     *bool   `json:"disable_chat"`

	ChannelOfflineImage          yimg.Images `json:"channel_offline_image"`
	ProfileBanner                yimg.Images `json:"profile_banner"`
	ProfileBannerBackgroundColor *string     `json:"profile_banner_background_color"`

	DmcaViolation           *bool      `json:"dmca_violation" `
	TermsOfServiceViolation *bool      `json:"terms_of_service_violation" `
	DeletedOn               *time.Time `json:"deleted_on" `
}

type UpdateChannelProperties struct {
	ID                  uint64     `json:"id" validate:"nonzero"`
	DirectoryHidden     *bool      `json:"directory_hidden"`
	Broadcaster         *string    `json:"broadcaster"`
	BroadcasterLanguage *string    `json:"broadcaster_language"`
	BroadcasterSoftware *string    `json:"broadcaster_software"`
	Game                *string    `json:"game"`
	GameID              *uint64    `json:"game_id"`
	Mature              *bool      `json:"mature"`
	Status              *string    `json:"status"`
	Title               *string    `json:"title"`
	UpdatedOn           *time.Time `json:"updated_on"`
	ViewsCount          *uint64    `json:"views_count"`

	BlockNonPublicAds *bool `json:"block_non_public_ads"`
	PrerollsDisabled  *bool `json:"prerolls_disabled"`
	PostrollsDisabled *bool `json:"postrolls_disabled"`
	FightAdBlock      *bool `json:"fight_ad_block"`

	LastBroadcastTime        *time.Time `json:"last_broadcast_time"`
	LastBroadcastID          *string    `json:"last_broadcast_id"`
	LastLiveNotificationSent *time.Time `json:"last_live_notification_sent"`

	About           *string `json:"about"`
	RedirectChannel *string `json:"redirect_channel"`
	PrimaryTeamID   *uint64 `json:"primary_team_id"`
	DisableChat     *bool   `json:"disable_chat"`
}

func (up *UpdateChannelProperties) FillFromProperties(p *ChannelProperties) *UpdateChannelProperties {
	uup := &UpdateChannelProperties{
		ID:                       up.ID,
		DirectoryHidden:          up.DirectoryHidden,
		Broadcaster:              up.Broadcaster,
		BroadcasterLanguage:      up.BroadcasterLanguage,
		BroadcasterSoftware:      up.BroadcasterSoftware,
		Game:                     up.Game,
		GameID:                   up.GameID,
		Mature:                   up.Mature,
		Status:                   up.Status,
		Title:                    up.Title,
		UpdatedOn:                up.UpdatedOn,
		ViewsCount:               up.ViewsCount,
		BlockNonPublicAds:        up.BlockNonPublicAds,
		PrerollsDisabled:         up.PrerollsDisabled,
		PostrollsDisabled:        up.PostrollsDisabled,
		FightAdBlock:             up.FightAdBlock,
		LastBroadcastTime:        up.LastBroadcastTime,
		LastBroadcastID:          up.LastBroadcastID,
		LastLiveNotificationSent: up.LastLiveNotificationSent,
		About:           up.About,
		RedirectChannel: up.RedirectChannel,
		PrimaryTeamID:   up.PrimaryTeamID,
		DisableChat:     up.DisableChat,
	}

	defaultBool := false
	if uup.DirectoryHidden == nil {
		uup.DirectoryHidden = &p.DirectoryHidden
	}

	if uup.Broadcaster == nil {
		uup.Broadcaster = p.Broadcaster
	}

	if uup.BroadcasterLanguage == nil {
		uup.BroadcasterLanguage = p.BroadcasterLanguage
	}

	if uup.BroadcasterSoftware == nil {
		uup.BroadcasterSoftware = p.BroadcasterSoftware
	}

	if uup.Game == nil {
		uup.Game = p.Game
	}

	if uup.GameID == nil {
		uup.GameID = p.GameID
	}

	if uup.Mature == nil {
		uup.Mature = p.Mature
	}

	if uup.Status == nil {
		uup.Status = p.Status
	}

	if uup.Title == nil {
		uup.Title = p.Title
	}

	if uup.ViewsCount == nil {
		uup.ViewsCount = p.ViewsCount
	}

	if uup.BlockNonPublicAds == nil {
		uup.BlockNonPublicAds = p.BlockNonPublicAds
	}

	if uup.PrerollsDisabled == nil {
		uup.PrerollsDisabled = p.PrerollsDisabled
	}

	if uup.PostrollsDisabled == nil {
		uup.PostrollsDisabled = p.PostrollsDisabled
	}

	if uup.FightAdBlock == nil {
		uup.FightAdBlock = p.FightAdBlock
	}

	if uup.LastBroadcastTime == nil {
		uup.LastBroadcastTime = p.LastBroadcastTime
	}

	if uup.LastBroadcastID == nil {
		uup.LastBroadcastID = p.LastBroadcastID
	}

	if uup.LastLiveNotificationSent == nil {
		uup.LastLiveNotificationSent = p.LastLiveNotificationSent
	}

	if uup.About == nil {
		uup.About = p.About
	}

	if uup.RedirectChannel == nil {
		uup.RedirectChannel = p.RedirectChannel
	}

	if uup.PrimaryTeamID == nil {
		uup.PrimaryTeamID = p.PrimaryTeamID
	}

	if uup.DisableChat == nil {
		if p.DisableChat == nil {
			uup.DisableChat = &defaultBool
		} else {
			uup.DisableChat = p.DisableChat
		}
	}
	return uup
}

const (
	ChannelsIDCacheKey    = "u.id"
	ChannelsLoginCacheKey = "login"
)

func ChannelIDToString(id uint64) string {
	return fmt.Sprintf("%d", id)
}

func (p ChannelProperties) CachePairs() []CachePair {
	var pairs []CachePair

	if p.ID != 0 {
		pairs = append(pairs, CachePair{
			Key:   ChannelsIDCacheKey,
			Value: ChannelIDToString(p.ID),
		})
	}

	if p.Name != "" {
		pairs = append(pairs, CachePair{
			Key:   ChannelsLoginCacheKey,
			Value: p.Name,
		})
	}

	return pairs
}

type ChannelPropertiesIterator []ChannelProperties

func (i ChannelPropertiesIterator) Each(f func(Cacheable) error) error {
	for _, prop := range i {
		if err := f(prop); err != nil {
			return err
		}
	}

	return nil
}
