package models

import (
	"time"

	"reflect"
)

// SNSRenameEvent is published to SNS when SetUserProperties would change either
// login or displayname
type SNSRenameEvent struct {
	UserID    string         `json:"user_id"`
	Data      *SNSRenameData `json:"data"`
	Timestamp time.Time      `json:"timestamp"`
}

type SNSRenameData struct {
	Original *SNSRenameProperties `json:"original"`
	Changes  *SNSRenameProperties `json:"changes"`
}

type SNSRenameProperties struct {
	Login       *string `json:"login,omitempty"`
	Displayname *string `json:"displayname,omitempty"`
}

func NewRenameEvent(id string, original *Properties, changes *UpdateableProperties) SNSRenameEvent {
	originalProperties := SNSRenameProperties{
		Login:       original.Login,
		Displayname: original.Displayname,
	}

	var changedProperties SNSRenameProperties

	if changes.NewLogin != nil && *changes.NewLogin != *original.Login {
		changedProperties.Login = changes.NewLogin
	}

	if changes.Displayname != nil && *changes.Displayname != *original.Displayname {
		changedProperties.Displayname = changes.Displayname
	}

	changeData := SNSRenameData{
		Original: &originalProperties,
		Changes:  &changedProperties,
	}

	return SNSRenameEvent{
		UserID:    id,
		Data:      &changeData,
		Timestamp: time.Now(),
	}
}

// SNSUnbanEvent is published to SNS when a user is unbanned (both via a manual
// unban or automatically following a temporary ban).
type SNSUnbanEvent struct {
	UserID      string     `json:"user_id"`
	Timestamp   time.Time  `json:"timestamp"`
	UpdatedUser Properties `json:"updated_user"`
}

type SNSBanEvent struct {
	UserID                  string    `json:"user_id"`
	DmcaViolation           bool      `json:"dmca_violation" `
	TermsOfServiceViolation bool      `json:"terms_of_service_violation" `
	Timestamp               time.Time `json:"timestamp"`
}

type SNSCreationEvent struct {
	UserID      string    `json:"user_id"`
	Login       string    `json:"login"`
	Displayname string    `json:"displayname"`
	Timestamp   time.Time `json:"timestamp"`
}

type SNSUpdateEvent struct {
	UserID    string                `json:"user_id" `
	Original  *Properties           `json:"original" `
	Changed   *UpdateableProperties `json:"changed" `
	Timestamp time.Time             `json:"timestamp"`
}

type SNSUpdateImageEvent struct {
	UserID    string           `json:"user_id" `
	Original  *Properties      `json:"original" `
	Changed   *ImageProperties `json:"changed" `
	Timestamp time.Time        `json:"timestamp"`
}

type SNSChannelUpdateEvent struct {
	Original  *ChannelProperties       `json:"original"`
	Changes   *UpdateChannelProperties `json:"changes"`
	Timestamp time.Time                `json:"timestamp"`
}

type SNSBanUserEvent struct {
	RecipientID string `json:"id"`
	Type        string `json:"type"`
	Warn        bool   `json:"warn"`
	Reason      string `json:"reason"`
	Description string `json:"description"`
}

type SNSSoftDeleteEvent struct {
	UserID    string    `json:"user_id" `
	Timestamp time.Time `json:"timestamp"`
}

type SNSHardDeleteEvent struct {
	UserID    string    `json:"user_id" `
	Timestamp time.Time `json:"timestamp"`
}

type SNSUndeleteEvent struct {
	UserID    string    `json:"user_id" `
	Timestamp time.Time `json:"timestamp"`
}

func (ue *SNSUpdateEvent) PopulateUpdateEvent(old *Properties, new *UpdateableProperties) {
	ue.Changed = &UpdateableProperties{}
	ue.Changed.ID = new.ID
	if new.NewLogin != nil && !reflect.DeepEqual(new.NewLogin, old.Login) {
		ue.Changed.Login = new.Login
	}

	if new.Birthday != nil && !reflect.DeepEqual(new.Birthday, old.Birthday) {
		ue.Changed.Birthday = new.Birthday
	}

	if new.Displayname != nil && !reflect.DeepEqual(new.Displayname, old.Displayname) {
		ue.Changed.Displayname = new.Displayname
	}

	if new.Email != nil && !reflect.DeepEqual(new.Email, old.Email) {
		ue.Changed.Email = new.Email
	}

	if new.Description != nil && !reflect.DeepEqual(old.Description, new.Description) {
		ue.Changed.Description = new.Description
	}

	if new.EmailVerified != nil && !reflect.DeepEqual(old.EmailVerified, new.EmailVerified) {
		ue.Changed.EmailVerified = new.EmailVerified
	}

	if new.PhoneNumber != nil && !reflect.DeepEqual(old.PhoneNumber, new.PhoneNumber) {
		ue.Changed.PhoneNumber = new.PhoneNumber
	}
}
