package main

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"sync/atomic"
	"time"

	"code.justin.tv/feeds/graphdb/proto/dynamoevent"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/cactus/go-statsd-client/statsd"
)

type EventForwarder struct {
	Stats       statsd.StatSender
	SampleRate  float32
	environment string
	region      string
	lambdaName  string

	statErrors int64
	client     dynamoevent.DynamoUpdates
}

func (e *EventForwarder) inc(name string, val int64, sample float32) {
	if err := e.Stats.Inc(name, val, sample); err != nil {
		atomic.AddInt64(&e.statErrors, 1)
	}
}

func (e *EventForwarder) timingDuration(name string, val time.Duration, sample float32) {
	if err := e.Stats.TimingDuration(name, val, sample); err != nil {
		atomic.AddInt64(&e.statErrors, 1)
	}
}

func (e *EventForwarder) Handler(ctx context.Context, event events.DynamoDBEvent) (retErr error) {
	defer func(startTime time.Time) {
		e.inc("total_records", int64(len(event.Records)), e.SampleRate)
		totalTime := time.Since(startTime)
		if retErr == nil {
			e.timingDuration("ok", totalTime, e.SampleRate)
		} else {
			e.timingDuration("err", totalTime, e.SampleRate)
		}
	}(time.Now())
	var buf bytes.Buffer
	if err := json.NewEncoder(&buf).Encode(event); err != nil {
		return err
	}
	resp, err := e.client.DynamoDBEvent(ctx, &dynamoevent.DynamoDBEventRequest{
		Environment:   e.environment,
		Region:        e.region,
		LambdaName:    e.lambdaName,
		DynamoDbEvent: buf.Bytes(),
	})
	if err != nil {
		fmt.Println("Unable to send dynamodb events", err)
		return err
	}
	if resp.LogMsg != "" {
		fmt.Println("Log msg:", resp.LogMsg)
	}
	return nil
}

func main() {
	env := os.Getenv("ENVIRONMENT")
	region := os.Getenv("AWS_REGION")
	lambdaName := os.Getenv("AWS_LAMBDA_FUNCTION_NAME")
	statsdHost := "statsd.central.twitch.a2z.com:8125"
	if os.Getenv("DYNAMO_FORWARD_LOCATION") == "" {
		fmt.Println("Missing DYNAMO_FORWARD_LOCATION")
		os.Exit(1)
	}
	forwardLocation := fmt.Sprintf("http://%s", os.Getenv("DYNAMO_FORWARD_LOCATION"))
	client := dynamoevent.NewDynamoUpdatesProtobufClient(forwardLocation, &http.Client{})
	stastdClient, err := statsd.New(statsdHost, "feeds."+env+".lambda."+region+"."+lambdaName)
	if err != nil {
		fmt.Println("Unable to dial in for statsd metrics", err)
		os.Exit(1)
	}
	e := EventForwarder{
		client:      client,
		environment: env,
		region:      region,
		lambdaName:  lambdaName,
		Stats:       stastdClient,
		SampleRate:  1.0,
	}
	lambda.Start(e.Handler)
}
