package cohesionpql

import (
	"context"
	"database/sql"
	"encoding/json"
	"strconv"
	"time"

	"fmt"

	"code.justin.tv/feeds/graphdb/cmd/graphdb-migrate/internal/migrator"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/log"
	"github.com/golang/protobuf/ptypes/timestamp"
	"github.com/lib/pq"
)

type CohesionPQL struct {
	Connection     string
	DriverName     string
	Limit          int64
	RowsToRead     int64
	TotalRowsRead  int64
	TypesToMigrate []string
	TypeRenameMap  map[string]string
	TableName      string
	Log            log.Logger
	db             *sql.DB
	lastItem       int64
}

const finishedCursor = "_FINISHED_"

var _ migrator.DataSource = &CohesionPQL{}

func (f *CohesionPQL) Setup() error {
	var err error
	f.db, err = sql.Open(f.DriverName, f.Connection)
	if err != nil {
		return err
	}
	pingCtx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	return f.db.PingContext(pingCtx)
}

func addFloat(m map[string]float64, k string, v float64) map[string]float64 {
	if m == nil {
		return map[string]float64{
			k: v,
		}
	}
	m[k] = v
	return m
}

func addInt(m map[string]int64, k string, v int64) map[string]int64 {
	if m == nil {
		return map[string]int64{
			k: v,
		}
	}
	m[k] = v
	return m
}

func addString(m map[string]string, k string, v string) map[string]string {
	if m == nil {
		return map[string]string{
			k: v,
		}
	}
	m[k] = v
	return m
}

func addBool(m map[string]bool, k string, v bool) map[string]bool {
	if m == nil {
		return map[string]bool{
			k: v,
		}
	}
	m[k] = v
	return m
}

func ToProtoDatabag(from map[string]interface{}) *graphdb.DataBag {
	data := &graphdb.DataBag{}
	if len(from) == 0 {
		return nil
	}
	for k, v := range from {
		switch casted := v.(type) {
		case float64:
			data.Doubles = addFloat(data.Doubles, k, casted)
		case int64:
			data.Ints = addInt(data.Ints, k, casted)
		case int:
			data.Ints = addInt(data.Ints, k, int64(casted))
		case bool:
			data.Bools = addBool(data.Bools, k, casted)
		case string:
			data.Strings = addString(data.Strings, k, casted)
		}
	}
	return data
}

func ToProtoCreated(from time.Time) *timestamp.Timestamp {
	return &timestamp.Timestamp{
		Seconds: from.Unix(),
		Nanos:   int32(from.UnixNano() % time.Second.Nanoseconds()),
	}
}

func (f *CohesionPQL) processRow(rows *sql.Rows) (*graphdb.EdgeCreateRequest, int64, error) {
	var id, from_id, to_id int64
	var from_kind, to_kind, assoc_kind, data_bag string
	var creation_date time.Time
	if err := rows.Scan(&id, &from_id, &from_kind, &to_id, &to_kind, &assoc_kind, &data_bag, &creation_date); err != nil {
		return nil, 0, err
	}
	var decoded_data map[string]interface{}
	if err := json.Unmarshal([]byte(data_bag), &decoded_data); err != nil {
		return nil, 0, err
	}
	if val, ok := f.TypeRenameMap[assoc_kind]; ok {
		assoc_kind = val
	}

	assoc := &graphdb.EdgeCreateRequest{
		Edge: &graphdb.Edge{
			From: &graphdb.Node{
				Type: from_kind,
				Id:   strconv.FormatInt(from_id, 10),
			},
			Type: assoc_kind,
			To: &graphdb.Node{
				Type: to_kind,
				Id:   strconv.FormatInt(to_id, 10),
			},
		},
		Data:      ToProtoDatabag(decoded_data),
		CreatedAt: ToProtoCreated(creation_date),
	}
	//ret = append(ret, assoc)
	return assoc, id, nil
}

func (f *CohesionPQL) queryLastItem(ctx context.Context) (int64, error) {
	stmt, err := f.db.PrepareContext(ctx, fmt.Sprintf("SELECT id FROM %s ORDER BY id DESC LIMIT 1", f.TableName))
	if err != nil {
		return 0, err
	}
	row := stmt.QueryRowContext(ctx)
	var lastId int64
	if err := row.Scan(&lastId); err != nil {
		return 0, err
	}
	if err := stmt.Close(); err != nil {
		return 0, err
	}
	return lastId, nil
}

func (f *CohesionPQL) Progress(ctx context.Context, cursor string) float64 {
	asInt, err := strconv.ParseInt(cursor, 10, 64)
	if err != nil {
		return 0.0
	}
	if f.lastItem == 0 {
		lastItem, err := f.queryLastItem(ctx)
		if err != nil {
			return 0.0
		}
		f.lastItem = lastItem
	}
	if f.lastItem == -1 {
		return 0.0
	}
	return float64(asInt) / float64(f.lastItem)
}

func (f *CohesionPQL) List(ctx context.Context, cursor string) ([]*graphdb.EdgeCreateRequest, string, error) {
	if cursor == finishedCursor {
		return nil, finishedCursor, nil
	}
	if f.RowsToRead != -1 && f.TotalRowsRead >= f.RowsToRead {
		return nil, "", nil
	}
	var startIndex int64
	if cursor != "" {
		var err error
		startIndex, err = strconv.ParseInt(cursor, 10, 64)
		if err != nil {
			return nil, "", err
		}
	}
	stmt, err := f.db.PrepareContext(ctx, "SELECT id, from_id, from_kind, to_id, to_kind, assoc_kind, data_bag, creation_date from "+f.TableName+" WHERE id > $1 AND assoc_kind = ANY($2) ORDER BY id LIMIT $3")
	if err != nil {
		return nil, "", err
	}
	rows, err := stmt.QueryContext(ctx, startIndex, pq.Array(f.TypesToMigrate), f.Limit)
	if err != nil {
		return nil, "", err
	}
	defer func() {
		err := rows.Close()
		if err != nil {
			f.Log.Log("err", err, "unable to close rows")
		}
	}()
	var lastId int64
	ret := make([]*graphdb.EdgeCreateRequest, 0, f.Limit)
	for rows.Next() {
		assoc, id, err := f.processRow(rows)
		if err != nil {
			return nil, "", err
		}
		lastId = id
		ret = append(ret, assoc)
	}
	if len(ret) == 0 {
		// At the end.  We're done
		return nil, finishedCursor, nil
	}
	return ret, strconv.FormatInt(lastId, 10), nil
}

func (f *CohesionPQL) Close() error {
	return f.db.Close()
}
