package migrator

import (
	"context"
	"time"

	"code.justin.tv/feeds/graphdb/internal/checkpoint"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/hygienic/statsdsender"
	"github.com/cep21/circuit"
)

type Migrator struct {
	Source             DataSource `nilcheck:"ignore"`
	Log                log.Logger
	Destination        graphdb.GraphDB
	startingCursor     string
	Checkpointer       checkpoint.Checkpointer
	SourceCircuit      *circuit.Circuit
	DestinationCircuit *circuit.Circuit
	MultiConcurrency   int64
	currentDelay       time.Duration
	onClose            chan struct{}
	Statsd             *statsdsender.ErrorlessStatSender `nilcheck:"nodepth"`
}

const finishedCursor = "_FINISHED_"

type DataSource interface {
	List(ctx context.Context, cursor string) ([]*graphdb.EdgeCreateRequest, string, error)
	Progress(ctx context.Context, cursor string) float64
	Setup() error
	Close() error
}

func (m *Migrator) Setup() error {
	m.onClose = make(chan struct{})
	var err error
	m.startingCursor, err = m.Checkpointer.Read(context.Background())
	if err != nil {
		return err
	}
	m.Log.Log("start_key", m.startingCursor)
	if m.Source == nil {
		return nil
	}
	return m.Source.Setup()
}

func intoSingleRequests(in []*graphdb.EdgeCreateRequest) []*graphdb.MultiAsyncRequest_SingleRequest {
	ret := make([]*graphdb.MultiAsyncRequest_SingleRequest, 0, len(in))
	for _, n := range in {
		ret = append(ret, &graphdb.MultiAsyncRequest_SingleRequest{
			RequestType: &graphdb.MultiAsyncRequest_SingleRequest_EdgeCreate{
				EdgeCreate: n,
			},
		})
	}
	return ret
}

func (m *Migrator) logProgress(ctx context.Context, cursor string, previousProgress float64, startTime time.Time, rowLen int) float64 {
	endTime := time.Now()
	currentProgress := m.Source.Progress(ctx, cursor)
	m.Statsd.GaugeC("progress.millions", int64(currentProgress*1000000), 1)
	changeTime := endTime.Sub(startTime)
	var durationLeft time.Duration
	if previousProgress != 0.0 && currentProgress != 0.0 {
		progressInThisTimePeriod := currentProgress - previousProgress
		progressLeft := 1.0 - currentProgress
		durationLeft = time.Duration(int64(progressLeft/progressInThisTimePeriod) * changeTime.Nanoseconds())
	}
	previousProgress = currentProgress
	m.Log.Log("rows", rowLen, "time", changeTime, "progress", currentProgress, "duration_left", durationLeft, "finished List")
	return previousProgress
}

func (m *Migrator) Start() error {
	if m.Source == nil {
		m.Log.Log("ending migration: no source")
		return nil
	}
	previousProgress := 0.0
	cursor := m.startingCursor
	if cursor == finishedCursor {
		return nil
	}
	for {
		select {
		case <-m.onClose:
			return nil
		default:
		}
		iterationCursor := context.Background()
		startTime := time.Now()
		var rows []*graphdb.EdgeCreateRequest
		var nextCursor string
		err := m.SourceCircuit.Run(iterationCursor, func(ctx context.Context) error {
			var err3 error
			rows, nextCursor, err3 = m.Source.List(ctx, cursor)
			return err3
		})

		if err != nil {
			m.backoff(err)
			continue
		}
		if len(rows) == 0 {
			cursor = nextCursor
			if cursor == finishedCursor {
				m.Log.Log("zero rows and no cursor.  Exiting")
				break
			}
			m.Log.Log("got zero rows.  Continuing")
			continue
		}
		m.Statsd.IncC("listed_rows", int64(len(rows)), 1)
		previousProgress = m.logProgress(iterationCursor, cursor, previousProgress, startTime, len(rows))
		err = m.DestinationCircuit.Run(iterationCursor, func(ctx context.Context) error {
			_, err2 := m.Destination.MultiAsync(ctx, &graphdb.MultiAsyncRequest{
				Requests: intoSingleRequests(rows),
			})
			return err2
		})

		if err != nil {
			m.backoff(err)
			continue
		}
		cursor = nextCursor
		m.Checkpointer.Write(cursor, previousProgress)
		if cursor == finishedCursor {
			break
		}
	}
	return nil
}

func (m *Migrator) Close() error {
	close(m.onClose)
	if m.Source != nil {
		return m.Source.Close()
	}
	return nil
}

func (m *Migrator) backoff(err error) {
	m.Log.Log("err", err, "starting backoff")
	m.currentDelay += time.Second
	m.currentDelay *= 2
	if m.currentDelay > time.Minute*5 {
		m.currentDelay = time.Minute * 5
	}
	select {
	case <-time.After(m.currentDelay):
	case <-m.onClose:
	}
}

func (m *Migrator) clearBackoff() {
	m.currentDelay = 0
}
