package cohesionpql

import (
	"context"
	"database/sql"
	"strconv"
	"time"

	"code.justin.tv/feeds/graphdb/cmd/graphdb-resync/internal/resync"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/log"
	"github.com/lib/pq"
)

type CohesionPQL struct {
	Connection     string
	DriverName     string
	Limit          int64
	RowsToRead     int64
	TotalRowsRead  int64
	TypesToMigrate []string
	Log            log.Logger
	db             *sql.DB
	lastItem       int64
}

const finishedCursor = "_FINISHED_"

var _ resync.DataSource = &CohesionPQL{}

func (f *CohesionPQL) Setup() error {
	var openErr error
	if f.db, openErr = sql.Open(f.DriverName, f.Connection); openErr != nil {
		return openErr
	}
	pingCtx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	if err := f.db.PingContext(pingCtx); err != nil {
		return err
	}
	_, err := f.db.Exec("set statement_timeout to 60000")
	return err
}

func processRow(rows *sql.Rows) (*graphdb.EdgeListRequest, error) {
	var from_id int64
	var from_kind, assoc_kind string
	if err := rows.Scan(&from_id, &from_kind, &assoc_kind); err != nil {
		return nil, err
	}
	assoc := &graphdb.EdgeListRequest{
		From: &graphdb.Node{
			Type: assoc_kind,
			Id:   strconv.FormatInt(from_id, 10),
		},
		EdgeType: assoc_kind,
	}
	return assoc, nil
}

func (f *CohesionPQL) queryLastItem(ctx context.Context) (int64, error) {
	stmt, err := f.db.PrepareContext(ctx, "SELECT from_id FROM associations ORDER BY from_id DESC LIMIT 1")
	if err != nil {
		return 0, err
	}
	row := stmt.QueryRowContext(ctx)
	var lastId int64
	if err := row.Scan(&lastId); err != nil {
		return 0, err
	}
	if err := stmt.Close(); err != nil {
		return 0, err
	}
	return lastId, nil
}

func (f *CohesionPQL) Progress(ctx context.Context, cursor string) float64 {
	asInt, err := strconv.ParseInt(cursor, 10, 64)
	if err != nil {
		return 0.0
	}
	if f.lastItem == 0 {
		lastItem, err := f.queryLastItem(ctx)
		if err != nil {
			return 0.0
		}
		f.lastItem = lastItem
	}
	if f.lastItem == -1 {
		return 0.0
	}
	return float64(asInt) / float64(f.lastItem)
}

func (f *CohesionPQL) List(ctx context.Context, cursor string) ([]*graphdb.EdgeListRequest, string, error) {
	if cursor == finishedCursor {
		return nil, finishedCursor, nil
	}
	if f.RowsToRead != -1 && f.TotalRowsRead >= f.RowsToRead {
		return nil, "", nil
	}
	var startIndex int64
	if cursor != "" {
		var err error
		startIndex, err = strconv.ParseInt(cursor, 10, 64)
		if err != nil {
			return nil, "", err
		}
	}
	stmt, err := f.db.PrepareContext(ctx, "SELECT DISTINCT from_id, from_kind, assoc_kind FROM associations WHERE from_id > $1 AND assoc_kind = ANY($2) ORDER BY from_id, from_kind, assoc_kind LIMIT $3")
	if err != nil {
		return nil, "", err
	}
	rows, err := stmt.QueryContext(ctx, startIndex, pq.Array(f.TypesToMigrate), f.Limit)
	if err != nil {
		return nil, "", err
	}
	defer func() {
		err := rows.Close()
		if err != nil {
			f.Log.Log("err", err, "unable to close rows")
		}
	}()
	var lastId string
	ret := make([]*graphdb.EdgeListRequest, 0, f.Limit)
	for rows.Next() {
		listReq, err := processRow(rows)
		if err != nil {
			return nil, "", err
		}
		lastId = listReq.From.Id
		ret = append(ret, listReq)
	}
	if len(ret) == 0 {
		// At the end.  We're done
		return nil, finishedCursor, nil
	}
	return ret, lastId, nil
}

func (f *CohesionPQL) Close() error {
	return f.db.Close()
}
