package main

import (
	"os"
	"os/signal"
	"time"

	"net/http"

	"fmt"

	"context"

	"strings"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/graphdb/cmd/graphdb-resync/internal/resync"
	"code.justin.tv/feeds/graphdb/cmd/graphdb-resync/internal/resync/cohesionpql"
	"code.justin.tv/feeds/graphdb/internal/checkpoint"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/graphdb/proto/graphdbadmin"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/hygienic/statsdsender"
	"github.com/cep21/circuit"
)

const (
	teamName    = "feeds"
	serviceName = "graphdb-resync"
)

// CodeVersion is set by build script
var CodeVersion string

func sfxStastdConfig() sfxstatsd.SetupConfig {
	return sfxstatsd.SetupConfig{}
}

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion: CodeVersion,
	},
}

type service struct {
	osExit           func(code int)
	sigChan          chan os.Signal
	serviceCommon    service_common.ServiceCommon
	runner           service_common.ServiceRunner
	dataSource       resync.DataSource
	destination      graphdb.GraphDB
	destinationAdmin graphdbadmin.GraphDBAdmin
	configs          struct {
		Config Config
	}
}

type Config struct {
	CohesionURL *distconf.Str
}

func (c *Config) Load(d *distconf.Distconf) error {
	c.CohesionURL = d.Str("graphdb.alb_dns", "")
	if c.CohesionURL.Get() == "" {
		return errors.New("unable to find graphdb.url")
	}
	return nil
}

func (f *service) setup() error {
	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(
		f.serviceCommon.Config,
		&f.configs.Config); err != nil {
		return err
	}
	destURL := fmt.Sprintf("http://%s", f.configs.Config.CohesionURL.Get())
	f.serviceCommon.Log.Log("destination_url", destURL)
	f.destination = graphdb.NewGraphDBProtobufClient(destURL, &http.Client{})
	f.destinationAdmin = graphdbadmin.NewGraphDBAdminProtobufClient(destURL, &http.Client{})
	migrationKey := "1"
	var err error
	f.dataSource, err = f.setupSource(migrationKey)
	return err
}

func (f *service) setupSource(migrationKey string) (resync.DataSource, error) {
	ctx := context.Background()
	resp, err := f.destination.NodeGet(ctx, &graphdb.NodeGetRequest{
		Node: &graphdb.Node{
			Type: "graphdb_resync",
			Id:   migrationKey,
		},
	})
	if err != nil {
		return nil, err
	}
	if resp.Node == nil {
		return nil, nil
	}
	migratorType := resp.Node.Data.GetData().GetStrings()["type"]
	if migratorType != "cohesion_pql" {
		return nil, errors.Errorf("unknown type %s", migratorType)
	}

	return &cohesionpql.CohesionPQL{
		DriverName:     "postgres",
		Connection:     f.serviceCommon.Config.Str("graphdb.truth_pql_connection", "").Get(),
		Limit:          100,
		RowsToRead:     -1,
		Log:            f.serviceCommon.Log,
		TypesToMigrate: strings.Split(resp.Node.Data.GetData().GetStrings()["types_to_migrate"], ","), //"follows"},
	}, nil
}

func (f *service) inject() { // nolint:dupl
	migrationKey := "1"
	m := resync.Migrator{
		Source: f.dataSource,
		Checkpointer: checkpoint.Checkpointer{
			Log:         f.serviceCommon.Log,
			Key:         migrationKey,
			Destination: f.destination,
			NodeType:    "graphdb_resync",
		},
		SourceCircuit: f.serviceCommon.Circuit.MustCreateCircuit("source", circuit.Config{
			Execution: circuit.ExecutionConfig{
				Timeout: time.Minute * 3,
			},
		}),
		DestinationCircuit: f.serviceCommon.Circuit.MustCreateCircuit("destination", circuit.Config{
			Execution: circuit.ExecutionConfig{
				Timeout: time.Minute * 3,
			},
		}),
		Destination: f.destinationAdmin,
		Log:         f.serviceCommon.Log,
		Statsd: &statsdsender.ErrorlessStatSender{
			StatSender: f.serviceCommon.Statsd,
		},
		MultiConcurrency: 10,
	}
	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.serviceCommon, &m, &m.Checkpointer,
		},
		SigChan:      f.sigChan,
		SignalNotify: signal.Notify,
	}
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http"},
	}).Check(f, f.configs, f.runner)
	res.MustBeEmpty(os.Stderr)
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		f.osExit(1)
		return
	}
	f.inject()
	if err := f.runner.Execute(); err != nil {
		f.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		time.Sleep(time.Second)
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
