package main

import (
	"bytes"
	"context"
	"strconv"
	"strings"
	"text/template"

	"code.justin.tv/hygienic/errors"
	"github.com/aws/aws-lambda-go/events"
)

var _ command = &counts{}

type counts struct {
	spacedCommand
}

func (c *counts) PrintHelp() string {
	return `
/graphdb counts <user_id>
  Prints the counts of all the edges types out of the user ID
`
}

func (c *counts) ShouldHandle(ctx context.Context, e *SlackListener, event events.APIGatewayProxyRequest) bool {
	body, err := c.getBody(event)
	if err != nil {
		return false
	}
	parts := strings.Split(body, " ")
	return len(parts) > 0 && parts[0] == "counts"
}

func (c *counts) Handle(ctx context.Context, e *SlackListener, event events.APIGatewayProxyRequest) (*textResponse, error) {
	bodyParts, err := c.verificationWithBodyParts(ctx, e, event)
	if err != nil {
		return nil, err
	}
	if len(bodyParts) != 2 {
		return nil, errors.New("invalid body.  Expect a single user id")
	}
	userID, err := strconv.ParseInt(bodyParts[1], 10, 64)
	if err != nil {
		return nil, err
	}
	var cr countsResponse
	if err := cr.Populate(ctx, userID, e); err != nil {
		return nil, err
	}
	var buf bytes.Buffer
	if err := countsResp.Execute(&buf, cr); err != nil {
		return nil, err
	}
	return &textResponse{
		Text:         buf.String(),
		ResponseType: "in_channel",
	}, nil
}

type countsResponse struct {
	UserID              int64
	FollowCount         int64
	FollowerCount       int64
	HiddenFollowCount   int64
	HiddenFollowerCount int64
}

func (c *countsResponse) Populate(ctx context.Context, UserID int64, e *SlackListener) error {
	c.UserID = UserID
	var err error
	c.FollowCount, err = e.getCount(ctx, strconv.FormatInt(UserID, 10), "follows")
	if err != nil {
		return err
	}
	c.FollowerCount, err = e.getCount(ctx, strconv.FormatInt(UserID, 10), "followed_by")
	if err != nil {
		return err
	}
	c.HiddenFollowCount, err = e.getCount(ctx, strconv.FormatInt(UserID, 10), "hidden_follows")
	if err != nil {
		return err
	}
	c.HiddenFollowerCount, err = e.getCount(ctx, strconv.FormatInt(UserID, 10), "hidden_followed_by")
	return err
}

var countsResp = template.Must(template.New("countsTemplate").Parse(countsRespText))

const countsRespText = "```" + `
Counts for User: {{.UserID}}
Follows:         {{.FollowCount}}
Followers:       {{.FollowerCount}}
HiddenFollows:   {{.HiddenFollowCount}}
HiddenFollowers: {{.HiddenFollowerCount}}
` + "```"
