package main

import (
	"bytes"
	"context"
	"net/url"
	"strings"
	"text/template"

	"github.com/aws/aws-lambda-go/events"
)

var _ command = &help{}

type help struct {
}

func (c *help) PrintHelp() string {
	return `
/graphdb help
  Prints help docs about the various graphdb agent commands
`
}

func (c *help) getBody(event events.APIGatewayProxyRequest) (string, error) {
	parsedBody, err := url.ParseQuery(event.Body)
	if err != nil {
		return "", nil
	}
	if parsedBody.Get("text") == "" {
		return "", nil
	}
	return parsedBody.Get("text"), nil
}

func (c *help) ShouldHandle(ctx context.Context, e *SlackListener, event events.APIGatewayProxyRequest) bool {
	body, err := c.getBody(event)
	if err != nil {
		return false
	}
	parts := strings.Split(body, " ")
	return len(parts) > 0 && parts[0] == "help"
}

func (c *help) Handle(ctx context.Context, e *SlackListener, event events.APIGatewayProxyRequest) (*textResponse, error) {
	if err := e.verifyToken(event); err != nil {
		return nil, err
	}
	var cr helpResponse
	cr.Populate(e)
	var buf bytes.Buffer
	if err := helpResp.Execute(&buf, cr); err != nil {
		return nil, err
	}
	return &textResponse{
		Text: buf.String(),
	}, nil
}

type helpResponse struct {
	HelpText []string
}

func (c *helpResponse) Populate(e *SlackListener) {
	for _, cmd := range e.commands {
		c.HelpText = append(c.HelpText, cmd.PrintHelp())
	}
}

var helpResp = template.Must(template.New("restore_follows").Parse(helpRespText))

const helpRespText = `
The /graphdb command allows you to interact with the service GraphDB, which stores connection
information about a user's following graph.  It also contains special helpers for follows directly.

{{ range $key, $value := .HelpText -}}
  {{ $value }}
{{ end -}}
`
