package main

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"os"

	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/log"
	"code.justin.tv/hygienic/log/fmtlogger"
	"code.justin.tv/hygienic/paramstoreconf"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/cactus/go-statsd-client/statsd"
)

type SlackListener struct {
	Stats         statsd.Statter
	SampleRate    float32
	Log           log.Logger
	environment   string
	region        string
	dconf         *distconf.Distconf
	secrets       *distconf.Distconf
	graphdbClient graphdb.GraphDB
	followsRPC    followsrpc.Follows
	lambdaName    string

	commands []command

	config        config
	secretsConfig secretsConfig
}

type config struct {
	clientID             *distconf.Str
	graphdbAddr          *distconf.Str
	followingServiceAddr *distconf.Str
	accessLogChan        *distconf.Str
}

func (c *config) Load(d *distconf.Distconf) error {
	c.clientID = d.Str("graphdb-slack.client_id", "")
	if c.clientID.Get() == "" {
		return errors.New("Unable to discover client id from distconf")
	}
	c.graphdbAddr = d.Str("graphdb.addr", "")
	if c.graphdbAddr.Get() == "" {
		return errors.New("Unable to discover graphdb address")
	}
	c.followingServiceAddr = d.Str("following-service.addr", "")
	if c.followingServiceAddr.Get() == "" {
		return errors.New("Unable to discover following service address")
	}
	c.accessLogChan = d.Str("graphdb-slack.access-logs", "#graphdb-access-logs")
	return nil
}

type secretsConfig struct {
	clientSecret      *distconf.Str
	oauthToken        *distconf.Str
	verificationToken *distconf.Str
}

func (c *secretsConfig) Load(d *distconf.Distconf) error {
	c.clientSecret = d.Str("graphdb-slack.client_secret", "")
	if c.clientSecret.Get() == "" {
		return errors.New("Unable to discover clientSecret from distconf")
	}
	c.verificationToken = d.Str("graphdb-slack.verification_token", "")
	if c.verificationToken.Get() == "" {
		return errors.New("Unable to discover verificationToken from distconf")
	}
	c.oauthToken = d.Str("graphdb-slack.oauth-token", "")
	if c.oauthToken.Get() == "" {
		return errors.New("Unable to discover oauth token from distconf")
	}
	return nil
}

const team = "feeds"
const service = "graphdb-slack-lambda"

type ChallengeResponse struct {
	Challenge string `json:"challenge"`
}

type ChallengeRequest struct {
	Token     string `json:"token"`
	Challenge string `json:"challenge"`
	Type      string `json:"type"`
}

func setupDistconf() (*distconf.Distconf, error) {
	ps, err := paramstoreconf.DefaultParameterStoreConfiguration()
	if err != nil {
		return nil, err
	}
	ps.Team = team
	ps.Service = service
	ps.Environment = env()
	dconf := &distconf.Distconf{
		Readers: []distconf.Reader{
			ps,
		},
	}
	return dconf, nil
}

func setupDistconfSecrets() (*distconf.Distconf, error) {
	ps, err := paramstoreconf.DefaultParameterStoreConfiguration()
	if err != nil {
		return nil, err
	}
	ps.AllowEncrypted = true
	ps.Team = team
	ps.Service = service
	ps.Environment = env()
	dconf := &distconf.Distconf{
		Readers: []distconf.Reader{
			ps,
		},
	}
	return dconf, nil
}

func env() string {
	return os.Getenv("ENVIRONMENT")
}

func (e *SlackListener) handleChallenge(body string) (*ChallengeResponse, error) {
	var req ChallengeRequest
	if err := json.Unmarshal([]byte(body), &req); err != nil {
		return nil, nil
	}
	if req.Type != "url_verification" {
		return nil, nil
	}
	return &ChallengeResponse{
		Challenge: req.Challenge,
	}, nil
}

func (e *SlackListener) Handler(ctx context.Context, event events.APIGatewayProxyRequest) (ret *events.APIGatewayProxyResponse, retErr error) {
	defer func() {
		e.Log.Log("reterr", retErr, "retCode", ret.StatusCode, "Handler returned")
	}()
	e.Log.Log("handler", event.Path, "body", event.Body, "handler called")
	if resp, err := e.handleSlash(ctx, event); resp != nil || err != nil {
		if err != nil {
			return &events.APIGatewayProxyResponse{
				StatusCode: 200,
				Body:       "unable to handle slash request " + err.Error(),
				Headers: map[string]string{
					"Content-Type": "application/json",
				},
			}, nil
		}
		body, err := json.Marshal(resp)
		if err != nil {
			return nil, err
		}
		return &events.APIGatewayProxyResponse{
			StatusCode: 200,
			Body:       string(body),
			Headers: map[string]string{
				"Content-Type": "application/json",
			},
		}, nil
	}
	if resp, err := e.handleChallenge(event.Body); resp != nil || err != nil {
		body, err := json.Marshal(resp)
		if err != nil {
			return nil, err
		}
		return &events.APIGatewayProxyResponse{
			StatusCode: 200,
			Body:       string(body),
		}, nil
	}
	return &events.APIGatewayProxyResponse{
		StatusCode: 200,
		Body:       "hello world",
		Headers: map[string]string{
			"X-testing": "hello-headers",
		},
	}, nil
}

func (e *SlackListener) setupGraphdbClient() {
	e.graphdbClient = graphdb.NewGraphDBProtobufClient(e.config.graphdbAddr.Get(), &http.Client{})
}

func (e *SlackListener) setupFollowsClient() {
	e.followsRPC = followsrpc.NewFollowsProtobufClient(e.config.followingServiceAddr.Get(), &http.Client{})
}

func (e *SlackListener) slackLog(msg string) error {
	e.Log.Log("access-log", msg)
	return nil
}

func (e *SlackListener) setupSlackClient() error {
	return nil
}

func (e *SlackListener) Setup() error {
	e.Log = fmtlogger.NewLogfmtLogger(os.Stderr, log.Discard)
	environ := env()
	region := os.Getenv("AWS_REGION")
	lambdaName := os.Getenv("AWS_LAMBDA_FUNCTION_NAME")
	statsdHost := "statsd.central.twitch.a2z.com:8125"
	stastdClient, err := statsd.New(statsdHost, team+"."+service+"."+environ+".lambda."+region+"."+lambdaName)
	if err != nil {
		return err
	}

	e.environment = environ
	e.region = region
	e.lambdaName = lambdaName
	e.SampleRate = 1.0

	e.Stats = stastdClient
	dconf, err := setupDistconf()
	if err != nil {
		return err
	}
	e.dconf = dconf

	secrets, err := setupDistconfSecrets()
	if err != nil {
		return err
	}
	e.secrets = secrets

	if err := e.secretsConfig.Load(e.secrets); err != nil {
		return err
	}

	if err := e.config.Load(e.dconf); err != nil {
		return err
	}

	e.setupGraphdbClient()
	e.setupFollowsClient()

	if err := e.setupSlackClient(); err != nil {
		return err
	}

	return nil
}

type textResponse struct {
	Text         string `json:"text"`
	ResponseType string `json:"response_type"`
}

func (e *SlackListener) verifyToken(event events.APIGatewayProxyRequest) error {
	parsedBody, err := url.ParseQuery(event.Body)
	if err != nil {
		return errors.New("Unable to parse request body")
	}
	if parsedBody.Get("token") == "" {
		return errors.New("Unable to find slack token")
	}
	parsedToken := parsedBody.Get("token")
	if parsedToken != e.secretsConfig.verificationToken.Get() {
		return errors.Errorf("Invalid verification token: %s", parsedToken)
	}
	return nil
}

func (e *SlackListener) logSlashEvent(vals url.Values) error {
	msg := fmt.Sprintf("User %s channel %s text %s", vals.Get("user_name"), vals.Get("channel_name"), vals.Get("text"))
	return e.slackLog(msg)
}

func (e *SlackListener) handleSlash(ctx context.Context, event events.APIGatewayProxyRequest) (*textResponse, error) {
	parsedBody, err := url.ParseQuery(event.Body)
	if err != nil {
		return nil, nil
	}
	if parsedBody.Get("token") == "" {
		return nil, nil
	}
	parsedToken := parsedBody.Get("token")
	if parsedToken != e.secretsConfig.verificationToken.Get() {
		return nil, errors.Errorf("Invalid verification token: %s", parsedToken)
	}
	if err := e.logSlashEvent(parsedBody); err != nil {
		return nil, errors.Wrap(err, "unable to log slack event")
	}
	body := parsedBody.Get("text")
	for _, cmd := range e.commands {
		if !cmd.ShouldHandle(ctx, e, event) {
			continue
		}
		return cmd.Handle(ctx, e, event)
	}
	return &textResponse{
		Text: "Unknown command: I saw " + body + "\nTry /graphdb help",
	}, nil
}

type command interface {
	PrintHelp() string
	ShouldHandle(ctx context.Context, e *SlackListener, event events.APIGatewayProxyRequest) bool
	Handle(ctx context.Context, e *SlackListener, event events.APIGatewayProxyRequest) (*textResponse, error)
}

func (e *SlackListener) getCount(ctx context.Context, userID string, edgeType string) (int64, error) {
	resp, err := e.graphdbClient.EdgeCount(ctx, &graphdb.EdgeCountRequest{
		From: &graphdb.Node{
			Type: "user",
			Id:   userID,
		},
		EdgeType: edgeType,
	})
	if err != nil {
		return 0, err
	}
	return resp.Count, nil
}

func main() {
	e := SlackListener{
		commands: []command{
			&counts{}, &unhideFollowers{}, &help{},
		},
	}
	if err := e.Setup(); err != nil {
		fmt.Println("Unable to setup server:", err)
		os.Exit(1)
	}
	lambda.Start(e.Handler)
}
