package main

import (
	"bytes"
	"context"
	"fmt"
	"net/url"
	"strconv"
	"strings"
	"text/template"

	"code.justin.tv/feeds/following-service/rpc/followsrpc"
	"code.justin.tv/hygienic/errors"
	"github.com/aws/aws-lambda-go/events"
)

var _ command = &counts{}

type spacedCommand struct {
}

func (s *spacedCommand) getBody(event events.APIGatewayProxyRequest) (string, error) {
	parsedBody, err := url.ParseQuery(event.Body)
	if err != nil {
		return "", nil
	}
	if parsedBody.Get("text") == "" {
		return "", nil
	}
	return parsedBody.Get("text"), nil
}

func (s *spacedCommand) verificationWithBodyParts(ctx context.Context, e *SlackListener, event events.APIGatewayProxyRequest) ([]string, error) {
	if err := e.verifyToken(event); err != nil {
		return nil, err
	}
	body, err := s.getBody(event)
	if err != nil {
		return nil, err
	}
	bodyParts := strings.Split(body, " ")
	return bodyParts, nil
}

type unhideFollowers struct {
	spacedCommand
}

func (c *unhideFollowers) PrintHelp() string {
	return `
/graphdb restore_follows <user_id>
  Unhides the followers of a user_id
`
}

func (c *unhideFollowers) ShouldHandle(ctx context.Context, e *SlackListener, event events.APIGatewayProxyRequest) bool {
	body, err := c.getBody(event)
	if err != nil {
		return false
	}
	parts := strings.Split(body, " ")
	return len(parts) > 0 && parts[0] == "restore_follows"
}

func (c *unhideFollowers) Handle(ctx context.Context, e *SlackListener, event events.APIGatewayProxyRequest) (*textResponse, error) {
	var err error
	bodyParts, err := c.verificationWithBodyParts(ctx, e, event)
	if err != nil {
		return nil, err
	}
	if len(bodyParts) != 2 {
		return nil, errors.New("invalid body.  Expect a single user id")
	}

	userID := bodyParts[1]
	if _, err := strconv.ParseInt(userID, 10, 64); err != nil {
		return nil, errors.New(fmt.Sprintf("user_id [%s] must be a valid user id", userID))
	}

	var cr unhideFollowersResponse
	if err := cr.Populate(ctx, userID, e); err != nil {
		return nil, err
	}
	var buf bytes.Buffer
	if err := unhideFollowersResp.Execute(&buf, cr); err != nil {
		return nil, err
	}
	return &textResponse{
		Text:         buf.String(),
		ResponseType: "in_channel",
	}, nil
}

type unhideFollowersResponse struct {
	UserID string
}

func (c *unhideFollowersResponse) Populate(ctx context.Context, userID string, e *SlackListener) error {
	c.UserID = userID
	_, err := e.followsRPC.RestoreAllFollows(ctx, &followsrpc.UserIDReq{
		UserId: userID,
	})
	return err
}

var unhideFollowersResp = template.Must(template.New("restore_follows").Parse(unhideFollowersText))

const unhideFollowersText = "```" + `
Restoring follows for User: {{.UserID}}
You can follow along progress with
/graphdb counts {{.UserID}}
` + "```"
