package accesslog

import (
	"context"
	"math/rand"
	"sync"
	"time"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/interngraphdb"
	"code.justin.tv/hygienic/messagebatch/ext/cwlogevent"
	"github.com/golang/protobuf/jsonpb"
)

// AccessLog adds cloudwatch log events so we can track access to graphdb.
type AccessLog struct {
	// Rand allows us to sample which read requests we log, since otherwise we would log too much stuff
	Rand *rand.Rand
	// CloudwatchLogBatcher groups together cloudwatch log events into a single API call
	cwlogevent.CloudwatchLogBatcher `nilcheck:"ignore"`

	randMu sync.Mutex
}

// If we logged all our read traffic, it would be too much.  Instead, we filter these requests and only log 100% of
// write traffic
var filteredMethods = map[string]struct{}{
	"BulkGetAssoc": {},
	"GetAssoc":     {},
	"CountAssoc":   {},
}

// shouldFilter will return true if we should *not* send an access log to cloudwatch
func (a *AccessLog) shouldFilter(msg *interngraphdb.AccessLog) bool {
	if _, exists := filteredMethods[msg.Method]; !exists {
		return false
	}
	// Do not filter requests slowe than 1 sec
	if msg.GetLatencyMs() > 1000 {
		return false
	}
	a.randMu.Lock()
	defer a.randMu.Unlock()
	// Of all these, filter 99% of them
	return a.Rand.Float64() >= .01
}

// Event sends an event to cloudwatch logs as a JSON blob.  The individual fields are documented in the protobuf file
func (a *AccessLog) Event(ctx context.Context, msg *interngraphdb.AccessLog) {
	if a.shouldFilter(msg) {
		return
	}
	if traceStrings, traceInts, floats, exists := traceValues(ctx); exists {
		if len(traceStrings) != 0 {
			msg.TraceStrings = traceStrings
		}
		if len(traceInts) != 0 {
			msg.TraceInts = traceInts
		}
		if len(floats) != 0 {
			msg.TraceFloats = floats
		}
	}
	e := jsonpb.Marshaler{
		OrigName: true,
	}
	s, err := e.MarshalToString(msg)
	if err != nil {
		a.CloudwatchLogBatcher.Log.Log("err", err, "could not marshal access log")
	}
	msgTime := time.Time{}
	if msg.MsgTime != nil {
		msgTime = time.Unix(msg.MsgTime.Seconds, int64(msg.MsgTime.Nanos))
	}
	a.CloudwatchLogBatcher.Event(s, msgTime)
}
