package repair

import (
	"context"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage"
	"code.justin.tv/feeds/graphdb/proto/graphdbadmin"
	"code.justin.tv/feeds/log"
	"code.justin.tv/hygienic/statsdsender"
)

// Repair fixes the "counts" dynamodb table for an edge type.  It is a best attempt fix and does no locking.  All it
// does is list every edge of a type from a user, count the number of listed edges, and overwrite the stored count value
// with this value.  Ideally we should never have to call this, but we can if we suspect the counts table is out of sync
type Repair struct {
	Log          log.Logger
	ExistingData storage.Client
	Stats        *statsdsender.ErrorlessStatSender `nilcheck:"nodepth"`
}

// RepairEdgeCount iterates each edge kind from a node.  This could take a very long time for large users.  It counts
// all the edges and overwrites the count table with the new value.
func (r *Repair) RepairEdgeCount(ctx context.Context, from graphdbmodel.Node, edgeKind string) (*graphdbadmin.RepairEdgeCountResponse, error) {
	cursor := ""
	var updatedCount int64
	// page through followers/followee's till the end to get an estimate of the count.
	// the computed count can be off in the case event that a row we have previously paged through was deleted while the loop is in process.
	// we estimate the likelihood of this to be very low.
	retryCnt := 0
	for {
		pr := graphdbmodel.PagedRequest{
			Cursor: cursor,
		}
		result, err := r.ExistingData.List(ctx, from, edgeKind, pr)
		if err != nil {
			if strings.Contains(err.Error(), "SerializationError") {
				// fail after 5 consecutive failures
				if retryCnt > 5 {
					return nil, err
				}
				retryCnt += 1
				// linear backoff
				time.Sleep(100 * time.Duration(retryCnt) * time.Millisecond)
				continue
			}
			return nil, err
		}
		retryCnt = 0
		updatedCount += int64(len(result.To))

		cursor = result.Cursor
		if cursor == "" {
			break
		}
	}

	oldCount, err := r.ExistingData.Count(ctx, from, edgeKind)
	if err != nil {
		return nil, err
	}

	if updatedCount == oldCount {
		r.Stats.IncC(fmt.Sprintf("%s.count.match", edgeKind), 1, 1.0)
		return &graphdbadmin.RepairEdgeCountResponse{
			UpdatedCount: int32(updatedCount),
			OldCount:     int32(oldCount),
		}, nil
	}

	oldCount, err = r.ExistingData.OverrideCount(ctx, from, edgeKind, updatedCount)
	if err != nil {
		return nil, err
	}

	r.Stats.IncC(fmt.Sprintf("%s.count.mismatch", edgeKind), 1, 1.0)
	return &graphdbadmin.RepairEdgeCountResponse{
		UpdatedCount: int32(updatedCount),
		OldCount:     int32(oldCount),
	}, nil
}
