package httpapi

import (
	"context"
	"net/http"
)

type ctxHeader int

const (
	reqHeaderKey ctxHeader = iota
)

// GetReqHeaders is used by the access logs to add HTTP headers to our access logs
func GetReqHeaders(ctx context.Context) map[string]string {
	headers, ok := ctx.Value(reqHeaderKey).(map[string]string)
	if !ok {
		return nil
	}
	return headers
}

// withReqHeaders wraps a Handler to add all of the `names` HTTP headers into each request's context so we can later add
// them to the access log.
func withReqHeaders(base http.Handler, names ...string) http.Handler {
	return http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
		ctx := req.Context()
		headers := make(map[string]string, len(names))
		for _, n := range names {
			if v := req.Header.Get(n); v != "" {
				headers[n] = v
			}
		}
		ctx = context.WithValue(ctx, reqHeaderKey, headers)
		req = req.WithContext(ctx)
		base.ServeHTTP(rw, req)
	})
}
