package httpapi

import (
	"context"
	"io"
	"net"
	"net/http"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/graphdb/proto/datastorerpc"
	"code.justin.tv/feeds/graphdb/proto/dynamoevent"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/graphdb/proto/graphdbadmin"
	"code.justin.tv/feeds/log"
)

type Config struct {
	// ListenAddr is the address to listen on
	ListenAddr *distconf.Str
	// ShutdownWait is the timeout for graceful shutdown
	ShutdownWait *distconf.Duration
}

const headerXForwardFor = "X-Forwarded-For"
const HeaderXCallerService = "X-Caller-Service"

// Verify config from distconf
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.ListenAddr = dconf.Str("graphdb.listen_addr", ":8000")
	c.ShutdownWait = dconf.Duration("graphdb.shutdown_wait", time.Second*5)
	return nil
}

// Server is the larger HTTP listener that callers use to interact with both the twirp API and the older API
type Server struct {
	Log                 *log.ElevatedLog
	Config              *Config
	Twirp               graphdb.TwirpServer
	DynamoEvents        dynamoevent.TwirpServer
	DatastoreReadTwirp  datastorerpc.TwirpServer
	DatastoreWriteTwirp datastorerpc.TwirpServer
	AdminTwirp          graphdbadmin.TwirpServer
	Server              http.Server
	OnListen            func(net.Addr) `nilcheck:"ignore"`
	listener            net.Listener
}

// Setup registers all the HTTPHandlers on the server's mux
func (s *Server) Setup() error {
	mux := http.ServeMux{}
	mux.Handle("/debug/running", http.HandlerFunc(s.Health))

	// These 5 are the twirp handlers
	mux.Handle(graphdb.GraphDBPathPrefix, withReqHeaders(s.Twirp, headerXForwardFor, HeaderXCallerService))
	mux.Handle(dynamoevent.DynamoUpdatesPathPrefix, withReqHeaders(s.DynamoEvents, headerXForwardFor, HeaderXCallerService))
	mux.Handle(datastorerpc.ReaderPathPrefix, withReqHeaders(s.DatastoreReadTwirp, headerXForwardFor, HeaderXCallerService))
	mux.Handle(datastorerpc.WriterPathPrefix, withReqHeaders(s.DatastoreWriteTwirp, headerXForwardFor, HeaderXCallerService))
	mux.Handle(graphdbadmin.GraphDBAdminPathPrefix, withReqHeaders(s.AdminTwirp, headerXForwardFor, HeaderXCallerService))

	s.Server = http.Server{
		Handler: &mux,
	}
	var err error
	s.listener, err = net.Listen("tcp", s.Config.ListenAddr.Get())
	if err == nil && s.OnListen != nil {
		s.OnListen(s.listener.Addr())
	}
	return err
}

// Health allows us to health check that the server is up and running.  It should return 'OK'
func (s *Server) Health(rw http.ResponseWriter, req *http.Request) {
	if _, err := io.WriteString(rw, "OK"); err != nil {
		s.Log.LogCtx(req.Context(), "err", err)
	}
}

// Start serves requests on the listening port
func (s *Server) Start() error {
	s.Log.Log("addr", s.listener.Addr().String(), "starting listen on server")
	err := s.Server.Serve(s.listener)
	s.Log.Log("err", err, "server finished listening")
	return err
}

func (s *Server) String() string {
	return "<httpapi.Server>"
}

// Close stops the running http server
func (s *Server) Close() error {
	shutdownCtx, cancel := context.WithTimeout(context.Background(), s.Config.ShutdownWait.Get())
	defer cancel()
	return s.Server.Shutdown(shutdownCtx)
}
