package twirpserver

import (
	"context"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/accesslog"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/proto/graphdbadmin"
	"code.justin.tv/hygienic/errors"
)

// Admin implements GraphDB's administrator interface and allows us to expose administrator functions on their own twirp
// RPC interface
type Admin struct {
	Resyncer          Resyncer
	RepairEdgeCounter RepairEdgeCounter
}

var _ graphdbadmin.GraphDBAdmin = &Admin{}

// Resyncer is a cohesion only interface that rewrites from cohesion into graphdb
type Resyncer interface {
	Resync(ctx context.Context, from graphdbmodel.Node, edgeKind string) (*graphdbadmin.ResyncResponse, error)
}

// RepairEdgeCounter fixes DynamoDB's counts table
type RepairEdgeCounter interface {
	RepairEdgeCount(ctx context.Context, from graphdbmodel.Node, edgeKind string) (*graphdbadmin.RepairEdgeCountResponse, error)
}

// Resync from cohesion to graphdb
func (s *Admin) Resync(ctx context.Context, req *graphdbadmin.ResyncRequest) (*graphdbadmin.ResyncResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	if s.Resyncer == nil {
		return nil, errors.New("GraphDB does not resync with Cohesion anymore")
	}
	return s.Resyncer.Resync(ctx, FromProtoEntity(req.From), req.EdgeKind)
}

// RepairEdgeCount fixes count table for DynamoDB
func (s *Admin) RepairEdgeCount(ctx context.Context, req *graphdbadmin.RepairEdgeCountRequest) (*graphdbadmin.RepairEdgeCountResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	return s.RepairEdgeCounter.RepairEdgeCount(ctx, FromProtoEntity(req.From), req.EdgeKind)
}
