package twirpserver

import (
	"time"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"github.com/golang/protobuf/ptypes/timestamp"
)

// FromProtoEntity converts from our public protobuf node interface into the internal node model we use in our internal abstractions
func FromProtoEntity(ent *graphdb.Node) graphdbmodel.Node {
	if ent == nil {
		return graphdbmodel.Node{}
	}
	return graphdbmodel.Node{
		Type: ent.Type,
		ID:   ent.Id,
	}
}

// ToProtoEntity converts from the private Node to the public protobuf Node
func ToProtoEntity(ent graphdbmodel.Node) *graphdb.Node {
	return &graphdb.Node{
		Type: ent.Type,
		Id:   ent.ID,
	}
}

// FromProtoPage converts from the public protobuf PagedRequest to the private PagedRequest
func FromProtoPage(pagedRequest *graphdb.PagedRequest) graphdbmodel.PagedRequest {
	if pagedRequest == nil {
		return graphdbmodel.PagedRequest{}
	}
	return graphdbmodel.PagedRequest{
		Cursor:          pagedRequest.Cursor,
		Limit:           pagedRequest.Limit,
		DescendingOrder: pagedRequest.Order == graphdb.PagedRequest_DESC,
	}
}

// FromProtoDataBag converts from our proto DataBag to our internal one
func FromProtoDataBag(in *graphdb.DataBag) *graphdbmodel.DataBag {
	ret := &graphdbmodel.DataBag{}
	for k, v := range in.GetStrings() {
		ret.AddString(k, v)
	}
	for k, v := range in.GetBools() {
		ret.AddBoolean(k, v)
	}
	for k, v := range in.GetInts() {
		ret.AddInt(k, v)
	}
	for k, v := range in.GetDoubles() {
		ret.AddDouble(k, v)
	}
	return ret
}

// ToProtoDataBag converts from our internal DataBag into the protobuf one
func ToProtoDataBag(in *graphdbmodel.DataBag) *graphdb.DataBag {
	ints, strings, bools, doubles := in.Types()
	return &graphdb.DataBag{
		Ints:    ints,
		Strings: strings,
		Bools:   bools,
		Doubles: doubles,
	}
}

// ToProtoLoadedEdge converts from our internal LoadedEdge to the public LoadedEdge
func ToProtoLoadedEdge(association *graphdbmodel.LoadedEdge) *graphdb.LoadedEdge {
	if association == nil {
		return nil
	}
	ret := &graphdb.LoadedEdge{
		Edge: &graphdb.Edge{
			From: ToProtoEntity(association.From),
			To:   ToProtoEntity(association.To),
			Type: association.Type,
		},
		Data: ToProtoData(association.LoadedData),
	}
	return ret
}

// ToProtoLoadedNode converts from our internal LoadedNode to the public LoadedNode
func ToProtoLoadedNode(node *graphdbmodel.LoadedNode) *graphdb.LoadedNode {
	if node == nil {
		return nil
	}
	ret := &graphdb.LoadedNode{
		Node: ToProtoEntity(node.Node),
		Data: ToProtoData(node.Data),
	}
	return ret
}

// ToProtoLoadedCursoredEdge converts from the internal CursoredLoadedEdge to the public LoadedCursoredEdge
func ToProtoLoadedCursoredEdge(association *graphdbmodel.CursoredLoadedEdge) *graphdb.LoadedCursoredEdge {
	if association == nil {
		return nil
	}

	ret := &graphdb.LoadedCursoredEdge{
		Edge:   ToProtoLoadedEdge(&association.LoadedEdge),
		Cursor: association.Cursor,
	}
	return ret
}

// ToProtoLoadedCursoredNode converts from the internal CursoredLoadedNode to the public LoadedCursoredNode
func ToProtoLoadedCursoredNode(node *graphdbmodel.CursoredLoadedNode) *graphdb.LoadedCursoredNode {
	if node == nil {
		return nil
	}

	ret := &graphdb.LoadedCursoredNode{
		Node:   ToProtoLoadedNode(&node.LoadedNode),
		Cursor: node.Cursor,
	}
	return ret
}

// FromProtoTime converts from a public Timestamp to the public time.Time
func FromProtoTime(t *timestamp.Timestamp) time.Time {
	if t == nil {
		return time.Time{}
	}
	return time.Unix(t.Seconds, int64(t.Nanos))
}

// FromProtoTimePointer converts from the protobuf Timestamp to a pointer of the internal time
func FromProtoTimePointer(t *timestamp.Timestamp) *time.Time {
	x := FromProtoTime(t)
	if x.IsZero() {
		return nil
	}
	return &x
}

// ToProtoTime converts from a public time.Time to the protobuf version
func ToProtoTime(t time.Time) *timestamp.Timestamp {
	unixNano := t.UnixNano()
	return &timestamp.Timestamp{
		Seconds: unixNano / time.Second.Nanoseconds(),
		Nanos:   int32(unixNano % time.Second.Nanoseconds()),
	}
}

// ToProtoData converts from the internal LoadedData to the public Data
func ToProtoData(in graphdbmodel.LoadedData) *graphdb.Data {
	return &graphdb.Data{
		CreatedAt: ToProtoTime(in.CreatedAt),
		UpdatedAt: ToProtoTime(in.UpdatedAt),
		Version:   in.Version,
		Data:      ToProtoDataBag(in.Data),
	}
}

// FromProtoEdge converts from the public Edge to the internal Edge type
func FromProtoEdge(edge *graphdb.Edge) graphdbmodel.Edge {
	if edge == nil {
		return graphdbmodel.Edge{}
	}
	return graphdbmodel.Edge{
		Type: edge.Type,
		From: FromProtoEntity(edge.From),
		To:   FromProtoEntity(edge.To),
	}
}

// ToProtoEdge converts from the internal Edge to the public Edge
func ToProtoEdge(edge *graphdbmodel.Edge) *graphdb.Edge {
	if edge == nil {
		return nil
	}
	return &graphdb.Edge{
		Type: edge.Type,
		From: ToProtoEntity(edge.From),
		To:   ToProtoEntity(edge.To),
	}
}

// FromProtoRequiredVersion converts from the public RequiredVersion to a time.Time pointer
func FromProtoRequiredVersion(version *graphdb.RequiredVersion) *time.Time {
	if version == nil {
		return nil
	}
	return FromProtoTimePointer(version.UpdatedAt)
}
