package twirpserver

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/accesslog"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/async"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/edgechange"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/interngraphdb"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/sns"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage/complexstorage"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/statsdsender"
	"code.justin.tv/hygienic/workerpool"
	"github.com/twitchtv/twirp"
)

// Server implements the public GraphDB twirp interface.  It generally shouldn't contain much logic.  The meat of node
// and edge creation is inside the Storage and NodeStorage abstractions.  Most of what this does is translate between
// data formats.
type Server struct {
	// Storage is the edge storage layer
	Storage *complexstorage.Storage
	// NodeStorage is the node storage layer
	NodeStorage *storage.NodeStorage
	// Async handles requests that we intend to finish later (not in band with the request)
	Async *async.Queue
	// AsyncMultiPool isn't actually used by Async.  Instead, it is used to throttle how many Multi requests (not async
	// multi requests) that we do at once.
	AsyncMultiPool workerpool.Pool
	// UpdateAllTypesMsgSender is the SQS queue processor that modifies all edges from one node type to another
	UpdateAllTypesMsgSender *edgechange.MsgSender

	Stats *statsdsender.ErrorlessStatSender

	SNSClient *sns.SNSClient
}

var _ graphdb.GraphDB = &Server{}

func (s *Server) EdgeUpdateAllEdgesFromNode(ctx context.Context, req *graphdb.EdgeUpdateAllEdgesFromNodeRequest) (*graphdb.EdgeUpdateAllEdgesFromNodeResponse, error) {
	return &graphdb.EdgeUpdateAllEdgesFromNodeResponse{}, s.UpdateAllTypesMsgSender.SendMessage(ctx, &interngraphdb.ChangeAllEdgeTypes{
		Source:  req.From,
		OldType: req.OldType,
		NewType: req.NewType,
	})
}

// MultiAsync should return almost instantly, after adding the async request to an SQS queue that we will process later.
func (s *Server) MultiAsync(ctx context.Context, req *graphdb.MultiAsyncRequest) (*graphdb.MultiAsyncResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	msg := &interngraphdb.AsyncRequestQueueMessage{
		Requests: req.Requests,
	}
	if err := s.Async.SendMessage(ctx, msg, 0); err != nil {
		return nil, err
	}
	return &graphdb.MultiAsyncResponse{}, nil
}

// EdgeCreate stores a new edge
func (s *Server) EdgeCreate(ctx context.Context, req *graphdb.EdgeCreateRequest) (*graphdb.EdgeCreateResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	newItem, err := s.Storage.Create(ctx, FromProtoEdge(req.Edge), FromProtoDataBag(req.Data), FromProtoTime(req.CreatedAt))
	if err != nil {
		fmt.Println("err", err)
		return nil, err
	}
	return &graphdb.EdgeCreateResponse{
		Edge: ToProtoLoadedEdge(newItem),
	}, nil
}

// EdgeUpdateType changes the type of an edge (removing the old edge and creating another)
func (s *Server) EdgeUpdateType(ctx context.Context, req *graphdb.EdgeUpdateTypeRequest) (*graphdb.EdgeUpdateTypeResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	newItem, err := s.Storage.ChangeType(ctx, FromProtoEdge(req.Edge), req.NewType)
	if err != nil {
		return nil, err
	}
	return &graphdb.EdgeUpdateTypeResponse{
		Edge: ToProtoLoadedEdge(newItem),
	}, nil
}

// multiEdgeCount is used during multi requests: it wraps a count request
func (s *Server) multiEdgeCount(ctx context.Context, singleReq *graphdb.EdgeCountRequest) *graphdb.MultiResponse_SingleResponse {
	res, err := s.EdgeCount(ctx, singleReq)
	if err != nil {
		return &graphdb.MultiResponse_SingleResponse{
			ResponseType: &graphdb.MultiResponse_SingleResponse_Error{
				Error: &graphdb.Error{
					Message: err.Error(),
				},
			},
		}
	}
	return &graphdb.MultiResponse_SingleResponse{
		ResponseType: &graphdb.MultiResponse_SingleResponse_EdgeCount{
			EdgeCount: res,
		},
	}
}

// multiEdgeCreate is used during multi requests: it wraps an edge creation request
func (s *Server) multiEdgeCreate(ctx context.Context, singleReq *graphdb.EdgeCreateRequest) *graphdb.MultiResponse_SingleResponse {
	res, err := s.EdgeCreate(ctx, singleReq)
	if err != nil {
		return &graphdb.MultiResponse_SingleResponse{
			ResponseType: &graphdb.MultiResponse_SingleResponse_Error{
				Error: &graphdb.Error{
					Message: err.Error(),
				},
			},
		}
	}
	return &graphdb.MultiResponse_SingleResponse{
		ResponseType: &graphdb.MultiResponse_SingleResponse_EdgeCreate{
			EdgeCreate: res,
		},
	}
}

// multiEdgeDelete is used during multi requests: it wraps an edge deletion request
func (s *Server) multiEdgeDelete(ctx context.Context, singleReq *graphdb.EdgeDeleteRequest) *graphdb.MultiResponse_SingleResponse {
	res, err := s.EdgeDelete(ctx, singleReq)
	if err != nil {
		return &graphdb.MultiResponse_SingleResponse{
			ResponseType: &graphdb.MultiResponse_SingleResponse_Error{
				Error: &graphdb.Error{
					Message: err.Error(),
				},
			},
		}
	}
	return &graphdb.MultiResponse_SingleResponse{
		ResponseType: &graphdb.MultiResponse_SingleResponse_EdgeDelete{
			EdgeDelete: res,
		},
	}
}

// multiEdgeUpdate is used during multi requests: it wraps an edge update request
func (s *Server) multiEdgeUpdate(ctx context.Context, singleReq *graphdb.EdgeUpdateRequest) *graphdb.MultiResponse_SingleResponse {
	res, err := s.EdgeUpdate(ctx, singleReq)
	if err != nil {
		return &graphdb.MultiResponse_SingleResponse{
			ResponseType: &graphdb.MultiResponse_SingleResponse_Error{
				Error: &graphdb.Error{
					Message: err.Error(),
				},
			},
		}
	}
	return &graphdb.MultiResponse_SingleResponse{
		ResponseType: &graphdb.MultiResponse_SingleResponse_EdgeUpdate{
			EdgeUpdate: res,
		},
	}
}

// multiEdgeGet is used during multi requests: it wraps an edge get request
func (s *Server) multiEdgeGet(ctx context.Context, singleReq *graphdb.EdgeGetRequest) *graphdb.MultiResponse_SingleResponse {
	res, err := s.EdgeGet(ctx, singleReq)
	if err != nil {
		return &graphdb.MultiResponse_SingleResponse{
			ResponseType: &graphdb.MultiResponse_SingleResponse_Error{
				Error: &graphdb.Error{
					Message: err.Error(),
				},
			},
		}
	}
	return &graphdb.MultiResponse_SingleResponse{
		ResponseType: &graphdb.MultiResponse_SingleResponse_EdgeGet{
			EdgeGet: res,
		},
	}
}

// multiEdgeChangeType is used during multi requests: it wraps an edge change type request
func (s *Server) multiEdgeChangeType(ctx context.Context, singleReq *graphdb.EdgeUpdateTypeRequest) *graphdb.MultiResponse_SingleResponse {
	res, err := s.EdgeUpdateType(ctx, singleReq)
	if err != nil {
		return &graphdb.MultiResponse_SingleResponse{
			ResponseType: &graphdb.MultiResponse_SingleResponse_Error{
				Error: &graphdb.Error{
					Message: err.Error(),
				},
			},
		}
	}
	return &graphdb.MultiResponse_SingleResponse{
		ResponseType: &graphdb.MultiResponse_SingleResponse_EdgeChangeType{
			EdgeChangeType: res,
		},
	}
}

// singleMultiRequest holds AsyncMultiPool work that we want to eventually do.  It can be thought of as a function call
// and the result of the function call is the singleMultiResponse
type singleMultiRequest struct {
	ctx       context.Context
	singleReq *graphdb.MultiRequest_SingleRequest
	s         *Server
}

// singleMultiResponse is the result of calling the function represented as singleMultiRequest
type singleMultiResponse struct {
	resp *graphdb.MultiResponse_SingleResponse
	err  error
}

// do executes the function wrapped as singleMultiRequest.  This really is just a type switch around singleReq
func (r *singleMultiRequest) do() interface{} {
	ctx := r.ctx
	select {
	case <-ctx.Done():
		return singleMultiResponse{nil, ctx.Err()}
	default:
	}
	if r.singleReq.GetEdgeCreate() != nil {
		return singleMultiResponse{r.s.multiEdgeCreate(ctx, r.singleReq.GetEdgeCreate()), nil}
	}
	if r.singleReq.GetEdgeCount() != nil {
		return singleMultiResponse{r.s.multiEdgeCount(ctx, r.singleReq.GetEdgeCount()), nil}
	}
	if r.singleReq.GetEdgeDelete() != nil {
		return singleMultiResponse{r.s.multiEdgeDelete(ctx, r.singleReq.GetEdgeDelete()), nil}
	}
	if r.singleReq.GetEdgeUpdate() != nil {
		return singleMultiResponse{r.s.multiEdgeUpdate(ctx, r.singleReq.GetEdgeUpdate()), nil}
	}
	if r.singleReq.GetEdgeGet() != nil {
		return singleMultiResponse{r.s.multiEdgeGet(ctx, r.singleReq.GetEdgeGet()), nil}
	}
	if r.singleReq.GetEdgeChangeType() != nil {
		return singleMultiResponse{r.s.multiEdgeChangeType(ctx, r.singleReq.GetEdgeChangeType()), nil}
	}
	return singleMultiResponse{nil, errors.New("unknown request type: " + r.singleReq.String())}
}

// Multi executes each of the requests inside req.Requests.  It blocks until they are all done.  It will do many of them
// in parallel, and uses a worker pool to limit how many multi requests are done at once.
//
// There is no guarantee as to what order the requests will be done in.  Multi should not be considered atomic.
func (s *Server) Multi(ctx context.Context, req *graphdb.MultiRequest) (*graphdb.MultiResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	resp := &graphdb.MultiResponse{
		Response: make([]*graphdb.MultiResponse_SingleResponse, len(req.Requests)),
	}

	futures := make([]*workerpool.Future, len(req.Requests))
	for i := range req.Requests {
		req := singleMultiRequest{
			ctx:       ctx,
			s:         s,
			singleReq: req.Requests[i],
		}
		futures[i] = s.AsyncMultiPool.Put(ctx, req.do)
	}
	for i, future := range futures {
		select {
		case <-ctx.Done():
			return nil, ctx.Err()
		case <-future.Done():
			futureResult := future.Result().(singleMultiResponse)
			if futureResult.err != nil {
				return nil, futureResult.err
			}
			resp.Response[i] = futureResult.resp
		}
	}
	return resp, nil
}

// EdgeUpdate changes and edge's data.  Depending upon the OverwriteData field of the request, it could force a creation
// , or fail if the edge doesn't already exist.
func (s *Server) EdgeUpdate(ctx context.Context, req *graphdb.EdgeUpdateRequest) (*graphdb.EdgeUpdateResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	var newItem *graphdbmodel.LoadedEdge
	var err error

	if req.OverwriteData {
		newItem, err = s.Storage.Put(ctx, FromProtoEdge(req.Edge), FromProtoDataBag(req.Data), FromProtoTimePointer(req.CreatedAt), FromProtoRequiredVersion(req.Version))
	} else {
		newItem, err = s.Storage.Update(ctx, FromProtoEdge(req.Edge), FromProtoDataBag(req.Data), FromProtoTimePointer(req.CreatedAt), FromProtoRequiredVersion(req.Version))
	}
	if err != nil {
		return nil, err
	}
	return &graphdb.EdgeUpdateResponse{
		Edge: ToProtoLoadedEdge(newItem),
	}, nil
}

// EdgeCount counts an edge type from a node
func (s *Server) EdgeCount(ctx context.Context, req *graphdb.EdgeCountRequest) (*graphdb.EdgeCountResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	c, err := s.Storage.Count(ctx, FromProtoEntity(req.From), req.EdgeType)
	if err != nil {
		return nil, err
	}

	edgeCountResponse := &graphdb.EdgeCountResponse{
		Count: c,
	}

	if sns.SendReadRequest() {
		go func() {
			err := s.SNSClient.SendEdgeCountRequestMessage(req, edgeCountResponse)
			if err != nil {
				s.Stats.IncC("send_edge_count_request_message.count.err", 1, 1.0)
			}
		}()
	}

	return edgeCountResponse, nil
}

// EdgeList returns edges out of a node
func (s *Server) EdgeList(ctx context.Context, req *graphdb.EdgeListRequest) (*graphdb.EdgeListResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	resp, err := s.Storage.List(ctx, FromProtoEntity(req.From), req.EdgeType, FromProtoPage(req.Page))
	if err != nil {
		return nil, err
	}
	retAssocs := make([]*graphdb.LoadedCursoredEdge, 0, len(resp.To))
	for _, r := range resp.To {
		retAssocs = append(retAssocs, ToProtoLoadedCursoredEdge(&r))
	}

	edgeListResponse := &graphdb.EdgeListResponse{
		Cursor: resp.Cursor,
		Edges:  retAssocs,
	}

	if sns.SendReadRequest() {
		go func() {
			err := s.SNSClient.SendEdgeListRequestMessage(req, edgeListResponse)
			if err != nil {
				s.Stats.IncC("send_edge_list_request_message.count.err", 1, 1.0)
			}
		}()
	}

	return edgeListResponse, nil
}

// EdgeDelete removes an edge
func (s *Server) EdgeDelete(ctx context.Context, req *graphdb.EdgeDeleteRequest) (*graphdb.EdgeDeleteResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)

	oldItem, err := s.Storage.Delete(ctx, FromProtoEdge(req.Edge), FromProtoRequiredVersion(req.Version))
	if err != nil {
		return nil, err
	}
	return &graphdb.EdgeDeleteResponse{
		Edge: ToProtoLoadedEdge(oldItem),
	}, nil
}

// EdgeGet returns an edge, if it exists.
func (s *Server) EdgeGet(ctx context.Context, req *graphdb.EdgeGetRequest) (*graphdb.EdgeGetResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	resp, err := s.Storage.Get(ctx, FromProtoEdge(req.Edge))
	if err != nil {
		return nil, err
	}

	edgeGetResponse := &graphdb.EdgeGetResponse{
		Edge: ToProtoLoadedEdge(resp),
	}

	if sns.SendReadRequest() {
		go func() {
			err := s.SNSClient.SendEdgeGetRequestMessage(req, edgeGetResponse)
			if err != nil {
				s.Stats.IncC("send_edge_get_request_message.count.err", 1, 1.0)
			}
		}()
	}

	return edgeGetResponse, nil
}

// BatchEdgeGet returns a list of edges if it exists
func (s *Server) BatchEdgeGet(ctx context.Context, req *graphdb.BatchEdgeGetRequest) (*graphdb.BatchEdgeGetResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	ret := make([]*graphdb.LoadedEdge, 0, len(req.Edges))
	for _, edge := range req.Edges {
		resp, err := s.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: edge,
		})
		if err != nil {
			return nil, err
		}

		if resp.Edge != nil {
			ret = append(ret, resp.Edge)
		}
	}
	return &graphdb.BatchEdgeGetResponse{
		Edges: ret,
	}, nil
}

// BulkEdgeUpdate updates all associations from a source node with a given edge type
func (s *Server) BulkEdgeUpdate(ctx context.Context, req *graphdb.BulkEdgeUpdateRequest) (*graphdb.BulkEdgeUpdateResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	from := FromProtoEntity(req.From)
	entKind := from.Type
	data := FromProtoDataBag(req.Data)

	go func() {
		reqCtx, cancel := context.WithTimeout(context.Background(), 4*time.Hour)
		defer cancel()
		start := time.Now()
		if err := s.Storage.BulkUpdateAssoc(reqCtx, from, req.EdgeType, entKind, data, req.NewEdgeType); err != nil {
			s.Stats.TimingDurationC("bulk_edge_update.err", time.Since(start), 1.0)
			s.Stats.IncC("bulk_edge_update.count.err", 1, 1.0)
			return
		}

		s.Stats.TimingDurationC("bulk_edge_update.success", time.Since(start), 1.0)
		s.Stats.IncC("bulk_edge_update.count.success", 1, 1.0)
	}()

	return &graphdb.BulkEdgeUpdateResponse{}, nil
}

// BulkEdgeDelete deletes all associations from a source node with a given edge type
func (s *Server) BulkEdgeDelete(ctx context.Context, req *graphdb.BulkEdgeDeleteRequest) (*graphdb.BulkEdgeDeleteResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	from := FromProtoEntity(req.From)
	entKind := from.Type

	go func() {
		reqCtx, cancel := context.WithTimeout(context.Background(), 4*time.Hour)
		defer cancel()
		start := time.Now()
		if err := s.Storage.BulkDelete(reqCtx, from, req.EdgeType, entKind); err != nil {
			s.Stats.TimingDurationC("bulk_edge_delete.err", time.Since(start), 1.0)
			s.Stats.IncC("bulk_edge_delete.err", 1, 1.0)
			return
		}

		s.Stats.TimingDurationC("bulk_edge_delete.success", time.Since(start), 1.0)
		s.Stats.IncC("bulk_edge_delete.count.success", 1, 1.0)
	}()

	return &graphdb.BulkEdgeDeleteResponse{}, nil
}

func (s *Server) BulkEdgeUpdateV2(ctx context.Context, req *graphdb.BulkEdgeUpdateRequestV2) (*graphdb.BulkEdgeUpdateResponseV2, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	if req.Limit > 2000 || req.Limit <= 0 {
		return nil, twirp.InvalidArgumentError("Limit", "must be between 1 and 2000")
	}
	from := FromProtoEntity(req.From)

	more, err := s.Storage.BulkUpdateV2(ctx, from, req.EdgeType, req.NewEdgeType, int(req.Limit))
	if err != nil {
		return nil, err
	}
	return &graphdb.BulkEdgeUpdateResponseV2{More: more}, nil
}

func (s *Server) BulkEdgeDeleteV2(ctx context.Context, req *graphdb.BulkEdgeDeleteRequestV2) (*graphdb.BulkEdgeDeleteResponseV2, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	if req.Limit > 2000 || req.Limit <= 0 {
		return nil, twirp.InvalidArgumentError("Limit", "must be between 1 and 2000")
	}
	from := FromProtoEntity(req.From)

	more, err := s.Storage.BulkDeleteV2(ctx, from, req.EdgeType, int(req.Limit))
	if err != nil {
		return nil, err
	}
	return &graphdb.BulkEdgeDeleteResponseV2{More: more}, nil
}

// NodeGet gets a node
func (s *Server) NodeGet(ctx context.Context, req *graphdb.NodeGetRequest) (*graphdb.NodeGetResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	resp, err := s.NodeStorage.NodeGet(ctx, FromProtoEntity(req.Node))
	if err != nil {
		return nil, err
	}
	return &graphdb.NodeGetResponse{
		Node: ToProtoLoadedNode(resp),
	}, nil
}

// NodeCreate creates a node
func (s *Server) NodeCreate(ctx context.Context, req *graphdb.NodeCreateRequest) (*graphdb.NodeCreateResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	resp, err := s.NodeStorage.NodeCreate(ctx, FromProtoEntity(req.Node), FromProtoDataBag(req.Data), FromProtoTime(req.CreatedAt))
	if err != nil {
		return nil, err
	}
	return &graphdb.NodeCreateResponse{
		Node: ToProtoLoadedNode(resp),
	}, nil
}

// NodeUpdate updates a node, if it exists (depending upon the OverwriteData field)
func (s *Server) NodeUpdate(ctx context.Context, req *graphdb.NodeUpdateRequest) (*graphdb.NodeUpdateResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	resp, err := s.NodeStorage.NodeUpdateOrPut(ctx, FromProtoEntity(req.Node), FromProtoDataBag(req.Data), !req.OverwriteData)
	if err != nil {
		return nil, err
	}
	return &graphdb.NodeUpdateResponse{
		Node: ToProtoLoadedNode(resp),
	}, nil
}

// NodeCount counts nodes of a type
func (s *Server) NodeCount(ctx context.Context, req *graphdb.NodeCountRequest) (*graphdb.NodeCountResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	resp, err := s.NodeStorage.NodeCount(ctx, req.NodeType)
	if err != nil {
		return nil, err
	}
	return &graphdb.NodeCountResponse{
		Count: resp,
	}, nil
}

// NodeDelete removes a node and returns the removed node, if it previously existed
func (s *Server) NodeDelete(ctx context.Context, req *graphdb.NodeDeleteRequest) (*graphdb.NodeDeleteResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	resp, err := s.NodeStorage.NodeDelete(ctx, FromProtoEntity(req.Node))
	if err != nil {
		return nil, err
	}
	return &graphdb.NodeDeleteResponse{
		Node: ToProtoLoadedNode(resp),
	}, nil
}

// NodeList lists all the nodes of a type
func (s *Server) NodeList(ctx context.Context, req *graphdb.NodeListRequest) (*graphdb.NodeListResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	resp, err := s.NodeStorage.NodeList(ctx, req.NodeType, FromProtoPage(req.Page))
	if err != nil {
		return nil, err
	}
	retNodes := make([]*graphdb.LoadedCursoredNode, 0, len(resp.Nodes))
	for _, r := range resp.Nodes {
		retNodes = append(retNodes, ToProtoLoadedCursoredNode(&r))
	}

	return &graphdb.NodeListResponse{
		Cursor: resp.Cursor,
		Nodes:  retNodes,
	}, nil
}
