package cacheinvalidation

import (
	"context"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/cache"
	"code.justin.tv/feeds/log"
	"code.justin.tv/hygienic/statsdsender"
)

type Poller struct {
	Queue       *Queue
	Log         log.Logger
	Stats       *statsdsender.ErrorlessStatSender `nilcheck:"nodepth"`
	closeSignal chan struct{}
	Cache       *cache.MemcacheCache
}

func (p *Poller) Setup() error {
	p.closeSignal = make(chan struct{})
	return nil
}

func (p *Poller) Start() error {
	for {
		select {
		case <-p.closeSignal:
			return nil
		default:
		}

		ctx := context.Background()
		msgs, err := p.Queue.ReceiveMessages(ctx)
		if err != nil {
			p.Stats.IncC("cacheinvalidation.receive_message.failed", 1, 1)
			continue
		}
		for _, msg := range msgs {
			p.processMessage(ctx, msg)
		}
	}
}

func (p *Poller) processMessage(ctx context.Context, msg *Message) {
	if err := p.Cache.InvalidateCache(ctx, msg.Edge); err != nil {
		// retrying message
		p.Log.Log("err", err, "unable to invalidate cache")
		p.Stats.IncC("cacheinvalidation.invalidate_cache.failed", 1, 1)
		return
	}
	p.Stats.IncC("cacheinvalidation.invalidate_cache.success", 1, 1)

	if err := p.Queue.DeleteMessage(ctx, msg); err != nil {
		p.Log.Log("err", err, "unable to remove previous request")
		p.Stats.IncC("cacheinvalidation.deleted_message.failed", 1, 1)
	} else {
		p.Stats.IncC("cacheinvalidation.deleted_message.success", 1, 1)
	}
}

func (p *Poller) Close() error {
	close(p.closeSignal)
	return nil
}
