package cacheinvalidation

import (
	"code.justin.tv/feeds/errors"
	"context"
	"encoding/json"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/log"
	"code.justin.tv/hygienic/statsdsender"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/cep21/circuit"
)

type QueueConfig struct {
	Queue             *distconf.Str
	VisibilityTimeout *distconf.Duration
	WaitTime          *distconf.Duration
}

func (q *QueueConfig) Load(d *distconf.Distconf) error {
	q.Queue = d.Str("graphdb.cache_invalidation.queue_url", "")
	if q.Queue.Get() == "" {
		return errors.New("Unable to find queue URL from cache_invalidation.queue_url")
	}
	q.VisibilityTimeout = d.Duration("graphdb.cache_invalidation.visibility_timeout", time.Minute*5)
	q.WaitTime = d.Duration("graphdb.cache_invalidation.wait_time", time.Second*5)
	return nil
}

// QueueCircuits is the hystrix circuits for all the queue operations we care about
type QueueCircuits struct {
	Receive *circuit.Circuit
	Delete  *circuit.Circuit
}

// Queue abstracts the SQS queue.  We use this to send/receive/mark deleted messages from the queue
type Queue struct {
	SQS         *sqs.SQS
	Stats       *statsdsender.ErrorlessStatSender `nilcheck:"nodepth"`
	QueueConfig *QueueConfig
	Log         log.Logger
	Circuits    QueueCircuits
}

// Message contains both the SQS message we originally got
type Message struct {
	msg  *sqs.Message
	Edge graphdbmodel.Edge
}

// ReceiveMessages reads and decodes messages from the queue
func (q *Queue) ReceiveMessages(ctx context.Context) ([]*Message, error) {
	req, out := q.SQS.ReceiveMessageRequest(&sqs.ReceiveMessageInput{
		QueueUrl:              aws.String(q.QueueConfig.Queue.Get()),
		AttributeNames:        []*string{aws.String("All")},
		MessageAttributeNames: []*string{aws.String("All")},
		VisibilityTimeout:     aws.Int64(int64(q.QueueConfig.VisibilityTimeout.Get().Seconds()) + 1),
		WaitTimeSeconds:       aws.Int64(int64(q.QueueConfig.WaitTime.Get().Seconds()) + 1),
		MaxNumberOfMessages:   aws.Int64(10),
	})
	err := q.Circuits.Receive.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	if err != nil {
		return nil, err
	}

	var ret []*Message
	for _, msg := range out.Messages {
		sqsMessageStruct := struct {
			Edge string `json:"Message"`
		}{}
		if err := json.Unmarshal([]byte(*msg.Body), &sqsMessageStruct); err != nil {
			q.Log.Log("err", err, "invalid sqs message body")
			invalidMsg := &Message{msg: msg}
			if err := q.DeleteMessage(ctx, invalidMsg); err != nil {
				q.Log.Log("err", err, "unable to delete invalid message")
			}
			continue
		}
		var edge graphdbmodel.Edge
		if err := json.Unmarshal([]byte(sqsMessageStruct.Edge), &edge); err != nil {
			q.Log.Log("err", err, "invalid sqs message body edge")
			invalidMsg := &Message{msg: msg}
			if err := q.DeleteMessage(ctx, invalidMsg); err != nil {
				q.Log.Log("err", err, "unable to delete invalid message")
			}
			continue
		}

		ret = append(ret, &Message{
			msg:  msg,
			Edge: edge,
		})
	}
	return ret, nil
}

// DeleteMessage removes a previously read message from the queue
func (q *Queue) DeleteMessage(ctx context.Context, msg *Message) error {
	req, _ := q.SQS.DeleteMessageRequest(&sqs.DeleteMessageInput{
		QueueUrl:      aws.String(q.QueueConfig.Queue.Get()),
		ReceiptHandle: msg.msg.ReceiptHandle,
	})
	err := q.Circuits.Delete.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	return err
}
