package edgechange

import (
	"context"
	"errors"
	"time"

	"encoding/base64"
	"hash/fnv"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/interngraphdb"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/hygienic/sqsextender"
	"code.justin.tv/hygienic/sqsprocessor"
	"code.justin.tv/hygienic/sqsprocessor/sqsprotoprocessor"
	"code.justin.tv/hygienic/statsdsender"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/cep21/circuit"
	"github.com/golang/protobuf/proto"
)

type Config struct {
	Queue             *distconf.Str
	VisibilityTimeout *distconf.Duration
	WaitTime          *distconf.Duration
	MsgReadDelay      *distconf.Duration
	ListItemLimit     *distconf.Int
}

func (q *Config) Load(d *distconf.Distconf) error {
	q.Queue = d.Str("graphdb.edgechange.queue", "")
	if q.Queue.Get() == "" {
		return errors.New("unable to find distconf value edgechange.queue")
	}
	q.VisibilityTimeout = d.Duration("graphdb.edgechange.visibility", time.Minute*10)
	q.WaitTime = d.Duration("graphdb.edgechange.wait", time.Second*10)
	q.MsgReadDelay = d.Duration("graphdb.edgechange.read_delay", time.Second*15)
	q.ListItemLimit = d.Int("graphdb.cache.list_limit", 2000)
	return nil
}

func New(SQS *sqs.SQS, logger log.Logger, circuits QueueCircuits, config Config, graphDB graphdb.GraphDB, stats *statsdsender.ErrorlessStatSender) (*sqsprocessor.SQSProcessor, *MsgSender) {
	processor := &edgeTypeChanger{
		listItemLimit: config.ListItemLimit,
		graphDB:       graphDB,
		stats:         stats,
	}
	protoProcessor := &sqsprotoprocessor.ProtoProcessor{
		MessageFactory: func() proto.Message {
			return &interngraphdb.ChangeAllEdgeTypes{}
		},
		Processor: processor,
	}
	extender := &sqsextender.SQSMessageTimeoutExtender{
		SQS:          SQS,
		QueueURL:     config.Queue.Get(),
		Circuit:      circuits.Extend,
		ExtendBuffer: config.VisibilityTimeout.Get(),
	}
	extendedProcessor := &sqsextender.ExtendedProcessor{
		SQSMessageTimeoutExtender: extender,
		MessageProcessor:          protoProcessor,
		OnPanic: func(panicVal interface{}) {
			logger.Log("panic processing SQS message")
		},
	}
	ret1 := &sqsprocessor.SQSProcessor{
		SQS: SQS,
		Circuits: sqsprocessor.QueueCircuits{
			Receive: circuits.Receive,
			Send:    circuits.Send,
			Delete:  circuits.Delete,
		},
		Log: logger,
		Config: sqsprocessor.Config{
			QueueURL:          config.Queue.Get(),
			VisibilityTimeout: config.VisibilityTimeout.Get(),
			WaitTime:          config.WaitTime.Get(),
			MsgReadDelay:      config.MsgReadDelay.Get(),
		},
		Processor: extendedProcessor,
	}
	protoSender := &sqsprotoprocessor.ProtoMessageSender{
		MessageSender: ret1,
	}
	ret2 := &MsgSender{
		protoSender: protoSender,
	}
	return ret1, ret2
}

// QueueCircuits is the hystrix circuits for all the queue operations we care about
type QueueCircuits struct {
	Receive *circuit.Circuit
	Extend  *circuit.Circuit
	Delete  *circuit.Circuit
	Send    *circuit.Circuit
}

// Circuits loads all the queue's circuits
func (q *QueueCircuits) Circuits(m *circuit.Manager) {
	q.Receive = m.MustCreateCircuit("edgechange.queue.read", circuit.Config{
		Execution: circuit.ExecutionConfig{
			Timeout: time.Minute,
		},
	})
	q.Extend = m.MustCreateCircuit("edgechange.queue.extend")
	q.Delete = m.MustCreateCircuit("edgechange.queue.delete")
	q.Send = m.MustCreateCircuit("edgechange.queue.send")
}

type edgeTypeChanger struct {
	listItemLimit *distconf.Int
	graphDB       graphdb.GraphDB
	stats         *statsdsender.ErrorlessStatSender
}

func (e *edgeTypeChanger) Process(ctx context.Context, msg *sqsprotoprocessor.Message) error {
	protoMsg := msg.ProtoMessage.(*interngraphdb.ChangeAllEdgeTypes)
	e.stats.IncC("process.msgs", 1, 1.0)
	for {
		allEdges, err := e.graphDB.EdgeList(ctx, &graphdb.EdgeListRequest{
			From:     protoMsg.Source,
			EdgeType: protoMsg.OldType,
			Page: &graphdb.PagedRequest{
				// Force to DB each time
				Limit: e.listItemLimit.Get() + 1,
			},
		})
		if err != nil {
			e.stats.IncC("process.errs", 1, 1.0)
			return err
		}
		if len(allEdges.Edges) == 0 {
			return nil
		}
		for _, edge := range allEdges.Edges {
			_, err := e.graphDB.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
				Edge: &graphdb.Edge{
					From: edge.Edge.Edge.From,
					To:   edge.Edge.Edge.To,
					Type: protoMsg.NewType,
				},
				Data:      edge.Edge.Data.Data,
				CreatedAt: edge.Edge.Data.CreatedAt,
			})
			if err != nil {
				e.stats.IncC("process.errs", 1, 1.0)
				return err
			}
			_, err = e.graphDB.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: edge.Edge.Edge,
			})
			if err != nil {
				e.stats.IncC("process.errs", 1, 1.0)
				return err
			}
			e.stats.IncC("process.edges", 1, 1.0)
		}
	}
}

func strPtr(s string) *string {
	return &s
}

type MsgSender struct {
	protoSender *sqsprotoprocessor.ProtoMessageSender
}

func hashKey(node *graphdb.Node) string {
	h := fnv.New128a()
	b := h.Sum([]byte(node.Type + ":" + node.Id))
	return base64.URLEncoding.EncodeToString(b)
}

func (e *MsgSender) SendMessage(ctx context.Context, msg *interngraphdb.ChangeAllEdgeTypes) error {
	return e.protoSender.SendMessage(ctx, msg, &sqs.SendMessageInput{
		MessageGroupId: strPtr(hashKey(msg.Source)),
	})
}
