package graphdbmodel

import (
	"encoding"
	"strings"
	"time"

	"fmt"

	"code.justin.tv/feeds/errors"
)

// Node represents nodes in the graph
type Node struct {
	Type string `json:"type"`
	ID   string `json:"id"`
}

func (e *Node) String() string {
	if e == nil {
		return "<nil>"
	}
	return fmt.Sprintf("%s:%s", e.Type, e.ID)
}

var _ encoding.TextMarshaler = &Node{}
var _ encoding.TextUnmarshaler = &Node{}

func (e *Node) MarshalText() (text []byte, err error) {
	return []byte(e.Type + ":" + e.ID), nil
}

func (e *Node) Encode() string {
	a, b := e.MarshalText()
	if b != nil {
		// Should never happen
		panic(b)
	}
	return string(a)
}

func (e *Node) UnmarshalText(text []byte) error {
	parts := strings.SplitN(string(text), ":", 2)
	if len(parts) != 2 {
		return errors.Errorf("unable to unmarshal %s", string(text))
	}
	e.Type = parts[0]
	e.ID = parts[1]
	return nil
}

type DataBag struct {
	Ints    map[string]int64
	Strings map[string]string
	Bools   map[string]bool
	Doubles map[string]float64
}

func (d *DataBag) String() string {
	ret := ""
	if d == nil {
		return ret
	}
	for k, v := range d.Ints {
		ret = ret + ", " + fmt.Sprintf("%s:%v", k, v)
	}
	for k, v := range d.Strings {
		ret = ret + ", " + fmt.Sprintf("%s:%v", k, v)
	}
	for k, v := range d.Bools {
		ret = ret + ", " + fmt.Sprintf("%s:%v", k, v)
	}
	for k, v := range d.Doubles {
		ret = ret + ", " + fmt.Sprintf("%s:%v", k, v)
	}
	return ret
}

// Subtract removes from this databag every value that is in other *and* equal to the current value
// stored in databag.  This is usually used during sync operations: anything left over is data that must be written
func (d *DataBag) Subtract(other *DataBag) *DataBag {
	if d == nil {
		return other
	}
	if other == nil {
		return d
	}
	ret := &DataBag{}
	for k, v := range d.Ints {
		if currentValue, exists := other.Ints[k]; !exists || currentValue != v {
			ret.AddInt(k, v)
		}
	}
	for k, v := range d.Strings {
		if currentValue, exists := other.Strings[k]; !exists || currentValue != v {
			ret.AddString(k, v)
		}
	}
	for k, v := range d.Bools {
		if currentValue, exists := other.Bools[k]; !exists || currentValue != v {
			ret.AddBoolean(k, v)
		}
	}
	for k, v := range d.Doubles {
		if currentValue, exists := other.Doubles[k]; !exists || currentValue != v {
			ret.AddDouble(k, v)
		}
	}
	return ret
}

func (d *DataBag) IsEmpty() bool {
	if d == nil {
		return true
	}
	return len(d.Ints) == 0 &&
		len(d.Strings) == 0 &&
		len(d.Bools) == 0 &&
		len(d.Doubles) == 0
}

// AddInt adds value to DataBag.Ints map.
//
// NOTE: AddInt assumes caller has initialized d to non-nil value
func (d *DataBag) AddInt(key string, value int64) {
	if d.Ints == nil {
		d.Ints = make(map[string]int64)
	}
	d.Ints[key] = value
}

// AddDouble adds value to DataBag.Doubles map.
//
// NOTE: AddDouble assumes caller has initialized d to non-nil value
func (d *DataBag) AddDouble(key string, value float64) {
	if d.Doubles == nil {
		d.Doubles = make(map[string]float64)
	}
	d.Doubles[key] = value
}

// AddString adds value to DataBag.Strings map.
//
// NOTE: AddString assumes caller has initialized d to non-nil value
func (d *DataBag) AddString(key string, value string) {
	if d.Strings == nil {
		d.Strings = make(map[string]string)
	}
	d.Strings[key] = value
}

// AddBoolean adds value to DataBag.Bools map.
//
// NOTE: AddBoolean assumes caller has initialized d to non-nil value
func (d *DataBag) AddBoolean(key string, value bool) {
	if d.Bools == nil {
		d.Bools = make(map[string]bool)
	}
	d.Bools[key] = value
}

// Types returns a copy of the contents of DataBag.
//
// NOTE: Types assumes caller has initialized d to non-nil value
func (d *DataBag) Types() (map[string]int64, map[string]string, map[string]bool, map[string]float64) {
	ints := make(map[string]int64, len(d.Ints))
	for k, v := range d.Ints {
		ints[k] = v
	}

	strs := make(map[string]string, len(d.Strings))
	for k, v := range d.Strings {
		strs[k] = v
	}

	doubles := make(map[string]float64, len(d.Doubles))
	for k, v := range d.Doubles {
		doubles[k] = v
	}

	bools := make(map[string]bool, len(d.Bools))
	for k, v := range d.Bools {
		bools[k] = v
	}
	return ints, strs, bools, doubles
}

type Edge struct {
	From Node   `json:"e1"`
	To   Node   `json:"e2"`
	Type string `json:"type"`
}

func (e Edge) String() string {
	return fmt.Sprintf("%s-%s-%s", e.From, e.Type, e.To)
}

func (e Edge) Reversed(reverseType string) Edge {
	return Edge{
		From: e.To,
		Type: reverseType,
		To:   e.From,
	}
}

// LoadedEdge represents a request for an association.
type LoadedEdge struct {
	Edge
	LoadedData
}

func (a *LoadedEdge) String() string {
	if a == nil {
		return "<nil>"
	}
	return fmt.Sprintf("%s (%s) %s", a.From, a.Type, a.To)
}

func (a LoadedEdge) Reverse(reverseEdgeType string) LoadedEdge {
	return LoadedEdge{
		Edge:       a.Edge.Reversed(reverseEdgeType),
		LoadedData: a.LoadedData,
	}
}

type ListResult struct {
	Cursor string
	To     []CursoredLoadedEdge
}

type PagedRequest struct {
	Cursor          string
	Limit           int64
	DescendingOrder bool
}

// CursoredLoadedEdge attaches a cursor to the LoadedEdge
type CursoredLoadedEdge struct {
	Cursor     string
	LoadedEdge LoadedEdge
}

// LoadedData is returned from a single data fetch
type LoadedData struct {
	CreatedAt time.Time `json:"created_at"`
	UpdatedAt time.Time `json:"updated_at"`
	Data      *DataBag  `json:"data"`
	Version   int64     `json:"version"`
}

type ListNodeResult struct {
	Cursor string
	Nodes  []CursoredLoadedNode
}

type CursoredLoadedNode struct {
	Cursor     string
	LoadedNode LoadedNode
}

type LoadedNode struct {
	Node Node
	Data LoadedData
}
