package conversion

import (
	"time"

	"strconv"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/api/twirpserver"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/proto/datastorerpc"
	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/datastore"
	"github.com/golang/protobuf/ptypes/timestamp"
)

func FromEnt(ent associations.Entity) graphdbmodel.Node {
	return graphdbmodel.Node{
		Type: ent.Kind.KindStr,
		ID:   ent.ID,
	}
}

func ToEntity(ent graphdbmodel.Node) *associations.Entity {
	idFormat, _ := associations.SchemaManager.EntityKind(ent.Type)
	return &associations.Entity{
		ID: ent.ID,
		Kind: associations.EntityKind{
			KindStr:  ent.Type,
			IDFormat: idFormat.IDFormat,
		},
	}
}

func ToProtoAssocationV1(a *associations.Association) *datastorerpc.AssociationV1 {
	if a == nil {
		return nil
	}
	return &datastorerpc.AssociationV1{
		E1:   ToProtoV1Entity(&a.E1),
		E2:   ToProtoV1Entity(&a.E2),
		Kind: FromV1AssocKind(a.Kind),
		D:    twirpserver.ToProtoDataBag(FromDatabag(a.D)),
	}
}

func ToAssocationV1(a *datastorerpc.AssociationV1) *associations.Association {
	if a == nil {
		return nil
	}
	return &associations.Association{
		E1:   ToV1Entity(a.E1),
		E2:   ToV1Entity(a.E2),
		Kind: ToV1AssocKind(a.Kind),
		D:    ToDatabag(twirpserver.FromProtoDataBag(a.D)),
	}
}

func ToAssocationV1Arr(rs []*datastorerpc.AssociationV1) []associations.Association {
	if rs == nil {
		return nil
	}
	ret := make([]associations.Association, 0, len(rs))
	for _, r := range rs {
		item := ToAssocationV1(r)
		if item == nil {
			continue
		}
		ret = append(ret, *item)
	}
	return ret
}

func ToProtoAssocationV1Arr(rs []*associations.Association) []*datastorerpc.AssociationV1 {
	if rs == nil {
		return nil
	}
	ret := make([]*datastorerpc.AssociationV1, 0, len(rs))
	for _, r := range rs {
		item := ToProtoAssocationV1(r)
		if item == nil {
			continue
		}
		ret = append(ret, item)
	}
	return ret
}

func FromDatabag(from associations.DataBag) *graphdbmodel.DataBag {
	data := &graphdbmodel.DataBag{}
	for k, v := range from {
		switch casted := v.(type) {
		case float64:
			data.AddDouble(k, casted)
		case int64:
			data.AddInt(k, casted)
		case int:
			data.AddInt(k, int64(casted))
		case bool:
			data.AddBoolean(k, casted)
		case string:
			data.AddString(k, casted)
		}
	}
	return data
}

func ToDatabag(from *graphdbmodel.DataBag) associations.DataBag {
	data := make(associations.DataBag)
	i, s, b, d := from.Types()
	for k, v := range i {
		data[k] = v
	}
	for k, v := range s {
		data[k] = v
	}
	for k, v := range b {
		data[k] = v
	}
	for k, v := range d {
		data[k] = v
	}
	return data
}

func ToAssocResponse(from *graphdbmodel.LoadedEdge) *associations.AssocResponse {
	if from == nil {
		return nil
	}
	return &associations.AssocResponse{
		E:      ToEntity(from.To),
		T:      from.CreatedAt,
		D:      ToDatabag(from.Data),
		Cursor: strconv.FormatInt(from.CreatedAt.UnixNano(), 10),
	}
}

func ToV1Entity(v1 *datastorerpc.EntityV1) associations.Entity {
	return associations.Entity{
		ID:   v1.Id,
		Kind: ToV1EntityKind(v1.GetKind()),
	}
}

func ToProtoV1Entity(v1 *associations.Entity) *datastorerpc.EntityV1 {
	if v1 == nil {
		return nil
	}
	return &datastorerpc.EntityV1{
		Id:   v1.ID,
		Kind: ToProtoEntityKind(v1.Kind),
	}
}

func ToProtoEntityKind(v1 associations.EntityKind) *datastorerpc.EntityKind {
	return &datastorerpc.EntityKind{
		KindStr:  v1.KindStr,
		IdFormat: v1.IDFormat,
	}
}

func ToV1EntityKind(v1 *datastorerpc.EntityKind) associations.EntityKind {
	return associations.EntityKind{
		KindStr:  v1.GetKindStr(),
		IDFormat: v1.GetIdFormat(),
	}
}

func FromProtoSortBy(s datastorerpc.BulkGetAssocRequest_SortBy) datastore.SortBy {
	if s == datastorerpc.BulkGetAssocRequest_DESC {
		return datastore.Desc
	}
	if s == datastorerpc.BulkGetAssocRequest_ASC {
		return datastore.Asc
	}
	// Hmmm now what do we return?
	return datastore.Asc
}

func ToV1AssocKind(k *datastorerpc.AssocKind) associations.AssocKind {
	return associations.AssocKind{
		KindStr:    k.GetKindStr(),
		InverseStr: k.GetInverseStr(),
		E1KindStr:  k.GetE1KindStr(),
		E2KindStr:  k.GetE2KindStr(),
		ShardFrom:  k.GetShardFrom(),
	}
}

func FromV1AssocKind(k associations.AssocKind) *datastorerpc.AssocKind {
	return &datastorerpc.AssocKind{
		KindStr:    k.KindStr,
		InverseStr: k.InverseStr,
		E1KindStr:  k.E1KindStr,
		E2KindStr:  k.E2KindStr,
		ShardFrom:  k.ShardFrom,
	}
}

func ToProtoAssocResponse(r *associations.AssocResponse) *datastorerpc.AssocResponse {
	if r == nil {
		return nil
	}

	return &datastorerpc.AssocResponse{
		E:      ToProtoV1Entity(r.E),
		T:      ToProtoTime(r.T),
		D:      twirpserver.ToProtoDataBag(FromDatabag(r.D)),
		Cursor: strconv.FormatInt(r.T.UnixNano(), 10),
	}
}

func ToProtoAssocResponseWithMetadata(r *associations.AssocResponseWithMeta) *datastorerpc.AssocResponseWithMeta {
	if r == nil {
		return nil
	}
	return &datastorerpc.AssocResponseWithMeta{
		Kind: r.Kind,
		A:    ToProtoAssocResponse(r.A),
	}
}

func ToProtoAssocResponseWithMetadataArr(rs []*associations.AssocResponseWithMeta) []*datastorerpc.AssocResponseWithMeta {
	if rs == nil {
		return nil
	}
	ret := make([]*datastorerpc.AssocResponseWithMeta, 0, len(rs))
	for _, r := range rs {
		item := ToProtoAssocResponseWithMetadata(r)
		if item == nil {
			continue
		}
		ret = append(ret, item)
	}
	return ret
}

func ToProtoTime(t time.Time) *timestamp.Timestamp {
	n := t.UnixNano()
	return &timestamp.Timestamp{
		Seconds: n / time.Second.Nanoseconds(),
		Nanos:   int32(n % time.Second.Nanoseconds()),
	}
}

func ToProtoAssocResponseArr(responses []*associations.AssocResponse) []*datastorerpc.AssocResponse {
	if responses == nil {
		return nil
	}
	ret := make([]*datastorerpc.AssocResponse, 0, len(responses))
	for _, resp := range responses {
		r := ToProtoAssocResponse(resp)
		if r != nil {
			ret = append(ret, r)
		}
	}
	return ret
}
