package conversion_test

import (
	"reflect"
	"testing"
	"time"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/oldapi/conversion"
	"code.justin.tv/feeds/graphdb/proto/datastorerpc"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/web/cohesion/associations"
	"github.com/stretchr/testify/suite"
)

type ConversionSuite struct {
	suite.Suite
}

func (s *ConversionSuite) TestAssociation() {
	rpcA := &datastorerpc.AssociationV1{
		E1: &datastorerpc.EntityV1{
			Kind: &datastorerpc.EntityKind{},
		},
		E2: &datastorerpc.EntityV1{
			Kind: &datastorerpc.EntityKind{},
		},
		Kind: &datastorerpc.AssocKind{},
		D: &graphdb.DataBag{
			Ints:    map[string]int64{},
			Bools:   map[string]bool{},
			Strings: map[string]string{},
			Doubles: map[string]float64{},
		},
	}
	a := &associations.Association{
		E1:   associations.Entity{},
		E2:   associations.Entity{},
		Kind: associations.AssocKind{},
		D:    associations.DataBag{},
	}

	convertedA := conversion.ToAssocationV1(rpcA)
	convertedRPCA := conversion.ToProtoAssocationV1(a)

	s.Require().True(reflect.DeepEqual(rpcA, convertedRPCA))
	s.Require().True(reflect.DeepEqual(a, convertedA))
}

func (s *ConversionSuite) TestDatabag() {
	aDatabag := associations.DataBag{}
	gdbDatabag := &graphdbmodel.DataBag{
		Strings: map[string]string{},
	}

	aDatabag["hi"] = "string"
	gdbDatabag.Strings["hi"] = "string"

	convertedADatabag := conversion.ToDatabag(gdbDatabag)
	convertedGDBDatabag := conversion.FromDatabag(aDatabag)

	s.Require().Equal(len(aDatabag), len(gdbDatabag.Strings))
	for k := range aDatabag {
		s.Require().Equal(aDatabag[k], gdbDatabag.Strings[k])
	}
	s.Require().Equal(convertedADatabag, aDatabag)
	s.Require().Equal(gdbDatabag, convertedGDBDatabag)
}

func (s *ConversionSuite) TestTime() {
	t := time.Now()

	timestamp := conversion.ToProtoTime(t)

	s.Require().Equal(timestamp.GetSeconds(), t.Unix())
	s.Require().Equal(int(timestamp.GetNanos()), t.Nanosecond())
}

func TestConversionSuite(t *testing.T) {
	suite.Run(t, new(ConversionSuite))
}
