package datastore

import (
	"errors"
	"sync"
	"time"

	"strconv"

	"fmt"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/oldapi/conversion"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage/complexstorage"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage/tablelookup"
	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/datastore"
	"golang.org/x/net/context"
)

// Datastore implements the interface Cohesion needs that allows it to read and write data from GraphDB, but using an
// interface that Cohesion needs.  We use this to mock Cohesion's v1 API
type Datastore struct {
	Storage *complexstorage.Storage
	Lookup  *tablelookup.Lookup

	nameMu sync.Mutex
	name   string
}

var _ datastore.HealthReporterWriter = &Datastore{}
var _ datastore.HealthReporterReader = &Datastore{}
var _ datastore.HealthReporter = &Datastore{}

func (b *Datastore) CreateAssoc(ctx context.Context, ent1 associations.Entity, kind associations.AssocKind, ent2 associations.Entity, data map[string]interface{}, creationTime time.Time) error {
	edgeType := kind.KindStr
	from := conversion.FromEnt(ent1)
	to := conversion.FromEnt(ent2)
	dataCreate := conversion.FromDatabag(data)
	edge := graphdbmodel.Edge{
		From: from,
		To:   to,
		Type: edgeType,
	}
	ret, err := b.Storage.Create(ctx, edge, dataCreate, creationTime)
	if ret == nil {
		// To conform to cohesion's API we have to return an error when the item to create already exists
		return associations.ErrAssociationExists{
			Source: errors.New("already exists"),
		}
	}
	return err
}

// Deletes everything of a kind ...
func (b *Datastore) BulkDeleteAssoc(ctx context.Context, ent1 associations.Entity, kind associations.AssocKind, entKind associations.EntityKind) error {
	from := conversion.FromEnt(ent1)
	edgeType := kind.KindStr
	return b.Storage.BulkDelete(ctx, from, edgeType, entKind.KindStr)
}

func (b *Datastore) DeleteAssoc(ctx context.Context, ent1 associations.Entity, kind associations.AssocKind, ent2 associations.Entity) error {
	from := conversion.FromEnt(ent1)
	edgeType := kind.KindStr
	to := conversion.FromEnt(ent2)
	edge := graphdbmodel.Edge{
		From: from,
		Type: edgeType,
		To:   to,
	}
	resp, err := b.Storage.Delete(ctx, edge, nil)
	if resp == nil {
		// To match the postgres API we have to also return nil (even though not found is probably the correct impl)
		return nil
	}
	return err
}

func (b *Datastore) UpdateAssoc(ctx context.Context, ent1 associations.Entity, kind associations.AssocKind, ent2 associations.Entity, data map[string]interface{}, newType associations.AssocKind) error {
	edgeType := kind.KindStr
	from := conversion.FromEnt(ent1)
	to := conversion.FromEnt(ent2)
	dataCreate := conversion.FromDatabag(data)
	edge := graphdbmodel.Edge{
		From: from,
		Type: edgeType,
		To:   to,
	}
	exists, err := b.Storage.UpdateDataAndType(ctx, edge, dataCreate, newType.KindStr)
	if err != nil {
		return err
	}
	if exists {
		return nil
	}
	return associations.ErrNotFound{
		Assoc: &associations.Association{
			E1:   ent1,
			E2:   ent2,
			Kind: kind,
		},
	}
}

// This API is dumb.  I quote "// We assume that E1 for each of the associations is the same"
//  This implementation doesn't.
func (b *Datastore) BatchUpdateAssoc(ctx context.Context, assocs []associations.Association) (int64, error) {
	for _, assoc := range assocs {
		if err := b.UpdateAssoc(ctx, assoc.E1, assoc.Kind, assoc.E2, assoc.D, assoc.Kind); err != nil {
			return 0, err
		}
	}
	return 1, nil
}

func (b *Datastore) BulkUpdateAssoc(ctx context.Context, ent associations.Entity, assocKind associations.AssocKind, entKind associations.EntityKind, data map[string]interface{}, newKind associations.AssocKind) error {
	from := conversion.FromEnt(ent)
	edgeType := assocKind.KindStr
	return b.Storage.BulkUpdateAssoc(ctx, from, edgeType, entKind.KindStr, conversion.FromDatabag(data), newKind.KindStr)
}

func (b *Datastore) GetAssoc(ctx context.Context, ent1 associations.Entity, kind associations.AssocKind, ent2 associations.Entity) ([]*associations.AssocResponse, error) {
	edgeType := kind.KindStr
	from := conversion.FromEnt(ent1)
	to := conversion.FromEnt(ent2)
	edge := graphdbmodel.Edge{
		From: from,
		Type: edgeType,
		To:   to,
	}
	ret, err := b.Storage.Get(ctx, edge)
	if err != nil {
		return nil, err
	}
	if ret == nil {
		// Cohesion treats not found as an error, so we have to return an error in this case
		return nil, associations.ErrNotFound{Assoc: &associations.Association{
			E1:   ent1,
			E2:   ent2,
			Kind: kind,
		}}
	}
	return []*associations.AssocResponse{
		conversion.ToAssocResponse(ret),
	}, nil
}

func (b *Datastore) BulkGetAssoc(ctx context.Context, ent1 associations.Entity, kind associations.AssocKind, entKind associations.EntityKind, sort datastore.SortBy, offset int, limit int, cursor string) ([]*associations.AssocResponse, error) {
	from := conversion.FromEnt(ent1)
	edgeType := kind.KindStr
	page := graphdbmodel.PagedRequest{
		Cursor:          cursor,
		Limit:           int64(limit) + int64(offset),
		DescendingOrder: sort == datastore.Desc,
	}
	res, err := b.Storage.List(ctx, from, edgeType, page)
	if err != nil {
		return nil, err
	}
	ret := make([]*associations.AssocResponse, 0, len(res.To))
	offsetCounter := 0
	for _, r := range res.To {
		if offsetCounter < offset {
			offsetCounter++
			continue
		}
		ret = append(ret, &associations.AssocResponse{
			E: conversion.ToEntity(r.LoadedEdge.To),
			T: r.LoadedEdge.CreatedAt,
			D: conversion.ToDatabag(r.LoadedEdge.Data),
			// cohesion uses a timestamp cursor
			Cursor: strconv.FormatInt(r.LoadedEdge.CreatedAt.UnixNano(), 10),
			// We cannot use our cursor style since cohesion expects a timestamp cursor
			//Cursor: r.Cursor,
		})
	}
	if res.Cursor != "" && offset > 0 && len(ret) == 0 {
		return nil, fmt.Errorf("Offset %d is too large for this query.  Please use cursors to page results instead", offset)
	}
	return ret, nil
}

func (b *Datastore) GetAllAssoc(ctx context.Context, ent1 associations.Entity, ent2 associations.Entity) ([]*associations.AssocResponseWithMeta, error) {
	from := conversion.FromEnt(ent1)
	to := conversion.FromEnt(ent2)
	ret := make([]*associations.AssocResponseWithMeta, 0, 12)
	for _, edgeKind := range b.Lookup.AllEdgeTypes() {
		edge := graphdbmodel.Edge{
			From: from,
			Type: edgeKind,
			To:   to,
		}
		assoc, err := b.Storage.Get(ctx, edge)
		if err != nil {
			return nil, err
		}
		if assoc == nil {
			continue
		}
		ret = append(ret, &associations.AssocResponseWithMeta{
			Kind: edgeKind,
			A: &associations.AssocResponse{
				E: conversion.ToEntity(assoc.To),
				T: assoc.CreatedAt,
				D: conversion.ToDatabag(assoc.Data),
				// We cannot use our cursor style since cohesion expects a timestamp cursor
				Cursor: strconv.FormatInt(assoc.CreatedAt.UnixNano(), 10),
				//Cursor: "",
			},
		})
	}
	return ret, nil
}

func (b *Datastore) GetAllAssocBatch(ctx context.Context, assocs []associations.Association) ([]*associations.AssocResponseWithMeta, error) {
	ret := make([]*associations.AssocResponseWithMeta, 0, len(assocs))
	for _, a := range assocs {
		toAdd, err := b.GetAllAssoc(ctx, a.E1, a.E2)
		if err != nil {
			return nil, err
		}
		ret = append(ret, toAdd...)
	}
	return ret, nil
}

// Count assocs of a kind. Note that we don't suppor the `endKind` parameter.  We will return the total count, not the count
// by a specific entity kind
func (b *Datastore) CountAssoc(ctx context.Context, ent associations.Entity, kind associations.AssocKind, entKind associations.EntityKind) (int, error) {
	// TODO: Check entKind (??)
	edgeKind := kind.KindStr
	from := conversion.FromEnt(ent)
	ret, err := b.Storage.Count(ctx, from, edgeKind)
	if err != nil {
		return 0, err
	}
	return int(ret), nil
}

func (b *Datastore) GetHitCounts(ctx context.Context, client string, minutes int) (map[string]int, error) {
	return nil, errors.New("unimplemented")
}

func (b *Datastore) Health(context.Context) error {
	return nil
}
func (b *Datastore) Utilization(context.Context) float32 {
	return 0.0
}

func (b *Datastore) SetName(s string) {
	b.nameMu.Lock()
	defer b.nameMu.Unlock()
	b.name = s
}

func (b *Datastore) Name() string {
	b.nameMu.Lock()
	defer b.nameMu.Unlock()
	return b.name
}
