package twirpdatastore

import (
	"context"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/accesslog"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/api/twirpserver"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/oldapi/conversion"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/oldapi/datastore"
	"code.justin.tv/feeds/graphdb/proto/datastorerpc"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/web/cohesion/associations"
	"github.com/cactus/go-statsd-client/statsd"
)

// Server stubs cohesion's reader/writer API to allow us to mirror cohesion writes into GraphDB and expose a cohesion
// like client library.  This layer only translates between protobuf to a cohesion writer
type Server struct {
	// Datastore is our cohesion like writer
	Store   *datastore.Datastore
	Statter statsd.SubStatter
	Log     log.Logger
}

// GetAssoc returns an AssociationV1 populated with meta data about
// the association. The direction of the provided AssociationV1 is
// returned returned in the response association. The meta data also
// reflects this, that is to say:
func (s *Server) GetAssoc(ctx context.Context, req *datastorerpc.GetAssocRequest) (*datastorerpc.GetAssocResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	a, err := s.Store.GetAssoc(ctx, conversion.ToV1Entity(req.GetFrom()), conversion.ToV1AssocKind(req.GetAssocKind()), conversion.ToV1Entity(req.GetTo()))
	if err != nil {
		if e2, ok := err.(associations.ErrNotFound); ok {
			return &datastorerpc.GetAssocResponse{
				NotFound: &datastorerpc.ErrNotFound{
					Assoc: conversion.ToProtoAssocationV1(e2.Assoc),
				},
			}, nil
		}
	}
	return &datastorerpc.GetAssocResponse{
		Assocs: conversion.ToProtoAssocResponseArr(a),
	}, err
}

// BulkGetAssoc returns a slice of `associations.AssocResponse` for the
// given entity. As this is a PostgreSQL data store offset, limit and
// sort logic is passed in to facilitate pagination.
func (s *Server) BulkGetAssoc(ctx context.Context, req *datastorerpc.BulkGetAssocRequest) (*datastorerpc.BulkGetAssocResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	a, err := s.Store.BulkGetAssoc(ctx, conversion.ToV1Entity(req.GetFrom()), conversion.ToV1AssocKind(req.GetAssocKind()), conversion.ToV1EntityKind(req.GetToKind()), conversion.FromProtoSortBy(req.GetSortBy()), int(req.GetOffset()), int(req.GetLimit()), req.GetCursor())
	return &datastorerpc.BulkGetAssocResponse{
		Assocs: conversion.ToProtoAssocResponseArr(a),
	}, err
}

// CountAssoc returns the number of edges to this node for the
// specified type. If err is set, disregard the return value provided
func (s *Server) CountAssoc(ctx context.Context, req *datastorerpc.CountAssocRequest) (*datastorerpc.CountAssocResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	i, err := s.Store.CountAssoc(ctx, conversion.ToV1Entity(req.GetFrom()), conversion.ToV1AssocKind(req.GetAssocKind()), conversion.ToV1EntityKind(req.GetToKind()))
	return &datastorerpc.CountAssocResponse{
		Count: int64(i),
	}, err
}

// GetAllAssoc returns the relevant edges between 2 entities.
func (s *Server) GetAllAssoc(ctx context.Context, req *datastorerpc.GetAllAssocRequest) (*datastorerpc.GetAllAssocResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	i, err := s.Store.GetAllAssoc(ctx, conversion.ToV1Entity(req.GetFrom()), conversion.ToV1Entity(req.GetTo()))
	return &datastorerpc.GetAllAssocResponse{
		Assocs: conversion.ToProtoAssocResponseWithMetadataArr(i),
	}, err
}

// GetAllAssocBatch returns the relevant edges from one EntityV1 to any of a list of other entities.
func (s *Server) GetAllAssocBatch(ctx context.Context, req *datastorerpc.GetAllAssocBatchRequest) (*datastorerpc.GetAllAssocBatchResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	i, err := s.Store.GetAllAssocBatch(ctx, conversion.ToAssocationV1Arr(req.Assocs))
	return &datastorerpc.GetAllAssocBatchResponse{
		Assocs: conversion.ToProtoAssocResponseWithMetadataArr(i),
	}, err
}

func (s *Server) CreateAssoc(ctx context.Context, req *datastorerpc.CreateAssocRequest) (*datastorerpc.CreateAssocResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	from := conversion.ToV1Entity(req.From)
	assocKind := conversion.ToV1AssocKind(req.AssocKind)
	to := conversion.ToV1Entity(req.To)
	databag := conversion.ToDatabag(twirpserver.FromProtoDataBag(req.Data))
	creationTime := twirpserver.FromProtoTime(req.CreationDate)

	if err := s.Store.CreateAssoc(ctx, from, assocKind, to, databag, creationTime); err != nil {
		// Cohesion uses errors for 404, so we have to translate those into not found errors instead of protocol errors
		// In other words, we cannot just return these errors back.
		if e2, ok := err.(associations.ErrAssociationExists); ok {
			return &datastorerpc.CreateAssocResponse{
				Exists: &datastorerpc.ErrAssociationExists{
					Source: &graphdb.Error{
						Message: e2.Source.Error(),
					},
				},
			}, nil
		}
		return &datastorerpc.CreateAssocResponse{}, err
	}

	return &datastorerpc.CreateAssocResponse{}, nil
}

// DeleteAssoc sets the hidden column as true for the relationship
// expressed by the supplied args
func (s *Server) DeleteAssoc(ctx context.Context, req *datastorerpc.DeleteAssocRequest) (*datastorerpc.DeleteAssocResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	from := conversion.ToV1Entity(req.From)
	assocKind := conversion.ToV1AssocKind(req.AssocKind)
	to := conversion.ToV1Entity(req.To)

	if err := s.Store.DeleteAssoc(ctx, from, assocKind, to); err != nil {
		// Cannot just return errors up the stack.  Check for ErrNotFound and return a real not-found response instead
		if e2, ok := err.(associations.ErrNotFound); ok {
			return &datastorerpc.DeleteAssocResponse{
				NotFound: &datastorerpc.ErrNotFound{
					Assoc: conversion.ToProtoAssocationV1(e2.Assoc),
				},
			}, nil
		}
		return &datastorerpc.DeleteAssocResponse{}, err
	}

	return &datastorerpc.DeleteAssocResponse{}, nil
}

// BulkDeleteAssoc deletes all of the associations of type ASSOCKIND for the
// given EntityV1 ENTITY.
func (s *Server) BulkDeleteAssoc(ctx context.Context, req *datastorerpc.BulkDeleteAssocRequest) (*datastorerpc.BulkDeleteAssocResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	from := conversion.ToV1Entity(req.From)
	assocKind := conversion.ToV1AssocKind(req.AssocKind)
	toKind := conversion.ToV1EntityKind(req.ToKind)
	if err := s.Store.BulkDeleteAssoc(ctx, from, assocKind, toKind); err != nil {
		return &datastorerpc.BulkDeleteAssocResponse{}, err
	}

	return &datastorerpc.BulkDeleteAssocResponse{}, nil
}

// UpdateAssoc updates the databag attached to this association. It
// merges the existing databag with the data in the provided `data`
// argument giving priority to the latter.
func (s *Server) UpdateAssoc(ctx context.Context, req *datastorerpc.UpdateAssocRequest) (*datastorerpc.UpdateAssocResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)

	from := conversion.ToV1Entity(req.From)
	assocKind := conversion.ToV1AssocKind(req.AssocKind)
	to := conversion.ToV1Entity(req.To)
	databag := conversion.ToDatabag(twirpserver.FromProtoDataBag(req.Data))
	newKind := conversion.ToV1AssocKind(req.NewKind)

	if err := s.Store.UpdateAssoc(ctx, from, assocKind, to, databag, newKind); err != nil {
		// Cannot just return these errors up the stack.  They aren't actually errors.  It just means the assoc didn't
		// exist
		if e2, ok := err.(associations.ErrNotFound); ok {
			return &datastorerpc.UpdateAssocResponse{
				NotFound: &datastorerpc.ErrNotFound{
					Assoc: conversion.ToProtoAssocationV1(e2.Assoc),
				},
			}, nil
		}
		return &datastorerpc.UpdateAssocResponse{}, err
	}

	return &datastorerpc.UpdateAssocResponse{}, nil
}

// BatchUpdateAssoc takes in a slice of associations and updates each association
// according the the data bag in each association
func (s *Server) BatchUpdateAssoc(ctx context.Context, req *datastorerpc.BatchUpdateAssocRequest) (*datastorerpc.BatchUpdateAssocResponse, error) {
	ctx = accesslog.WithRequestBody(ctx, req)
	assocs := conversion.ToAssocationV1Arr(req.Assocs)
	i, err := s.Store.BatchUpdateAssoc(ctx, assocs)

	return &datastorerpc.BatchUpdateAssocResponse{
		Unused: i,
	}, err
}

// BulkUpdateAssoc updates the databag for all associations of type `assocKind`.
// It merges the existing databags with the data in the provided `data` argument.
func (s *Server) BulkUpdateAssoc(ctx context.Context, req *datastorerpc.BulkUpdateAssocRequest) (*datastorerpc.BulkUpdateAssocResponse, error) {
	// This context is ignored
	_ = accesslog.WithRequestBody(ctx, req)
	ent := conversion.ToV1Entity(req.Entity)
	assocKind := conversion.ToV1AssocKind(req.AssocKind)
	toKind := conversion.ToV1EntityKind(req.ToKind)
	databag := conversion.ToDatabag(twirpserver.FromProtoDataBag(req.Data))
	newKind := conversion.ToV1AssocKind(req.NewKind)

	// These requests take so long to complete, that we have to do them in the background in a goroutine.
	go func() {
		start := time.Now()
		// TODO: Move this to SQS eventually
		reqCtx, cancel := context.WithTimeout(context.Background(), 4*time.Hour)
		defer cancel()
		err := s.Store.BulkUpdateAssoc(reqCtx, ent, assocKind, toKind, databag, newKind)
		if err != nil {
			s.Statter.TimingDuration("bulk_update_assoc.timing.err", time.Since(start), 1.0)
			s.Statter.Inc("bulk_update_assoc.count.err", 1.0, 1.0)
			s.Log.Log("err", errors.Wrap(err, "unable to background bulk update"), "entity", ent.String(), "from", req.AssocKind.String(), "to", req.ToKind.String())
			return
		}

		s.Statter.TimingDuration("bulk_update_assoc.timing.success", time.Since(start), 1.0)
		s.Statter.Inc("bulk_update_assoc.count.success", 1.0, 1.0)
	}()

	return &datastorerpc.BulkUpdateAssocResponse{}, nil
}

var _ datastorerpc.Reader = &Server{}
var _ datastorerpc.Writer = &Server{}
