package resync

import (
	"context"
	"database/sql"
	"encoding/json"
	"errors"
	"strconv"
	"time"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/oldapi/conversion"
	"code.justin.tv/feeds/log"
	"code.justin.tv/web/cohesion/associations"
	_ "github.com/lib/pq"
)

// ListPQLFactory is only used for resync.  It connects directly to a cohesion postgres database to read the edges.
type ListPQLFactory struct {
	Connection string
	DriverName string
	TableName  string
	Log        log.Logger
}

func (f *ListPQLFactory) Setup() error {
	var err error
	if f.Connection == "" {
		return nil
	}
	db, err := sql.Open(f.driverName(), f.Connection)
	if err != nil {
		return err
	}
	pingCtx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	return db.PingContext(pingCtx)
}

func (f *ListPQLFactory) driverName() string {
	if f.DriverName == "" {
		return "postgres"
	}
	return f.DriverName
}

// NewListPQL creates a new ListPQL that can read rows.  Each returned ListPQL is not thread safe, but it is ok to
// use multiple ListPQL at the same time.
func (f *ListPQLFactory) NewListPQL() (*ListPQL, error) {
	if f.Connection == "" {
		return nil, errors.New("unable to make pql connection: no connection string set")
	}
	db, err := sql.Open(f.driverName(), f.Connection)
	if err != nil {
		return nil, err
	}
	return &ListPQL{
		TableName: f.TableName,
		db:        db,
	}, nil
}

// ListPQL iterates graphdb style edges from a cohesion pql database
type ListPQL struct {
	TableName string
	nextErr   error
	db        *sql.DB
}

func (f *ListPQL) tableName() string {
	if f.TableName == "" {
		return "associations"
	}
	return f.TableName
}

func (f *ListPQL) Close() error {
	if f.db != nil {
		return f.db.Close()
	}
	return nil
}

// processRow turns a sql row into a graphdb row.  If you change the format of the PrepareContext inside List, you must
// change how the `Scan` here works.
func processRow(rows *sql.Rows) (graphdbmodel.CursoredLoadedEdge, error) {
	var from_id, to_id int64
	var from_kind, to_kind, assoc_kind, data_bag string
	var creation_date time.Time
	if err := rows.Scan(&from_id, &from_kind, &assoc_kind, &to_id, &to_kind, &data_bag, &creation_date); err != nil {
		return graphdbmodel.CursoredLoadedEdge{}, err
	}
	var decoded_data map[string]interface{}
	if err := json.Unmarshal([]byte(data_bag), &decoded_data); err != nil {
		return graphdbmodel.CursoredLoadedEdge{}, err
	}
	return graphdbmodel.CursoredLoadedEdge{
		Cursor: strconv.FormatInt(creation_date.UnixNano(), 10),
		LoadedEdge: graphdbmodel.LoadedEdge{
			Edge: graphdbmodel.Edge{
				From: graphdbmodel.Node{
					Type: from_kind,
					ID:   strconv.FormatInt(from_id, 10),
				},
				Type: assoc_kind,
				To: graphdbmodel.Node{
					Type: to_kind,
					ID:   strconv.FormatInt(to_id, 10),
				},
			},
			LoadedData: graphdbmodel.LoadedData{
				Data:      conversion.FromDatabag(associations.DataBag(decoded_data)),
				CreatedAt: creation_date,
				UpdatedAt: creation_date,
				Version:   0,
			},
		},
	}, nil
}

// List edges from Cohesion in a GraphDB manner.  This purposely looks like the List API inside the Storage interface.  Obeys the same
// API constraints.
func (f *ListPQL) List(ctx context.Context, from graphdbmodel.Node, edgeKind string, page graphdbmodel.PagedRequest) (*graphdbmodel.ListResult, error) {
	if f.nextErr != nil {
		return nil, f.nextErr
	}
	if page.DescendingOrder {
		return nil, errors.New("can only page in ascending order")
	}
	if page.Cursor == "" {
		page.Cursor = "0"
	}
	cursorAsInt, err := strconv.ParseInt(page.Cursor, 10, 64)
	if err != nil {
		return nil, err
	}
	cursorTime := time.Unix(0, cursorAsInt)
	stmt, err := f.db.PrepareContext(ctx, "SELECT from_id, from_kind, assoc_kind, to_id, to_kind, data_bag, creation_date from "+f.tableName()+" WHERE from_id = $1 AND from_kind = $2 AND assoc_kind = $3 AND creation_date > $4 ORDER BY creation_date LIMIT $5")
	if err != nil {
		return nil, err
	}
	rows, err := stmt.QueryContext(ctx, from.ID, from.Type, edgeKind, cursorTime, page.Limit)
	if err != nil {
		return nil, err
	}
	defer func() {
		// Have to remember to close your PQL connection!
		err := rows.Close()
		if err != nil {
			f.nextErr = err
		}
	}()
	ret := &graphdbmodel.ListResult{
		To: make([]graphdbmodel.CursoredLoadedEdge, 0, page.Limit+1),
	}
	//
	for rows.Next() {
		assoc, err := processRow(rows)
		if err != nil {
			return nil, err
		}
		ret.Cursor = assoc.Cursor
		ret.To = append(ret.To, assoc)
	}
	// With SQL, we should always fetch Limit items.  If we don't, then we know we are at the end of the list query
	if len(ret.To) < int(page.Limit) {
		ret.Cursor = ""
	}
	return ret, nil
}
