package complexstorage

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage/tablelookup"
	"code.justin.tv/feeds/log"
	"code.justin.tv/hygienic/statsdsender"
	"golang.org/x/sync/errgroup"
	"golang.org/x/sync/semaphore"
)

// Storage allows more complex operations on top of the basic storage client
type Storage struct {
	storage.Client
	Lookup    *tablelookup.Lookup
	Log       log.Logger
	Semaphore *semaphore.Weighted
	Stats     *statsdsender.ErrorlessStatSender
}

type EdgeGet struct {
	From graphdbmodel.Node
	To   graphdbmodel.Node
}

// Update just forwards to the more core UpdateOrPut function.
func (s *Storage) Update(ctx context.Context, edge graphdbmodel.Edge, data *graphdbmodel.DataBag, createdAt *time.Time, requiredVersion *time.Time) (*graphdbmodel.LoadedEdge, error) {
	return s.Client.UpdateOrPut(ctx, edge, data, true, createdAt, requiredVersion)
}

// Put just forwards to the more core UpdateOrPut function.
func (s *Storage) Put(ctx context.Context, edge graphdbmodel.Edge, data *graphdbmodel.DataBag, createdAt *time.Time, requiredVersion *time.Time) (*graphdbmodel.LoadedEdge, error) {
	return s.Client.UpdateOrPut(ctx, edge, data, false, createdAt, requiredVersion)
}

// ChangeType changes the type of an edge.  We don't support this atomically since they could potentially live in different
// DynamoDB tables (however, it's worth optimizing this for when they are on the same table).  A type change is commonly
// a creation followed by a deletion.  This is not atomic and there is currently no smart recovery for when a creation
// works, but the deletion fails.
func (s *Storage) ChangeType(ctx context.Context, edge graphdbmodel.Edge, newType string) (*graphdbmodel.LoadedEdge, error) {
	if newType == edge.Type {
		return s.Client.Get(ctx, edge)
	}
	infoOriginalType := s.Lookup.LookupEdge(edge.Type)
	if infoOriginalType == nil {
		return nil, errors.Errorf("unable to find information for edge type %s", edge.Type)
	}
	infoNewType := s.Lookup.LookupEdge(newType)
	if infoNewType == nil {
		return nil, errors.Errorf("unable to find information for edge type %s", newType)
	}
	originalEdge, err := s.Client.Get(ctx, edge)
	if err != nil {
		return nil, err
	}
	if originalEdge == nil {
		return nil, nil
	}
	newEdge := graphdbmodel.Edge{
		From: edge.From,
		To:   edge.To,
		Type: newType,
	}
	createdEdge, err := s.Client.Create(ctx, newEdge, originalEdge.Data, originalEdge.CreatedAt)
	if err != nil {
		return nil, err
	}
	// Bit of a potential data create that is lost here (no recovery from createdEdge).  We should add some smart recovery here
	_, err = s.Client.Delete(ctx, edge, nil)
	if err != nil {
		return nil, err
	}
	return createdEdge, nil
}

const bulkAtOnce = 10
const bulkUpdateRateLimit = time.Second / 20
const bulkAtOnceV2 = 20
const bulkUpdateRateLimitV2 = time.Second / 50

// bulkAddToChannel is a helper used by BulkFunction to add results to a channel in bulk, throttling how many we add
// per second.  This lets us do bulk operations, but not so many at once that we hit our DynamoDB limits
func bulkAddToChannel(ctx context.Context, addInto chan graphdbmodel.LoadedEdge, listRes *graphdbmodel.ListResult, throttle time.Duration) error {
	for _, assoc := range listRes.To {
		select {
		case <-ctx.Done():
			return ctx.Err()
		// Rate limit how often we add items
		case <-time.After(throttle):
			select {
			case addInto <- assoc.LoadedEdge:
			case <-ctx.Done():
				return ctx.Err()
			}
		}
	}
	return nil
}

// bulkFunction is a helper for public bulk operations on data.  It lists all edges of a type from a node and calls mutator on each edge
func (s *Storage) bulkFunction(ctx context.Context, from graphdbmodel.Node, edgeType string, mutator func(ctx context.Context, association graphdbmodel.LoadedEdge) error) error {
	// ---- This block limits how many bulk update operations we do at once.
	// It is different than the limitation that limits how *quickly* we do any single bulk update operation
	if err := s.Semaphore.Acquire(ctx, 1); err != nil {
		return err
	}
	defer s.Semaphore.Release(1)
	// ----

	var page graphdbmodel.PagedRequest
	for {
		// List nodes forever until we either have no more nodes to list, or get an error
		listRes, err := s.Client.List(ctx, from, edgeType, page)
		if err != nil {
			return err
		}
		s.Stats.IncC("bulkFunction_list", int64(len(listRes.To)), 1)
		// use goroutines to operate on at most bulkAtOnce nodes.
		toDo := make(chan graphdbmodel.LoadedEdge, bulkAtOnce)
		eg, egCtx := errgroup.WithContext(ctx)
		for i := 0; i < bulkAtOnce; i++ {
			eg.Go(func() error {
				// Ends when to Do is closed or ctx dies
				for {
					select {
					case <-egCtx.Done():
						return egCtx.Err()
					case assoc, ok := <-toDo:
						if !ok {
							return nil
						}
						// Call the mutator on each node, one node at a time
						if err := mutator(egCtx, assoc); err != nil {
							s.Log.Log("err", err)
							// We don't want to end early if a single one of the bulk fails
							if egCtx.Err() != nil {
								return err
							}
						}
					}
				}
			})
		}
		eg.Go(func() error {
			// Note: close is what stops the forever loops above
			defer close(toDo)
			// Slowly add nodes to the channel to do
			return bulkAddToChannel(egCtx, toDo, listRes, bulkUpdateRateLimit)
		})
		if err := eg.Wait(); err != nil {
			return err
		}
		if listRes.Cursor == "" {
			// This means there are no more nodes to operate on
			return nil
		}
		page.Cursor = listRes.Cursor
	}
}

// bulkFunctionV2 is a helper for public bulk operations on data.
// It lists edges of a type from a node based up to the limit and calls mutator on each edge,
// returns how many edges were mutated.
func (s *Storage) bulkFunctionV2(ctx context.Context, from graphdbmodel.Node, edgeType string, limit int,
	mutator func(ctx context.Context, association graphdbmodel.LoadedEdge) error) (count int, more bool, err error) {
	page := graphdbmodel.PagedRequest{
		Cursor:          "",
		Limit:           int64(limit),
		DescendingOrder: false,
	}
	listRes, err := s.Client.List(ctx, from, edgeType, page)
	if err != nil {
		return 0, false, err
	}

	// use goroutines to operate on at most bulkAtOnce nodes.
	jobs := make(chan graphdbmodel.LoadedEdge, bulkAtOnceV2)
	eg, egCtx := errgroup.WithContext(ctx)
	for i := 0; i < bulkAtOnceV2; i++ {
		eg.Go(func() error {
			// Ends when jobs is closed or ctx dies
			for {
				select {
				case <-egCtx.Done():
					return egCtx.Err()
				case assoc, more := <-jobs:
					if !more {
						return nil
					}
					// Call the mutator on each node, one node at a time
					if err := mutator(egCtx, assoc); err != nil {
						s.Log.Log("err", err)
						// We don't want to end early if a single one of the bulk fails
						if egCtx.Err() != nil {
							return err
						}
					}
				}
			}
		})
	}
	eg.Go(func() error {
		// Note: close is what stops the forever loops above
		defer close(jobs)
		// Slowly add nodes to the channel to do
		return bulkAddToChannel(egCtx, jobs, listRes, bulkUpdateRateLimitV2)
	})
	if err := eg.Wait(); err != nil {
		return 0, false, err
	}

	return len(listRes.To), listRes.Cursor != "", nil
}

// UpdateDataAndType does exactly what the name says (update the data and type of an edge).  This is the API
// that Cohesion expects, and isn't used outside that context.
func (s *Storage) UpdateDataAndType(ctx context.Context, edge graphdbmodel.Edge, data *graphdbmodel.DataBag, newType string) (bool, error) {
	if !data.IsEmpty() {
		if newVersion, err := s.Update(ctx, edge, data, nil, nil); err != nil {
			return false, err
		} else if newVersion == nil {
			return false, nil
		}
	}
	if edge.Type != newType && newType != "" && newType != "unknown" {
		if _, err := s.ChangeType(ctx, edge, newType); err != nil {
			return false, err
		}
	}
	return true, nil
}

// BulkUpdateAssoc is cohesion's update function.  We don't normally call it from GraphDB's API, but it updates both
// the data (if not nil) and edge type (if not nil or unknown) from a node of an edge type
func (s *Storage) BulkUpdateAssoc(ctx context.Context, from graphdbmodel.Node, edgeType string, entKind string, data *graphdbmodel.DataBag, newType string) error {
	if data.IsEmpty() && (edgeType == newType || newType == "") {
		return nil
	}
	return s.bulkFunction(ctx, from, edgeType, func(ctx context.Context, assoc graphdbmodel.LoadedEdge) error {
		if assoc.To.Type != entKind {
			return nil
		}
		edge := graphdbmodel.Edge{
			From: assoc.From,
			Type: assoc.Type,
			To:   assoc.To,
		}
		_, err := s.UpdateDataAndType(ctx, edge, data, newType)
		return err
	})
}

// BulkDelete is kind of dangerous, but exists inside cohesion so we have to write it too.  It removes all edges
// of a type from a node.
func (s *Storage) BulkDelete(ctx context.Context, from graphdbmodel.Node, edgeType string, entKind string) error {
	return s.bulkFunction(ctx, from, edgeType, func(ctx context.Context, assoc graphdbmodel.LoadedEdge) error {
		if assoc.To.Type != entKind {
			return nil
		}
		edge := graphdbmodel.Edge{
			From: assoc.From,
			Type: edgeType,
			To:   assoc.To,
		}
		_, err := s.Client.Delete(ctx, edge, nil)
		return err
	})
}

// BulkUpdateV2 updates the edge type (if not nil or unknown) from a node of an edge type,
// returns whether there are more edges to update and error
func (s *Storage) BulkUpdateV2(ctx context.Context, from graphdbmodel.Node, edgeType string, newType string, limit int) (bool, error) {
	if edgeType == newType || newType == "" {
		return false, nil
	}

	start := time.Now()
	count, more, err := s.bulkFunctionV2(ctx, from, edgeType, limit, func(ctx context.Context, assoc graphdbmodel.LoadedEdge) error {
		edge := graphdbmodel.Edge{
			From: assoc.From,
			Type: assoc.Type,
			To:   assoc.To,
		}
		_, err := s.UpdateDataAndType(ctx, edge, nil, newType)
		return err
	})

	if err != nil {
		d := time.Now().Sub(start).Nanoseconds() / int64(time.Millisecond)
		s.Log.Log(fmt.Sprintf("processed %d edges in BulkUpdateV2 in %d milliseconds", count, d))
	}
	return more, err
}

// BulkDeleteV2 delete edges of a type from a node and returns whether there are more edges to delete and error
func (s *Storage) BulkDeleteV2(ctx context.Context, from graphdbmodel.Node, edgeType string, limit int) (bool, error) {
	start := time.Now()
	count, more, err := s.bulkFunctionV2(ctx, from, edgeType, limit, func(ctx context.Context, assoc graphdbmodel.LoadedEdge) error {
		edge := graphdbmodel.Edge{
			From: assoc.From,
			Type: edgeType,
			To:   assoc.To,
		}
		_, err := s.Client.Delete(ctx, edge, nil)
		return err
	})

	if err != nil {
		d := time.Now().Sub(start).Nanoseconds() / int64(time.Millisecond)
		s.Log.Log(fmt.Sprintf("processed %d edges in BulkUpdateV2 in %d milliseconds", count, d))
	}
	return more, err
}
