package storage

import (
	"strconv"
	"time"

	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/service-common/feedsdynamo"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

func DatabagFromDynamoDB(in *dynamodb.AttributeValue) *graphdbmodel.DataBag {
	if in == nil || in.M == nil {
		return &graphdbmodel.DataBag{}
	}
	dataValue := in.M
	ret := &graphdbmodel.DataBag{}
	for k, v := range dataValue {
		if v.S != nil {
			ret.AddString(k, *v.S)
			continue
		}
		if v.BOOL != nil {
			ret.AddBoolean(k, *v.BOOL)
			continue
		}
		if v.N != nil {
			if n, err := strconv.ParseInt(*v.N, 10, 64); err == nil {
				ret.AddInt(k, n)
				continue
			}
		}
	}
	return ret
}

// DynamodbItems returns what the databag looks like as a single item on the table
func DynamodbItems(db *graphdbmodel.DataBag) map[string]*dynamodb.AttributeValue {
	ret := make(map[string]*dynamodb.AttributeValue)
	ints, strs, bools, floats := db.Types()
	for k, v := range ints {
		ret[k] = &dynamodb.AttributeValue{
			N: aws.String(strconv.FormatInt(v, 10)),
		}
	}
	for k, v := range floats {
		ret[k] = &dynamodb.AttributeValue{
			N: aws.String(strconv.FormatFloat(v, 'f', -1, 64)),
		}
	}
	for k, v := range strs {
		ret[k] = &dynamodb.AttributeValue{
			S: aws.String(v),
		}
	}
	for k, v := range bools {
		ret[k] = &dynamodb.AttributeValue{
			BOOL: aws.Bool(v),
		}
	}
	return ret
}

type DefaultSortKeyCreation interface {
	DefaultSortKeyValue(sortKeyName string) *dynamodb.AttributeValue
}

func DynamoDBAssociation(item map[string]*dynamodb.AttributeValue) (graphdbmodel.LoadedEdge, error) {
	if item == nil {
		return graphdbmodel.LoadedEdge{}, nil
	}
	strs, err := feedsdynamo.AwsStrings(item, []string{dbFrom, dbTo, dbEdgeType})
	if err != nil {
		return graphdbmodel.LoadedEdge{}, err
	}
	ints, err := feedsdynamo.AwsInts(item, []string{dbVersion, dbCreatedAt, dbUpdatedAt})
	if err != nil {
		return graphdbmodel.LoadedEdge{}, err
	}
	ret := graphdbmodel.LoadedEdge{
		Edge: graphdbmodel.Edge{
			Type: strs[dbEdgeType],
		},
		LoadedData: graphdbmodel.LoadedData{
			CreatedAt: time.Unix(0, ints[dbCreatedAt]),
			UpdatedAt: time.Unix(0, ints[dbUpdatedAt]),
			Data:      DatabagFromDynamoDB(item[dbDataBag]),
			Version:   ints[dbVersion],
		},
	}
	if err := ret.From.UnmarshalText([]byte(strs[dbFrom])); err != nil {
		return graphdbmodel.LoadedEdge{}, err
	}
	if err := ret.To.UnmarshalText([]byte(strs[dbTo])); err != nil {
		return graphdbmodel.LoadedEdge{}, err
	}
	return ret, nil
}

func DynamoDBLoadedNode(item map[string]*dynamodb.AttributeValue) (graphdbmodel.LoadedNode, error) {
	if item == nil {
		return graphdbmodel.LoadedNode{}, nil
	}
	strs, err := feedsdynamo.AwsStrings(item, []string{dbNodeKey})
	if err != nil {
		return graphdbmodel.LoadedNode{}, err
	}
	ints, err := feedsdynamo.AwsInts(item, []string{dbVersion, dbCreatedAt, dbUpdatedAt})
	if err != nil {
		return graphdbmodel.LoadedNode{}, err
	}
	ret := graphdbmodel.LoadedNode{
		Data: graphdbmodel.LoadedData{
			CreatedAt: time.Unix(0, ints[dbCreatedAt]),
			UpdatedAt: time.Unix(0, ints[dbUpdatedAt]),
			Data:      DatabagFromDynamoDB(item[dbDataBag]),
			Version:   ints[dbVersion],
		},
	}
	if err := ret.Node.UnmarshalText([]byte(strs[dbNodeKey])); err != nil {
		return graphdbmodel.LoadedNode{}, err
	}
	return ret, nil
}
