package storage

import (
	"context"
	"fmt"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/async"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/interngraphdb"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage/tablelookup"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/service-common/feedsdynamo"
	"code.justin.tv/hygienic/dynamocursor"
	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/workerpool"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/cep21/circuit"
)

// NodeStorage does logic to store nodes inside GraphDB
type NodeStorage struct {
	// Lookup tells NodeStorage which nodes GraphDB supports and important information about each node type, like
	// which DynamoDB table to store it in
	Lookup *tablelookup.Lookup
	// CursorFactory helps us encode and decode dynamodb cursors
	CursorFactory *dynamocursor.Factory
	Dynamo        *dynamodb.DynamoDB
	Logger        CtxLogger
	// When a count operation fails, we queue it inside SQS to execute later
	AsyncQueue *async.Queue
	// deprecated: we should remove this and do counts inline with the operation all the time
	CountsPool workerpool.Pool
	// deprecated: should just assume true
	BlockForCountRepair *distconf.Bool
}

// NodeCreate creates a node inside DynamoDB for GraphDB
func (n *NodeStorage) NodeCreate(ctx context.Context, node graphdbmodel.Node, data *graphdbmodel.DataBag, creationTime time.Time) (*graphdbmodel.LoadedNode, error) {
	info := n.Lookup.LookupNode(node.Type)

	if info == nil {
		return nil, errors.Errorf("unable to find information for node type %s", node.Type)
	}

	// convert empty creation times into time.Now
	creationTime = validateCreationTime(creationTime)

	items := map[string]*dynamodb.AttributeValue{
		dbNodeType: {
			S: &node.Type,
		},
		dbNodeKey: {
			S: aws.String(node.Encode()),
		},
		dbDataBag: {
			M: DynamodbItems(data),
		},
		dbCreatedAt: {
			N: aws.String(strconv.FormatInt(creationTime.UTC().UnixNano(), 10)),
		},
		dbUpdatedAt: {
			N: aws.String(strconv.FormatInt(creationTime.UTC().UnixNano(), 10)),
		},
		dbVersion: {
			N: aws.String("0"),
		},
	}
	// set the sort_key from the node metadata.  This is usually just the `dbCreatedAt` key.
	items[dbSortKey] = sortKeyFromCreation(items, info.SortKey, &info.Table)
	req, _ := n.Dynamo.PutItemRequest(&dynamodb.PutItemInput{
		TableName: &info.Table.Name,
		// Creation requires that the node doesn't already exist
		ConditionExpression: aws.String("attribute_not_exists(#nk)"),
		ExpressionAttributeNames: map[string]*string{
			"#nk": aws.String(dbNodeKey),
		},
		Item: items,
	})

	err := info.Circuits.Table.Write.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		err := req.Send()
		if isFailedConditionalCheck(err) {
			// don't count ConditionExpression failures as errors so we don't break the circuit for otherwise
			// healthy operations
			return nil
		}
		return err
	})
	if isFailedConditionalCheck(req.Error) {
		// If the node already exists, return nil/nil to signal this
		return nil, nil
	}
	if err != nil {
		n.Logger.LogCtx(ctx, "err", err, "function", "NodeCreate", "unable to create node")
		return nil, err
	}
	// Update total node counts
	n.QueueRepair(node, true)
	// Load a GraphDB node from the creation data so we can return it to the caller
	loadedNode, err := DynamoDBLoadedNode(items)
	if err != nil {
		panic(fmt.Sprintf("this is a logic error we should never fail to make from a create: %v", err))
	}
	return &loadedNode, nil
}

// NodeUpdateOrPut does an update or a put operation into DynamoDB.  It is required that the node already exist.
// updateDatabag=true will not delete existing keys that are not inside `data`, while updateDatabag=false will
// set the node's databag to exactly equal `data`
func (n *NodeStorage) NodeUpdateOrPut(ctx context.Context, node graphdbmodel.Node, data *graphdbmodel.DataBag, updateDatabag bool) (*graphdbmodel.LoadedNode, error) {
	info := n.Lookup.LookupNode(node.Type)
	if info == nil {
		return nil, errors.Errorf("unable to find information for node type %s", node.Type)
	}
	updateItemReq := &dynamodb.UpdateItemInput{
		TableName: &info.Table.Name,
		Key: map[string]*dynamodb.AttributeValue{
			dbNodeKey: {
				S: aws.String(node.Encode()),
			},
		},
		ExpressionAttributeNames: map[string]*string{
			"#node_key": aws.String(dbNodeKey),
			"#version":  aws.String(dbVersion),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			// Increment the version number by 1 on every change
			":one": {
				N: aws.String("1"),
			},
		},
		ReturnValues: aws.String("ALL_NEW"),
		// update/put operations require that the node already exist
		ConditionExpression: aws.String("attribute_exists(#node_key)"),
	}
	// setValues is a large string of a=a,b=b keys that the DynamoDB API expects to overwrite data in the node
	setValues := setUpdateOrPutDatabag(updateDatabag, data, updateItemReq, &info.Table, info.SortKey, nil)
	updateItemReq.UpdateExpression = aws.String(fmt.Sprintf("SET %s ADD #version :one", strings.Join(setValues, ", ")))
	req, resp := n.Dynamo.UpdateItemRequest(updateItemReq)
	err := info.Circuits.Table.Read.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		err := req.Send()
		if isFailedConditionalCheck(err) {
			// If the node didn't exist, don't try to open the circuit
			return nil
		}
		return err
	})
	if isFailedConditionalCheck(req.Error) {
		return nil, nil
	}
	if err != nil {
		n.Logger.LogCtx(ctx, "err", err, "function", "NodeUpdateOrPut", "unable to update node")
		return nil, err
	}
	if resp.Attributes == nil {
		// We shouldn't actually get this far since we have a ConditionExpression check above, but this would mean that
		// the node didn't previously exist
		return nil, nil
	}
	// Create a GraphDB node from the DynamoDB data
	loadedNode, err := DynamoDBLoadedNode(resp.Attributes)
	if err != nil {
		// This would be strange and should never happen, but would mean the data stored inside DynamoDB is corrupted
		return nil, err
	}
	return &loadedNode, nil
}

// NodeGet loads a node from DynamoDB
func (n *NodeStorage) NodeGet(ctx context.Context, node graphdbmodel.Node) (*graphdbmodel.LoadedNode, error) {
	info := n.Lookup.LookupNode(node.Type)
	if info == nil {
		return nil, errors.Errorf("unable to find information for node type %s", node.Type)
	}
	req, res := n.Dynamo.GetItemRequest(&dynamodb.GetItemInput{
		TableName: &info.Table.Name,
		Key: map[string]*dynamodb.AttributeValue{
			dbNodeKey: {
				S: aws.String(node.Encode()),
			},
		},
	})
	err := info.Circuits.Table.Read.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	if err != nil {
		n.Logger.LogCtx(ctx, "err", err, "function", "NodeGet", "unable to get node")
		return nil, err
	}
	if res.Item == nil {
		// If DynamoDB returns no item, then it doens't exist and we should return nil on a 404
		return nil, nil
	}
	// Create a GraphDB node from the DynamoDB data
	loadedNode, err := DynamoDBLoadedNode(res.Item)
	if err != nil {
		return nil, err
	}
	return &loadedNode, nil
}

// NodeCount uses the count DynamoDB table to count how many nodes there are of a type
func (n *NodeStorage) NodeCount(ctx context.Context, nodeType string) (int64, error) {
	info := n.Lookup.LookupNode(nodeType)
	if info == nil {
		// Someone's trying to get a count for a node type GraphDB doesn't support
		return 0, errors.Errorf("unable to find information for node type %s", nodeType)
	}
	req, res := n.Dynamo.GetItemRequest(&dynamodb.GetItemInput{
		TableName: &info.CountTableName,
		Key: map[string]*dynamodb.AttributeValue{
			dbNodeType: {
				S: &nodeType,
			},
		},
	})
	err := info.Circuits.Count.Read.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	if err != nil {
		n.Logger.LogCtx(ctx, "err", err, "function", "NodeCount", "unable to count number of nodes")
		return 0, err
	}
	foundItem := res.Item
	if foundItem == nil {
		// If the counts don't exist, it means no create operations have happened yet to increment the counts row
		return 0, nil
	}
	ints, err := feedsdynamo.AwsInts(foundItem, []string{dbCount})
	if err != nil {
		return 0, err
	}
	return ints[dbCount], nil
}

// NodeList iterates DynamoDB to return nodes inside GraphDB
func (n *NodeStorage) NodeList(ctx context.Context, nodeType string, page graphdbmodel.PagedRequest) (*graphdbmodel.ListNodeResult, error) {
	info := n.Lookup.LookupNode(nodeType)
	if info == nil {
		return nil, errors.Errorf("unable to find information for node type %s", nodeType)
	}

	indexName := "node_list"   // -- DynamoDB table index required for list queries
	indexHashKey := dbNodeType // -- All DynamoDB List operations page by a single node type

	// Decode the URL encoded cursor that we previously returned
	var c dynamocursor.Cursor
	if err := c.URLDecode(page.Cursor); err != nil {
		return nil, err
	}
	// esk means ExclusiveStartKey and is where we start the List operation when paging between node list queries
	esk, err := n.CursorFactory.ExclusiveStartKey(c, info.Table.Name, indexName)
	if err != nil {
		return nil, err
	}

	inputParams := &dynamodb.QueryInput{
		ScanIndexForward: aws.Bool(!page.DescendingOrder),
		TableName:        &info.Table.Name,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":hash_key": {
				S: &nodeType,
			},
		},
		ExpressionAttributeNames: map[string]*string{
			"#list_hash": &indexHashKey,
		},
		KeyConditionExpression: aws.String("#list_hash=:hash_key"),
		// The start key is exclusive (meaning this value isn't returned in the list) and is a combination
		// of the hash and two sort keys of where the query last stopped.
		ExclusiveStartKey: esk,
		IndexName:         &indexName,
	}

	// We force a limit.  Note that setting a really high limit doesn't mean that many values are actually returned
	// DynamoDB is fine returning # values <= the limit.
	if page.Limit != 0 {
		inputParams.Limit = &page.Limit
	}

	req, res := n.Dynamo.QueryRequest(inputParams)
	err = info.Circuits.Index.List.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	if err != nil {
		n.Logger.LogCtx(ctx, "err", err, "function", "NodeList", "unable to list the nodes")
		return nil, err
	}
	// This turns the DynamoDB items into return values
	return n.createListReturnNodes(info.Table.Name, res, indexName)
}

// createListReturnNodes converts from DynamoDB's format to GraphDB's format
func (n *NodeStorage) createListReturnNodes(tableName string, res *dynamodb.QueryOutput, index string) (*graphdbmodel.ListNodeResult, error) {
	retCursor, err := n.CursorFactory.Cursor(res.LastEvaluatedKey, tableName, index)
	if err != nil {
		return nil, err
	}
	ret := graphdbmodel.ListNodeResult{
		Nodes: make([]graphdbmodel.CursoredLoadedNode, 0, len(res.Items)),
		// The default cursor is decoded from DynamoDB's cursor
		Cursor: retCursor.URLEncode(),
	}

	for idx, item := range res.Items {
		node, err := DynamoDBLoadedNode(item)
		if err != nil {
			// If an item fails to decode, that is a data format error inside DynamoDB
			// Context does not exist
			n.Logger.LogCtx(context.Background(), "err", err, "function", "createListReturnNodes", "unable to decode data in DynamoDB")
			return nil, err
		}
		// CursorFactory allows us to create per item cursors, even though DynamoDB doesn't explicitly support them
		// We create these cursors by encoding the item's hash/range key into a string
		c, err := n.CursorFactory.Cursor(item, tableName, index)
		if err != nil {
			// Context does not exist
			n.Logger.LogCtx(context.Background(), "err", err, "function", "createListReturnNodes", "unable to create cursor for an item")
			return nil, err
		}
		ca := graphdbmodel.CursoredLoadedNode{
			LoadedNode: node,
			Cursor:     c.URLEncode(),
		}
		if idx == len(res.Items)-1 {
			// We special case the per-item cursor of the last item to be DynamoDB's cursor
			ca.Cursor = ret.Cursor
		}
		ret.Nodes = append(ret.Nodes, ca)
	}
	return &ret, nil
}

func (n *NodeStorage) NodeDelete(ctx context.Context, node graphdbmodel.Node) (*graphdbmodel.LoadedNode, error) {
	info := n.Lookup.LookupNode(node.Type)
	if info == nil {
		return nil, errors.Errorf("unable to find information for node type %s", node.Type)
	}
	req, res := n.Dynamo.DeleteItemRequest(&dynamodb.DeleteItemInput{
		TableName: &info.Table.Name,
		Key: map[string]*dynamodb.AttributeValue{
			dbNodeKey: {
				S: aws.String(node.Encode()),
			},
		},
		// ALL_OLD allows us to know if the item previously existed
		ReturnValues: aws.String("ALL_OLD"),
	})
	err := info.Circuits.Table.Write.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	if err != nil {
		n.Logger.LogCtx(ctx, "err", err, "function", "NodeDelete", "unable to delete a node")
		return nil, err
	}
	if res.Attributes == nil {
		return nil, nil
	}
	loadedNode, err := DynamoDBLoadedNode(res.Attributes)
	if err != nil {
		return nil, err
	}
	if res.Attributes != nil {
		// If the item *did* previously exist, we have to decrement the count value for that node type
		n.QueueRepair(node, false)
	}
	return &loadedNode, nil
}

// repairNodeCountRequest either increments or decrements the count table for a node type
type repairNodeCountRequest struct {
	// The node that was created or deleted
	node graphdbmodel.Node
	// True if we should increment counts, false otherwise
	isCreation bool
	n          *NodeStorage
}

func (r *repairNodeCountRequest) retryRequest() *interngraphdb.CountNodeRepairRequest {
	return &interngraphdb.CountNodeRepairRequest{
		Node: &graphdb.Node{
			Type: r.node.Type,
			Id:   r.node.ID,
		},
		IsCreation: r.isCreation,
	}
}

func (r *repairNodeCountRequest) do() interface{} {
	// Count changes shouldn't fail if we get this far, even if the original context dies.  That's because we have
	// to keep counts in sync.  Because of this, we give counts their own context.
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
	defer cancel()
	err := r.n.executeRepair(ctx, r.node, r.isCreation)
	if err != nil {
		// If the count table change failed, we have to queue the count to happen later or counts will get out of sync.
		// This often happens when we have a spike in write traffic and the count table is throttled.
		r.n.Logger.LogCtx(ctx, "err", err, "unable to repair counts.  Will add to SQS queue")
		ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
		defer cancel()
		queueErr := r.n.AsyncQueue.SendMessage(ctx, &interngraphdb.AsyncRequestQueueMessage{
			CountRetries: []*interngraphdb.CountRepairRequest{
				// TODO: Add retry aspect
				// Note: we don't actually support repair for node types, since nobody uses them heavily.  We should if
				// someone later wants to.
				//r.retryRequest(),
			},
		}, time.Minute*5)
		if queueErr != nil {
			// At this point, the count table will be out of sync. This isn't great: there will be more nodes than
			// the count table says a type should have.
			r.n.Logger.LogCtx(ctx, "err", err, "node", r.node, "is_creation", r.isCreation, "unable to queue count repair.  Count lost")
		}
	}
	return err
}

// QueueRepair executes the count incr/decr request for a node
func (n *NodeStorage) QueueRepair(node graphdbmodel.Node, isCreation bool) {
	req := repairNodeCountRequest{
		n:          n,
		node:       node,
		isCreation: isCreation,
	}
	future := workerpool.OfferOrDo(&n.CountsPool, req.do)
	if n.BlockForCountRepair.Get() {
		// This waits for the item to be fully processed by the queue
		<-future.Done()
	}
}

// executeRepair switches between creation and deletion operations
func (n *NodeStorage) executeRepair(ctx context.Context, node graphdbmodel.Node, isCreation bool) error {
	repairCtx, can := context.WithTimeout(ctx, time.Minute)
	defer can()
	if isCreation {
		return n.repairNodeCreation(repairCtx, node)
	}
	return n.repairNodeDeletion(repairCtx, node)
}

// updateCountForNodeChange changes the counts table that tells us how many of a node type there are.
func (n *NodeStorage) updateCountForNodeChange(ctx context.Context, node graphdbmodel.Node, tableName string, circuit *circuit.Circuit, countChange int) error {
	now := time.Now()
	req, res := n.Dynamo.UpdateItemRequest(&dynamodb.UpdateItemInput{
		TableName: &tableName,
		Key: map[string]*dynamodb.AttributeValue{
			dbNodeType: {
				S: &node.Type,
			},
		},
		ExpressionAttributeNames: map[string]*string{
			"#count":      aws.String(dbCount),
			"#version":    aws.String(dbVersion),
			"#updated_at": aws.String(dbUpdatedAt),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":updated_at": {
				S: aws.String(now.UTC().Format(time.RFC3339)),
			},
			// The version always goes up by one
			":one": {
				N: aws.String("1"),
			},
			// The # of a node changes either +1 or -1
			":count_change": {
				N: aws.String(strconv.Itoa(countChange)),
			},
		},
		// We return UPDATED_NEW so we could preemptively cache the new count value, if we so choose (we don't
		// currently)
		ReturnValues:     aws.String("UPDATED_NEW"),
		UpdateExpression: aws.String("ADD #version :one, #count :count_change SET #updated_at=:updated_at"),
	})
	retErr := circuit.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	if retErr != nil {
		n.Logger.LogCtx(ctx, "err", retErr, "function", "updateCountForNodeChange", "updateItemRequest for node change failed")
		return retErr
	}
	// If we get a count back and it happens to be zero, we should try to clear the current count
	if res.Attributes != nil && res.Attributes[dbCount] != nil && res.Attributes[dbCount].N != nil {
		ints, err := feedsdynamo.AwsInts(res.Attributes, []string{dbCount})
		if err != nil {
			return err
		}
		if ints[dbCount] == 0 {
			// Clean up the counts table
			return n.clearCountsIfZero(ctx, node, tableName, circuit)
		}
	}
	return nil
}

// clearCountsIfZero removes an item's count value if the value is zero. It's a nice way to save space, since we already
// consider an empty count value to be a count of zero
func (n *NodeStorage) clearCountsIfZero(ctx context.Context, node graphdbmodel.Node, tableName string, circuit *circuit.Circuit) error {
	req, _ := n.Dynamo.DeleteItemRequest(&dynamodb.DeleteItemInput{
		TableName: &tableName,
		Key: map[string]*dynamodb.AttributeValue{
			dbNodeType: {
				S: &node.Type,
			},
		},
		// Only remove the count row if the count is actually zero
		ConditionExpression: aws.String("#count = :zero"),
		ExpressionAttributeNames: map[string]*string{
			"#count": aws.String(dbCount),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":zero": {
				N: aws.String("0"),
			},
		},
	})
	circErr := circuit.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	if circErr != nil {
		n.Logger.LogCtx(ctx, "err", circErr, "function", "clearCountsIfZero", "DeleteItemRequest for NodeStorage failed")
	}

	return circErr
}

// repairNodeCreation is called when the node was freshly created
func (n *NodeStorage) repairNodeCreation(ctx context.Context, node graphdbmodel.Node) error {
	info := n.Lookup.LookupNode(node.Type)
	if info == nil {
		return errors.Errorf("unable to find information for node type %s", node.Type)
	}
	return n.updateCountForNodeChange(ctx, node, info.CountTableName, info.Circuits.Count.Write, 1)
}

// repairNodeDeletion is called when an existing node was deleted
func (n *NodeStorage) repairNodeDeletion(ctx context.Context, node graphdbmodel.Node) error {
	info := n.Lookup.LookupNode(node.Type)
	if info == nil {
		return errors.Errorf("unable to find information for node type %s", node.Type)
	}
	return n.updateCountForNodeChange(ctx, node, info.CountTableName, info.Circuits.Count.Write, -1)
}
