package storage

import (
	"context"
	"fmt"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/accesslog"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/async"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/interngraphdb"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage/tablelookup"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsdynamo"
	"code.justin.tv/hygienic/dynamocursor"
	"code.justin.tv/hygienic/statsdsender"
	"code.justin.tv/hygienic/workerpool"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/cep21/circuit"
)

// These are all the columns inside DynamoDB that GraphDB uses.
const dbFromKey = "fk"
const dbToKey = "tk"
const dbFrom = "fr"
const dbTo = "to"
const dbUpdatedAt = "ua"
const dbCreatedAt = "ca"
const dbVersion = "vr"
const dbEdgeType = "et"
const dbNodeType = "nt"
const dbNodeKey = "nk"
const dbCount = "ct"
const dbSortKey = "sk"
const dbDataBag = "db"

func init() {
	// coreTypes is only used to make sure no two values of const above are repeated
	var coreTypes = map[string]struct{}{
		dbFromKey:   {},
		dbToKey:     {},
		dbFrom:      {},
		dbTo:        {},
		dbUpdatedAt: {},
		dbCreatedAt: {},
		dbVersion:   {},
		dbEdgeType:  {},
		dbCount:     {},
		dbDataBag:   {},
		dbNodeType:  {},
		dbNodeKey:   {},
	}
	if len(coreTypes) != 12 {
		panic("Added a core type that has a conflicting key")
	}
}

// OnCountChange is a callback for when the count value of an edge type changed
type OnCountChange interface {
	NewCount(from graphdbmodel.Node, edgeKind string, newCount int64)
}

type Client interface {
	// Errors if the item already exists.  Returns created item.
	Create(ctx context.Context, edge graphdbmodel.Edge, data *graphdbmodel.DataBag, creationTime time.Time) (*graphdbmodel.LoadedEdge, error)

	// Returns nil, nil if the item does not exist.  Update adds to the databag, put replaces it.  Returns full new item.
	UpdateOrPut(ctx context.Context, edge graphdbmodel.Edge, data *graphdbmodel.DataBag, updateDatabag bool, creationTime *time.Time, requiredVersion *time.Time) (*graphdbmodel.LoadedEdge, error)

	// Get an assoc.  Nil if it does not exist.
	Get(ctx context.Context, edge graphdbmodel.Edge) (*graphdbmodel.LoadedEdge, error)

	// Count outgoing edges from a node.
	Count(ctx context.Context, from graphdbmodel.Node, edgeKind string) (int64, error)

	// List the nodes from an edge.  If sortkey is empty, uses default sort (usually created_at).  Not all sortkey are supported on all types.
	List(ctx context.Context, from graphdbmodel.Node, edgeKind string, page graphdbmodel.PagedRequest) (*graphdbmodel.ListResult, error)

	// Delete (remove) an assoc.  Returns the old assoc (nil if it did not exist).
	Delete(ctx context.Context, edge graphdbmodel.Edge, requiredVersion *time.Time) (*graphdbmodel.LoadedEdge, error)

	// Override the current value in the Count table with count. Creates an entry if one does not exist.
	// OverrideCount returns the previous value that was overridden by count
	OverrideCount(ctx context.Context, from graphdbmodel.Node, edgeKind string, count int64) (int64, error)
}

type CtxLogger interface {
	LogCtx(ctx context.Context, keyvals ...interface{})
}

// BackoffAsyncRequests allows the storage layer to signal to Async processing that it is throttled and async
// queue draining should slow down.
type BackoffAsyncRequests interface {
	// BackoffAsync is called any time count values cannot be updated fast enough.  We don't baccokoff on the edge
	// creation throttle events, just the count throttle events, because the count throttles are worse since they force
	// extra SQS messages to keep track of
	BackoffAsync()
}

// Storage does edge fetch and mutation operations
type Storage struct {
	// CountChange allows us to signal back to Memcache that the count of an edge type has changed. This allows us to
	// write the new count value directly into memcache
	CountChange OnCountChange
	// Lookup returns metadata about an edge. For example, it tells Storage which DynamoDB table to write the 'follows'
	// edge type to
	Lookup *tablelookup.Lookup
	// CursorFactory is used during List queries to create per item cursors
	CursorFactory *dynamocursor.Factory
	Logger        CtxLogger
	// deprecated: We should just remove this.  It's never used
	CountsPool workerpool.Pool
	// deprecated: We should just remove this.  It's always true
	BlockForCountRepair *distconf.Bool
	// Dynamo is the aws client that talks to DynamoDB
	Dynamo *dynamodb.DynamoDB
	// BackoffAsyncRequests gets signals about throttling during count repair, slowing down async processing
	BackoffAsyncRequests BackoffAsyncRequests
	// AsyncQueue is where we put count operations that fail so we can do them again later.  Otherwise, the edge
	// and edge-count table could get out of sync if an edge operation works and the edge-count operation fails
	AsyncQueue *async.Queue
	Stats      *statsdsender.ErrorlessStatSender `nilcheck:"ignore"`
}

// Setup/Start/Close could probably be removed since we're removing CountsPool too
func (s *Storage) Setup() error {
	return nil
}

func (s *Storage) Start() error {
	return nil
}

func (s *Storage) Close() error {
	return s.CountsPool.Close()
}

var _ Client = &Storage{}

// setUpdateOrPutDatabag populates a UpdateExpression for a dynamodb.UpdateItemInput operation.  This includes setting
// metadata like updated_at or created_at, as well as updating the databag key with parts of data that are changing.
//
// The return values look something like []string{"#updated_at=:updated_at", "db.#notify=:notify"}, with updateItemReq
// getting updated ExpressionAttributeNames and ExpressionAttributeValues that have the correct key pairs.  The returned
// string can be joined(",") and is escape safe to place inside the UpdateItemInput
func setUpdateOrPutDatabag(updateDatabag bool, data *graphdbmodel.DataBag, updateItemReq *dynamodb.UpdateItemInput, info DefaultSortKeyCreation, sortKey string, creationTime *time.Time) []string {
	now := time.Now()
	setValues := make([]string, 0, 6)
	// All update operations change the updated_at value
	updateItemReq.ExpressionAttributeNames["#updated_at"] = aws.String(dbUpdatedAt)
	updateItemReq.ExpressionAttributeValues[":updated_at"] = &dynamodb.AttributeValue{
		N: aws.String(strconv.FormatInt(now.UTC().UnixNano(), 10)),
	}
	setValues = append(setValues, "#updated_at=:updated_at")

	if sortKey == dbUpdatedAt {
		// All DynamoDB tables have a sort key.  For GraphDB, we usually sort by the creation date.  However, that is
		// not required.  Because of this, we have two explicit values creation date and sort key.  If the sort key
		// is the creation date, then both values change on each operation
		updateItemReq.ExpressionAttributeNames["#sort_key"] = aws.String(dbSortKey)
		setValues = append(setValues, fmt.Sprintf("#sort_key=%s", ":updated_at"))
	}
	if creationTime != nil {
		// For Sync operations, we allow changing the creation time of a node or edge.  Usually, this is null
		updateItemReq.ExpressionAttributeNames["#created_at"] = aws.String(dbCreatedAt)
		updateItemReq.ExpressionAttributeValues[":created_at"] = &dynamodb.AttributeValue{
			N: aws.String(strconv.FormatInt(creationTime.UTC().UnixNano(), 10)),
		}
		setValues = append(setValues, "#created_at=:created_at")
	}
	if updateDatabag {
		// If the databag is being updated, then we need to explicitly change each key in the data bag column
		idx := 0
		// Convert the databag into individual DynamoDB items and store them one at a time with the []string{} return
		for k, v := range DynamodbItems(data) {
			k := k
			v := v
			//  We insert something like []string{"db.#dbidx1=:dbidx1"} into the return value, then set #dbidx1 and
			// :dbidx1 to the correct key/value inside the databag
			expressionAttributeName := fmt.Sprintf("#dbidx%d", idx)
			expressionAttributeValue := fmt.Sprintf(":dbidx%d", idx)
			setValues = append(setValues, fmt.Sprintf("%s.%s=%s", dbDataBag, expressionAttributeName, expressionAttributeValue))
			updateItemReq.ExpressionAttributeNames[expressionAttributeName] = &k
			updateItemReq.ExpressionAttributeValues[expressionAttributeValue] = v
			idx++

			// If the sort key is a databag item, and not just creation date, then we update sort key as well when that
			// value changes
			if dbDataBag+"."+k == sortKey {
				updateItemReq.ExpressionAttributeNames["#sort_key"] = aws.String(dbSortKey)
				setValues = append(setValues, fmt.Sprintf("#sort_key=%s", expressionAttributeValue))
			}
		}
	} else {
		// In this case, we don't want to individually change items inside databag, we want to overwrite databag with
		// the contents of `data`.
		databagItems := DynamodbItems(data)

		if strings.HasPrefix(sortKey, dbDataBag+".") {
			// If the sort key is a value inside the databag, then we need to set it from what the new value will be
			updateItemReq.ExpressionAttributeNames["#sort_key"] = aws.String(dbSortKey)
			setValues = append(setValues, "#sort_key=:sort_key")
			databagKey := sortKey[len(dbDataBag+"."):]
			if dbValue, exists := databagItems[databagKey]; exists {
				updateItemReq.ExpressionAttributeValues[":sort_key"] = dbValue
			} else {
				// In this case, the sort key is a databag value, but we are doing an overwrite operation that will not
				// have a sort key inside it (the sort key isn't inside databag).  We have to set the sort key to
				// empty in this case and empty depends upon the data type of the sort key.
				updateItemReq.ExpressionAttributeValues[":sort_key"] = info.DefaultSortKeyValue(dbSortKey)
			}
		}

		// While update operations have individual `db.#dbidx1=:dbidx1` values, overwrite operations can be represented
		// as a single #data=:data where :data is the entire databag.
		setValues = append(setValues, "#data=:data")
		updateItemReq.ExpressionAttributeNames["#data"] = aws.String(dbDataBag)
		updateItemReq.ExpressionAttributeValues[":data"] = &dynamodb.AttributeValue{
			M: databagItems,
		}
	}
	return setValues
}

// UpdateOrPut modifies an existing row in DynamoDB
// requiredVersion is an optional argument that is used for optimistic locking.
// If the requiredVersion is provided, it must match the current record in dynamo.
// Otherwise, the row is not updated and a nil Edge is returned.
func (s *Storage) UpdateOrPut(ctx context.Context, edge graphdbmodel.Edge, data *graphdbmodel.DataBag, updateDatabag bool, creationTime *time.Time, requiredVersion *time.Time) (*graphdbmodel.LoadedEdge, error) {
	info := s.Lookup.LookupEdge(edge.Type)
	if info == nil {
		return nil, errors.Errorf("unable to find information for edge type %s", edge.Type)
	}
	if !info.PrimaryEdgeType {
		// If the update is for the reverse direction, then only do primary edge operations.  We have to UpdateOrPut
		// the reverse edge
		return s.reverseAssoc(s.UpdateOrPut(ctx, edge.Reversed(info.ReverseName), data, updateDatabag, creationTime, requiredVersion))
	}

	conditions := strings.Builder{}
	// Check the existence of an attribute to prevent the item from being created if it does not exist.
	must(conditions.WriteString("attribute_exists(#from_key) AND attribute_exists(#t)"))

	if requiredVersion != nil {
		must(conditions.WriteString(" AND #updated_at = :required_version"))
	}

	updateItemReq := &dynamodb.UpdateItemInput{
		// Return ReturnConsumedCapacity so we can add to our access logs how many DynamoDB units a request used
		ReturnConsumedCapacity: aws.String("TOTAL"),
		TableName:              &info.Table.Name,
		Key: map[string]*dynamodb.AttributeValue{
			dbFromKey: {
				S: aws.String(fmt.Sprintf("%s:%s", edge.From.Encode(), edge.Type)),
			},
			dbTo: {
				S: aws.String(edge.To.Encode()),
			},
		},
		ExpressionAttributeNames: map[string]*string{
			"#from_key": aws.String(dbFromKey),
			"#t":        aws.String(dbTo),
			"#version":  aws.String(dbVersion),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			// During mutations, the version always changes by one
			":one": {
				N: aws.String("1"),
			},
		},
		ReturnValues:        aws.String("ALL_NEW"),
		ConditionExpression: aws.String(conditions.String()),
	}
	setValues := setUpdateOrPutDatabag(updateDatabag, data, updateItemReq, &info.Table, info.SortKey, creationTime)
	updateItemReq.UpdateExpression = aws.String(fmt.Sprintf("SET %s ADD #version :one", strings.Join(setValues, ", ")))

	if requiredVersion != nil {
		updateItemReq.ExpressionAttributeValues[":required_version"] = &dynamodb.AttributeValue{
			N: aws.String(strconv.FormatInt(requiredVersion.UTC().UnixNano(), 10)),
		}
	}

	req, resp := s.Dynamo.UpdateItemRequest(updateItemReq)

	start := time.Now()
	err := info.Circuits.Table.Write.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		err := req.Send()
		if isFailedConditionalCheck(err) {
			// This means there was an update operation on an edge that doesn't exist.  We shouldn't open the circuit
			// for this
			return nil
		}
		return err
	})

	traceDynamoLatency(ctx, time.Since(start), req.RequestID)
	// Log consumed capacity for the access logs
	incrConsumedCapacity(ctx, resp.ConsumedCapacity)
	if isFailedConditionalCheck(req.Error) {
		//  Again, if the edge just didn't exist, that's not actually an error.
		return nil, nil
	}
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "Get", "err", err, "ddb_request_id", req.RequestID, "table_name", info.Table.Name, "circuit_time", time.Since(start), "error in DDB circuit call")
		return nil, err
	}
	if resp.Attributes == nil {
		// This should't happen.  It means there was no ReturnValues
		s.Stats.IncC("update.notexist", 1, 1.0)
		return nil, nil
	}
	// Turn the DynamoDB row into a GraphDB edge
	assoc, err := DynamoDBAssociation(resp.Attributes)
	if err != nil {
		return nil, err
	}
	return &assoc, nil
}

// must will panic if the call fails.  It's useful for calls that should never fail. I.E. strings.Builder
func must(_ int, err error) {
	if err != nil {
		panic(err)
	}
}

// incrConsumedCapacity adds consumed capacity for read and write operations to the access logs
func incrConsumedCapacity(ctx context.Context, cap *dynamodb.ConsumedCapacity) {
	if cap != nil && cap.CapacityUnits != nil {
		accesslog.TraceFloatInc(ctx, "consumed_dynamo_capacity", *cap.CapacityUnits)
	}
}

// traceDynamoLatency logs client-perceived latency for dynamoDB calls
func traceDynamoLatency(ctx context.Context, callDuration time.Duration, requestID string) {
	accesslog.TraceIntInc(ctx, "dynamo_latency_ms", callDuration.Nanoseconds()/time.Millisecond.Nanoseconds())
	accesslog.TraceStringSet(ctx, "dynamodb_request_id", requestID)
}

// reverseAssoc reverses an edge.  For example, a follow edge becomes a followed_by edge
func (s *Storage) reverseAssoc(assoc *graphdbmodel.LoadedEdge, err error) (*graphdbmodel.LoadedEdge, error) {
	if err != nil {
		return nil, err
	}
	if assoc == nil {
		return nil, nil
	}
	info := s.Lookup.LookupEdge(assoc.Type)
	if info == nil {
		return nil, errors.Errorf("unable to find information for edge type %s", assoc.Type)
	}

	a := assoc.Reverse(info.ReverseName)
	return &a, nil
}

// Get a row from DynamoDB and turn it into a GraphDB edge
func (s *Storage) Get(ctx context.Context, edge graphdbmodel.Edge) (*graphdbmodel.LoadedEdge, error) {
	info := s.Lookup.LookupEdge(edge.Type)
	if info == nil {
		return nil, errors.Errorf("unable to find information for edge type %s", edge.Type)
	}
	if !info.PrimaryEdgeType {
		// Reverse and get the reverse type.  Then we have to reverse it back
		return s.reverseAssoc(s.Get(ctx, edge.Reversed(info.ReverseName)))
	}
	req, res := s.Dynamo.GetItemRequest(&dynamodb.GetItemInput{
		ReturnConsumedCapacity: aws.String("TOTAL"),
		TableName:              &info.Table.Name,
		Key: map[string]*dynamodb.AttributeValue{
			dbFromKey: {
				S: aws.String(fmt.Sprintf("%s:%s", edge.From.Encode(), edge.Type)),
			},
			dbTo: {
				S: aws.String(edge.To.Encode()),
			},
		},
	})

	var circitRunStartDiff time.Duration
	var circitRunEndDiff time.Duration
	start := time.Now()
	err := info.Circuits.Table.Read.Run(ctx, func(ctx context.Context) error {
		circitRunStartDiff = time.Since(start)
		req.SetContext(ctx)
		circErr := req.Send()
		circitRunEndDiff = time.Since(start)
		return circErr
	})

	if time.Since(start) > time.Duration(500*time.Millisecond) { // randomly chosen for debugging
		s.Logger.LogCtx(ctx, "function", "Get", "ddb_request_id", req.RequestID, "table_name", info.Table.Name, "startTime", start, "endTimeDiff", time.Since(start), "circitRunStartDiff", circitRunStartDiff, "circitRunEndDiff", circitRunEndDiff)
	}
	s.Stats.TimingDurationC("storage.Get.circitRunStartDiff", circitRunStartDiff, 0.1)
	s.Stats.TimingDurationC("storage.Get.circitRunEndDiff", circitRunEndDiff, 0.1)
	s.Stats.TimingDurationC("storage.Get.total", time.Since(start), 0.1)

	traceDynamoLatency(ctx, time.Since(start), req.RequestID)
	incrConsumedCapacity(ctx, res.ConsumedCapacity)
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "Get", "err", err, "ddb_request_id", req.RequestID, "table_name", info.Table.Name, "circuit_time", time.Since(start), "error in DDB circuit call")
		return nil, err
	}
	if res.Item == nil {
		// This happens when the item doesn't exist
		return nil, nil
	}
	assoc, err := DynamoDBAssociation(res.Item)
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "Get", "err", err, "ddb_request_id", req.RequestID, "table_name", info.Table.Name, "error in DynamoDBAssociation call")
		return nil, err
	}
	return &assoc, nil
}

// List iterates and returns edges from DynamoDB
func (s *Storage) List(ctx context.Context, from graphdbmodel.Node, edgeKind string, page graphdbmodel.PagedRequest) (*graphdbmodel.ListResult, error) {
	info := s.Lookup.LookupEdge(edgeKind)
	if info == nil {
		return nil, errors.Errorf("unable to find information for edge type %s", edgeKind)
	}

	// There are two indexes in the DynamoDB table.  One index allows us to iterate edges by one direction (follows)
	// while the other index allows iteration of edges by the other direction (followed_by).  This is not to be
	// confused with iterating edge types by direction.  For example, you could iterate 'Who follows user X
	// in ascending order' as well as 'Who is followed_by user X in descending order' as well as 'Who follows user X
	// in descending order'
	var indexName string    // --- The name of the DynamoDB index to use
	var indexHashKey string // --- The hash key that's important for that index
	if info.PrimaryEdgeType {
		indexName = "from_list"
		indexHashKey = dbFromKey
	} else {
		indexName = "to_list"
		indexHashKey = dbToKey
	}

	// Page queries with a cursor never get cached.  We want to possible throttle, but for sure monitor, how many of
	// this type we force DynamoDB to deal with
	if page.Cursor != "" {
		s.Stats.IncC(fmt.Sprintf("%s.%s.cursor", info.Table.Name, indexName), 1, .5)
	} else {
		s.Stats.IncC(fmt.Sprintf("%s.%s.nocursor", info.Table.Name, indexName), 1, .5)
	}

	inputParams := &dynamodb.QueryInput{
		// Return consumed capacity so we can put it in the access logs for debug purposes
		ReturnConsumedCapacity: aws.String("TOTAL"),
		ScanIndexForward:       aws.Bool(!page.DescendingOrder),
		TableName:              &info.Table.Name,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			// The hash key is which node we're iterating from, along with which edge type we're iterating
			":hash_key": {
				S: aws.String(fmt.Sprintf("%s:%s", from.Encode(), edgeKind)),
			},
		},
		ExpressionAttributeNames: map[string]*string{
			"#list_hash": &indexHashKey,
		},
		KeyConditionExpression: aws.String("#list_hash=:hash_key"),
		IndexName:              &indexName,
	}

	var c dynamocursor.Cursor
	var rdsCursorTimestamp string
	if err := c.URLDecode(page.Cursor); err != nil {
		// TODO: remove this code once RDS is no longer being used.
		// Cohesion RDS uses timestamp as a cursor.  To fluidly move between both datastore, we also support a
		// timestamp cursor.  This feature, however, should be considered temporary and removed when Cohesion is dead
		if _, err := strconv.ParseInt(page.Cursor, 10, 64); err != nil {
			// TODO: Return that this is a 4xx style error, not a 5xx style error: since the cursor is invalid
			return nil, err
		}
		rdsCursorTimestamp = page.Cursor
	}

	if !c.IsZero() {
		// If there is a cursor, we must set an ExclusiveStartKey to know where to start our list query
		esk, err := s.CursorFactory.ExclusiveStartKey(c, info.Table.Name, indexName)
		if err != nil {
			return nil, err
		}
		inputParams.ExclusiveStartKey = esk
	}

	// TODO: remove this code once RDS is no longer being used.
	if c.IsZero() && rdsCursorTimestamp != "" {
		// This hack only works if the DynamoDB sort key is creation date
		inputParams.ExpressionAttributeValues[":rdsCursor"] = &dynamodb.AttributeValue{N: aws.String(rdsCursorTimestamp)}
		inputParams.ExpressionAttributeNames["#createdAt"] = aws.String(dbSortKey)

		if page.DescendingOrder {
			inputParams.KeyConditionExpression = aws.String("#list_hash=:hash_key AND #createdAt<:rdsCursor")
		} else {
			inputParams.KeyConditionExpression = aws.String("#list_hash=:hash_key AND #createdAt>:rdsCursor")
		}
	}

	if page.Limit != 0 {
		// Set a limit if one was asked for.  Otherwise, will return as many items as DynamoDB decides to give us.  Note
		// that a limit is an upper bound, it is not a lower bound on how many items may be returned.
		inputParams.Limit = &page.Limit
	}

	req, res := s.Dynamo.QueryRequest(inputParams)

	var circitRunStartDiff time.Duration
	var circitRunEndDiff time.Duration
	start := time.Now()
	err := info.Circuits.Index.List.Run(ctx, func(ctx context.Context) error {
		circitRunStartDiff = time.Since(start)
		req.SetContext(ctx)
		circErr := req.Send()
		circitRunEndDiff = time.Since(start)
		return circErr
	})

	if time.Since(start) > time.Duration(500*time.Millisecond) { // randomly chosen for debugging
		s.Logger.LogCtx(ctx, "function", "List", "ddb_request_id", req.RequestID, "table_name", info.Table.Name, "startTime", start, "endTimeDiff", time.Since(start), "circitRunStartDiff", circitRunStartDiff, "circitRunEndDiff", circitRunEndDiff)
	}
	s.Stats.TimingDurationC("storage.List.circitRunStartDiff", circitRunStartDiff, 0.1)
	s.Stats.TimingDurationC("storage.List.circitRunEndDiff", circitRunEndDiff, 0.1)
	s.Stats.TimingDurationC("storage.List.total", time.Since(start), 0.1)

	traceDynamoLatency(ctx, time.Since(start), req.RequestID)
	incrConsumedCapacity(ctx, res.ConsumedCapacity)
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "List", "ddb_request_id", req.RequestID, "table_name", info.Table.Name, "err", err, "circuit_time", time.Since(start), "error in DDB circuit call")
		return nil, err
	}
	return s.createListReturn(info, res, indexName)
}

// createListReturn turns a DynamoDB Query into a list of GraphDB edges
func (s *Storage) createListReturn(info *tablelookup.EdgeInfo, res *dynamodb.QueryOutput, index string) (*graphdbmodel.ListResult, error) {
	retCursor, err := s.CursorFactory.Cursor(res.LastEvaluatedKey, info.Table.Name, index)
	if err != nil {
		return nil, err
	}
	ret := graphdbmodel.ListResult{
		To: make([]graphdbmodel.CursoredLoadedEdge, 0, len(res.Items)),
		// This cursor we get from DynamoDB and can put directly in the return.  Other per-item cursors we have to
		// calculate from each item
		Cursor: retCursor.URLEncode(),
	}

	for idx, item := range res.Items {
		assoc, err := DynamoDBAssociation(item)
		if err != nil {
			return nil, err
		}
		if !info.PrimaryEdgeType {
			// If people asked for the reverse edge type, followed_by rather than follows, we have to reverse
			// each item since it is stored in DynamoDB as the primary `follows` type
			assoc = assoc.Reverse(info.Name)
		}

		c, err := s.CursorFactory.Cursor(item, info.Table.Name, index)
		if err != nil {
			return nil, err
		}
		ca := graphdbmodel.CursoredLoadedEdge{
			LoadedEdge: assoc,
			Cursor:     c.URLEncode(),
		}
		if idx == len(res.Items)-1 {
			// We can special case the last item's cursor to the exclusive start key that's returned by the DynamoDB
			// request, rather than using the CursorFactory to generate it
			ca.Cursor = ret.Cursor
		}
		ret.To = append(ret.To, ca)
	}
	return &ret, nil
}

// OverrideCount directly changes the count to a value.  This is used when we have to fix the stored counts and
// need to fix out of sync tables.
func (s *Storage) OverrideCount(ctx context.Context, from graphdbmodel.Node, edgeKind string, count int64) (int64, error) {
	info := s.Lookup.LookupEdge(edgeKind)
	if info == nil {
		return 0, errors.Errorf("unable to find information for edge type %s", edgeKind)
	}

	req, res := s.Dynamo.UpdateItemRequest(&dynamodb.UpdateItemInput{
		TableName:              &info.CountTableName,
		ReturnConsumedCapacity: aws.String("TOTAL"),
		Key: map[string]*dynamodb.AttributeValue{
			dbFrom:     {S: aws.String(from.Encode())},
			dbEdgeType: {S: aws.String(edgeKind)},
		},
		ExpressionAttributeNames: map[string]*string{
			"#updated_at": aws.String(dbUpdatedAt),
			"#count":      aws.String(dbCount),
			"#version":    aws.String(dbVersion),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":updated_at": {S: aws.String(time.Now().UTC().Format(time.RFC3339))},
			":count":      {N: aws.String(strconv.FormatInt(count, 10))},
			":one":        {N: aws.String("1")},
		},
		UpdateExpression: aws.String("SET #updated_at=:updated_at, #count=:count ADD #version :one"),
		// We return the previous count when we override, because the new count is obviously 'count'.  We can use
		// this returned value to see if the counts were actually out of sync and by how much
		ReturnValues: aws.String("UPDATED_OLD"),
	})

	start := time.Now()
	err := info.Circuits.Count.Write.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})

	traceDynamoLatency(ctx, time.Since(start), req.RequestID)
	incrConsumedCapacity(ctx, res.ConsumedCapacity)
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "OverrideCount", "ddb_request_id", req.RequestID, "table_name", info.Table.Name, "err", err, "circuit_time", time.Since(start), "error in DDB circuit call")
		return 0, err
	}

	if res.Attributes == nil {
		// If there was no previously stored count, then that's like the count was previously zero
		return 0, nil
	}
	ints, err := feedsdynamo.AwsInts(res.Attributes, []string{dbCount})
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "OverrideCount", "err", "ddb_request_id", req.RequestID, "table_name", info.Table.Name, err, "error in converting res Attributes")
		return 0, err
	}
	// Return the previously stored count
	return ints[dbCount], err
}

// Count looks up the number of edges of a type using the count DynamoDB table
func (s *Storage) Count(ctx context.Context, from graphdbmodel.Node, edgeKind string) (int64, error) {
	info := s.Lookup.LookupEdge(edgeKind)
	if info == nil {
		return 0, errors.Errorf("unable to find information for edge type %s", edgeKind)
	}

	req, res := s.Dynamo.GetItemRequest(&dynamodb.GetItemInput{
		TableName:              &info.CountTableName,
		ReturnConsumedCapacity: aws.String("TOTAL"),
		Key: map[string]*dynamodb.AttributeValue{
			dbFrom: {
				S: aws.String(from.Encode()),
			},
			dbEdgeType: {
				S: aws.String(edgeKind),
			},
		},
	})

	// Because counts are stored in a different dynamodb table than edges, we use their own circuit for reads
	// from that table
	var circitRunStartDiff time.Duration
	var circitRunEndDiff time.Duration
	start := time.Now()

	err := info.Circuits.Index.List.Run(ctx, func(ctx context.Context) error {
		circitRunStartDiff = time.Since(start)
		req.SetContext(ctx)
		circErr := req.Send()
		circitRunEndDiff = time.Since(start)
		return circErr
	})

	if time.Since(start) > time.Duration(500*time.Millisecond) { // randomly chosen for debugging
		s.Logger.LogCtx(ctx, "function", "Count", "ddb_request_id", req.RequestID, "table_name", info.Table.Name, "startTime", start, "endTimeDiff", time.Since(start), "circitRunStartDiff", circitRunStartDiff, "circitRunEndDiff", circitRunEndDiff)
	}
	s.Stats.TimingDurationC("storage.Count.circitRunStartDiff", circitRunStartDiff, 0.1)
	s.Stats.TimingDurationC("storage.Count.circitRunEndDiff", circitRunEndDiff, 0.1)
	s.Stats.TimingDurationC("storage.Count.total", time.Since(start), 0.1)

	traceDynamoLatency(ctx, time.Since(start), req.RequestID)
	incrConsumedCapacity(ctx, res.ConsumedCapacity)
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "Count", "ddb_request_id", req.RequestID, "table_name", info.Table.Name, "err", err, "circuit_time", time.Since(start), "error in DDB circuit call")
		return 0, err
	}

	foundItem := res.Item
	if foundItem == nil {
		return 0, nil
	}
	ints, err := feedsdynamo.AwsInts(foundItem, []string{dbCount})
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "Count", "err", err, "error in converting foundItem to AwsInts circuit call")
		return 0, err
	}
	return ints[dbCount], nil
}

// Delete removes an edge and decrements the count value for the edge's type
// requiredVersion is an optional argument that is used for optimistic locking.
// If the requiredVersion is provided, it must match the current record in dynamo.
// Otherwise, the row is not deleted and a nil Edge is returned.
func (s *Storage) Delete(ctx context.Context, edge graphdbmodel.Edge, requiredVersion *time.Time) (*graphdbmodel.LoadedEdge, error) {
	info := s.Lookup.LookupEdge(edge.Type)
	if info == nil {
		return nil, errors.Errorf("unable to find information for edge type %s", edge.Type)
	}
	if !info.PrimaryEdgeType {
		// Reverse and delete the reverse type
		// Note we have to reverse the reversed delete, so the returned edge is the same type as the delete request
		return s.reverseAssoc(s.Delete(ctx, edge.Reversed(info.ReverseName), requiredVersion))
	}
	deleteItem := &dynamodb.DeleteItemInput{
		TableName:              &info.Table.Name,
		ReturnConsumedCapacity: aws.String("TOTAL"),
		Key: map[string]*dynamodb.AttributeValue{
			dbFromKey: {
				S: aws.String(fmt.Sprintf("%s:%s", edge.From.Encode(), edge.Type)),
			},
			dbTo: {
				S: aws.String(edge.To.Encode()),
			},
		},
		// Return the old value so callers can know if the deleted row previously existed
		ReturnValues: aws.String("ALL_OLD"),
	}

	if requiredVersion != nil {
		deleteItem.ExpressionAttributeNames = map[string]*string{
			"#updated_at": aws.String(dbUpdatedAt),
		}

		deleteItem.ExpressionAttributeValues = map[string]*dynamodb.AttributeValue{
			":required_version": {N: aws.String(strconv.FormatInt(requiredVersion.UTC().UnixNano(), 10))},
		}

		condition := "#updated_at = :required_version"
		deleteItem.ConditionExpression = aws.String(condition)
	}

	req, res := s.Dynamo.DeleteItemRequest(deleteItem)

	start := time.Now()
	err := info.Circuits.Table.Write.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		err := req.Send()
		if isFailedConditionalCheck(err) {
			// This means there was a delete operation did not match the expected version.
			// Or that the edge never existed.
			// We shouldn't open the circuit for this
			return nil
		}
		return err
	})

	traceDynamoLatency(ctx, time.Since(start), req.RequestID)
	incrConsumedCapacity(ctx, res.ConsumedCapacity)
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "Delete", "err", err, "circuit_time", time.Since(start), "table_name", info.Table.Name, "error in DDB circuit call")
		return nil, err
	}
	if isFailedConditionalCheck(req.Error) {
		//  Again, if the version does not match, that's not actually an error.
		return nil, nil
	}
	if res.Attributes == nil {
		s.Stats.IncC("delete.notexist", 1, 1.0)
		return nil, nil
	}
	assoc, err := DynamoDBAssociation(res.Attributes)
	if err != nil {
		return nil, err
	}
	if res.Attributes != nil {
		// If the edge previously existed, we have to decrement the `count` value for that edge type so that table
		// doesn't get out of sync
		s.QueueRepair(edge.From, edge.Type, edge.To, false)
	}
	return &assoc, nil
}

// validateCreationTime returns the current time if the `t` is empty.  This is usually during edge creations, where
// the caller can optionally pass a creation time
func validateCreationTime(t time.Time) time.Time {
	if t.IsZero() {
		return time.Now()
	}
	return t
}

// createKey returns the DynamoDB hash key for an node+edge
func createKey(e graphdbmodel.Node, edgeKind string) string {
	return fmt.Sprintf("%s:%s", e.Encode(), edgeKind)
}

// Create an edge and increment the count row
func (s *Storage) Create(ctx context.Context, edge graphdbmodel.Edge, data *graphdbmodel.DataBag, creationTime time.Time) (*graphdbmodel.LoadedEdge, error) {
	info := s.Lookup.LookupEdge(edge.Type)

	if info == nil {
		return nil, errors.Errorf("unable to find information for edge type %s", edge.Type)
	}
	if !info.PrimaryEdgeType {
		// Reverse and store the reverse type, since we only store the primary edge
		return s.reverseAssoc(s.Create(ctx, edge.Reversed(info.ReverseName), data, creationTime))
	}

	// Most of the time, people don't pass a creation time when we make an edge, so we insert time.Now as the creation
	creationTime = validateCreationTime(creationTime)

	items := map[string]*dynamodb.AttributeValue{
		// Where the edge is from
		dbFrom: {
			S: aws.String(edge.From.Encode()),
		},
		// The DynamoDB hash key for the from list, and primary table
		dbFromKey: {
			S: aws.String(createKey(edge.From, edge.Type)),
		},
		// The destination of the edge
		dbTo: {
			S: aws.String(edge.To.Encode()),
		},
		// DynamoDB hash key for the to list
		dbToKey: {
			S: aws.String(createKey(edge.To, info.ReverseName)),
		},
		// Type of the edge, 'follows' vs 'followed_by'.  Also the sort key of the primary table
		dbEdgeType: {
			S: &edge.Type,
		},
		// Data stored with the edge, as a JSON object
		dbDataBag: {
			M: DynamodbItems(data),
		},
		// WHen the edge was created
		dbCreatedAt: {
			N: aws.String(strconv.FormatInt(creationTime.UTC().UnixNano(), 10)),
		},
		// When the edge was last updated
		dbUpdatedAt: {
			N: aws.String(strconv.FormatInt(creationTime.UTC().UnixNano(), 10)),
		},
		// Incremented each time we modify the edge
		dbVersion: {
			N: aws.String("0"),
		},
	}
	// Set the correct sort key for list queries (to and from).  This is usually the creation date
	items[dbSortKey] = sortKeyFromCreation(items, info.SortKey, &info.Table)
	req, resp := s.Dynamo.PutItemRequest(&dynamodb.PutItemInput{
		ReturnConsumedCapacity: aws.String("TOTAL"),
		TableName:              &info.Table.Name,
		// Only allow a creation if the item doesn't already exist
		ConditionExpression: aws.String("attribute_not_exists(#f) AND attribute_not_exists(#t)"),
		ExpressionAttributeNames: map[string]*string{
			"#t": aws.String(dbTo),
			"#f": aws.String(dbFromKey),
		},
		Item: items,
	})

	start := time.Now()
	err := info.Circuits.Table.Write.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		err := req.Send()
		// Don't consider creating an edge twice as a circuit failure
		if isFailedConditionalCheck(err) {
			return nil
		}
		return err
	})

	traceDynamoLatency(ctx, time.Since(start), req.RequestID)
	incrConsumedCapacity(ctx, resp.ConsumedCapacity)
	if isFailedConditionalCheck(req.Error) {
		s.Stats.IncC("create.exists", 1, 1.0)
		return nil, nil
	}
	if err != nil {
		s.Logger.LogCtx(ctx, "function", "Create", "err", err, "circuit_time", time.Since(start), "table_name", info.Table.Name, "error in DDB circuit call")
		return nil, err
	}
	// Increment the count table to keep counts in sync with the edges
	s.QueueRepair(edge.From, edge.Type, edge.To, true)
	assoc, err := DynamoDBAssociation(items)
	if err != nil {
		panic(fmt.Sprintf("this is a logic error we should never fail to make from a create: %v", err))
	}
	return &assoc, nil
}

// sortKeyFromCreation returns the to/from sort key from a creation, by looking at the data inside the creation
// request.  Usually we just extract the creation time
func sortKeyFromCreation(items map[string]*dynamodb.AttributeValue, sortKey string, info DefaultSortKeyCreation) *dynamodb.AttributeValue {
	currentValue, exists := items[sortKey]
	if exists {
		return currentValue
	}
	if strings.HasPrefix(sortKey, dbDataBag+".") {
		// Also support a sort key that exist inside the data bag
		databagKey := sortKey[len(dbDataBag+"."):]
		if dbValue, exists := items[dbDataBag].M[databagKey]; exists {
			return dbValue
		}
	}
	// A sort key is required: so when the sort key isn't in the creation we need to set a default empty key
	return info.DefaultSortKeyValue(dbSortKey)
}

// isFailedConditionalCheck returns true if the error is a failed conditional.  If we don't check this, a circuit could
// end up closing on failed conditionals even if the table is healthy
func isFailedConditionalCheck(err error) bool {
	if err == nil {
		return false
	}
	if awsErr, ok := err.(awserr.Error); ok {
		// Just means the item already exists.  Don't fail the circuit!
		if awsErr.Code() == dynamodb.ErrCodeConditionalCheckFailedException {
			return true
		}
	}
	return false
}

// repairRequest holds a request to incr/decr the count of an edge type
type repairRequest struct {
	// where the edge is from
	from graphdbmodel.Node
	// The type of the edge
	edgeKind string
	// Where the edge is to
	to graphdbmodel.Node
	// If we should increment or decrement
	isCreation bool
	// The storage struct that the repair request came from
	s *Storage
}

// retryRequest creates the async request we later put on an SQS queue for when a count change fails and we have to
// retry keeping the count/edge table in sync later
func (r *repairRequest) retryRequest() *interngraphdb.CountRepairRequest {
	return &interngraphdb.CountRepairRequest{
		Edge: &graphdb.Edge{
			From: &graphdb.Node{
				Type: r.from.Type,
				Id:   r.from.ID,
			},
			Type: r.edgeKind,
			To: &graphdb.Node{
				Type: r.to.Type,
				Id:   r.to.ID,
			},
		},
		IsCreation: r.isCreation,
	}
}

// Do executes the repair request
func (r *repairRequest) do() interface{} {
	// We want this count change to work even if the original request fails, otherwise the two get out of sync.  So we
	// have an explicit context here
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*15)
	defer cancel()
	err := r.s.executeRepair(ctx, r.from, r.edgeKind, r.to, r.isCreation)
	if err != nil {
		r.s.BackoffAsyncRequests.BackoffAsync()
		// If the repair fails put the repair on SQS and try it again
		r.s.Logger.LogCtx(ctx, "err", err, "unable to repair counts.  Will add to SQS queue")
		ctx, cancel := context.WithTimeout(context.Background(), time.Second*15)
		defer cancel()
		queueErr := r.s.AsyncQueue.SendMessage(ctx, &interngraphdb.AsyncRequestQueueMessage{
			CountRetries: []*interngraphdb.CountRepairRequest{
				r.retryRequest(),
			},
			// Retry the count change 5 minutes later
		}, time.Minute*5)
		if queueErr != nil {
			// This would be bad and means the count table will be out of sync because not only did the count increment
			// fail, but putting the count on the SQS queue also failed
			r.s.Logger.LogCtx(ctx, "err", err, "from", r.from, "edge_kind", r.edgeKind, "to", r.to, "is_creation", r.isCreation, "unable to queue count repair.  Count lost.")
		}
	}
	return err
}

// QueueRepair was previously used to allow count changes async of edge changes, but now that BlockForCountRepair is
// deprecated, they just go directly to BlockingRepair
func (s *Storage) QueueRepair(from graphdbmodel.Node, edgeKind string, to graphdbmodel.Node, isCreation bool) {
	if s.BlockForCountRepair.Get() {
		if err := s.BlockingRepair(from, edgeKind, to, isCreation); err != nil {
			s.Logger.LogCtx(context.Background(), "err", err)
		}
		return
	}
	req := repairRequest{
		s:          s,
		from:       from,
		edgeKind:   edgeKind,
		to:         to,
		isCreation: isCreation,
	}
	workerpool.OfferOrDo(&s.CountsPool, req.do)
}

func (s *Storage) BlockingRepair(from graphdbmodel.Node, edgeKind string, to graphdbmodel.Node, isCreation bool) error {
	req := repairRequest{
		s:          s,
		from:       from,
		edgeKind:   edgeKind,
		to:         to,
		isCreation: isCreation,
	}
	err := req.do()
	if err == nil {
		return nil
	}
	return err.(error)
}

func (s *Storage) executeRepair(ctx context.Context, from graphdbmodel.Node, edgeKind string, to graphdbmodel.Node, isCreation bool) error {
	repairCtx, can := context.WithTimeout(ctx, time.Minute)
	defer can()
	// This switch controls if we increment or decrement the count value
	if isCreation {
		return s.repairEdgeCreation(repairCtx, from, edgeKind, to)
	}
	return s.repairEdgeDeletion(repairCtx, from, edgeKind, to)
}

// updateCountForEdgeChange is eventually called each time an edge is created or deleted and keeps the count table in
// sync
func (s *Storage) updateCountForEdgeChange(ctx context.Context, from graphdbmodel.Node, edgeKind string, tableName string, circuit *circuit.Circuit, countChange int) error {
	now := time.Now()
	req, res := s.Dynamo.UpdateItemRequest(&dynamodb.UpdateItemInput{
		TableName: &tableName,
		// Return the new count so if two increment operations happen at the same time, we can return the correct new
		// value
		ReturnConsumedCapacity: aws.String("TOTAL"),
		Key: map[string]*dynamodb.AttributeValue{
			dbFrom: {
				S: aws.String(from.Encode()),
			},
			dbEdgeType: {
				S: aws.String(edgeKind),
			},
		},
		ExpressionAttributeNames: map[string]*string{
			"#count":      aws.String(dbCount),
			"#version":    aws.String(dbVersion),
			"#updated_at": aws.String(dbUpdatedAt),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":updated_at": {
				S: aws.String(now.UTC().Format(time.RFC3339)),
			},
			":one": {
				N: aws.String("1"),
			},
			// Depending if the edge is created or deleted, we have to increment or decrement the new count
			":count_change": {
				N: aws.String(strconv.Itoa(countChange)),
			},
		},
		ReturnValues:     aws.String("UPDATED_NEW"),
		UpdateExpression: aws.String("ADD #version :one, #count :count_change SET #updated_at=:updated_at"),
	})

	start := time.Now()
	var circitRunStartDiff time.Duration
	var circitRunEndDiff time.Duration
	retErr := circuit.Run(ctx, func(ctx context.Context) error {
		circitRunStartDiff = time.Since(start)
		req.SetContext(ctx)
		circErr := req.Send()
		circitRunEndDiff = time.Since(start)
		return circErr
	})

	if time.Since(start) > time.Duration(500*time.Millisecond) {
		s.Logger.LogCtx(ctx, "function", "updateCountForEdgeChange", "ddb_request_id", req.RequestID, "table_name", tableName, "startTime", start, "endTimeDiff", time.Since(start), "circitRunStartDiff", circitRunStartDiff, "circitRunEndDiff", circitRunEndDiff)
	}
	s.Stats.TimingDurationC("storage.updateCountForEdgeChange.circitRunStartDiff", circitRunStartDiff, 0.1)
	s.Stats.TimingDurationC("storage.updateCountForEdgeChange.circitRunEndDiff", circitRunEndDiff, 0.1)
	s.Stats.TimingDurationC("storage.updateCountForEdgeChange.total", time.Since(start), 0.1)

	traceDynamoLatency(ctx, time.Since(start), req.RequestID)
	incrConsumedCapacity(ctx, res.ConsumedCapacity)
	if retErr != nil {
		s.Logger.LogCtx(ctx, "function", "UpdateCountForEdgeChange", "ddb_request_id", req.RequestID, "table_name", tableName, "err", retErr, "circuit_time", time.Since(start), "error in DDB circuit call")
		return retErr
	}
	// If we get a count back and it happens to be zero, we should try to clear the current count
	if res.Attributes != nil && res.Attributes[dbCount] != nil && res.Attributes[dbCount].N != nil {
		ints, err := feedsdynamo.AwsInts(res.Attributes, []string{dbCount})
		if err != nil {
			return err
		}
		// NewCount stores back into memcache the new count value after this operation.
		s.CountChange.NewCount(from, edgeKind, ints[dbCount])
		if ints[dbCount] == 0 {
			// Clean up the counts table if we changed the count back to zero
			return s.clearCountsIfZero(ctx, from, edgeKind, tableName, circuit)
		}
	}
	return nil
}

// clearCountsIfZero deletes a row from the count table if the count is zero, since an empty count is the same as
// zero count, and it keeps the DynamoDB table smaller
func (s *Storage) clearCountsIfZero(ctx context.Context, from graphdbmodel.Node, edgeKind string, tableName string, circuit *circuit.Circuit) error {
	req, resp := s.Dynamo.DeleteItemRequest(&dynamodb.DeleteItemInput{
		TableName:              &tableName,
		ReturnConsumedCapacity: aws.String("TOTAL"),
		Key: map[string]*dynamodb.AttributeValue{
			dbFrom: {
				S: aws.String(from.Encode()),
			},
			dbEdgeType: {
				S: aws.String(edgeKind),
			},
		},
		// We should only delete this row if the count is still zero
		ConditionExpression: aws.String("#count = :zero"),
		ExpressionAttributeNames: map[string]*string{
			"#count": aws.String(dbCount),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":zero": {
				N: aws.String("0"),
			},
		},
	})

	start := time.Now()
	circErr := circuit.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		err := req.Send()
		traceDynamoLatency(ctx, time.Since(start), req.RequestID)
		incrConsumedCapacity(ctx, resp.ConsumedCapacity)
		if isFailedConditionalCheck(err) {
			// This isn't a problem, it just means the count isn't still zero
			return nil
		}
		return err
	})
	if circErr != nil {
		s.Logger.LogCtx(ctx, "function", "clearCountsIfZero", "err", circErr, "ddb_request_id", req.RequestID, "table_name", tableName, "circuit_time", time.Since(start), "error in DDB circuit call")
	}
	return circErr
}

// repairEdgeCreation is eventually called when an edge is created
func (s *Storage) repairEdgeCreation(ctx context.Context, from graphdbmodel.Node, edgeKind string, to graphdbmodel.Node) error {
	info := s.Lookup.LookupEdge(edgeKind)
	if info == nil {
		return errors.Errorf("unable to find information for edge type %s", edgeKind)
	}
	reverseInfo := s.Lookup.LookupEdge(info.ReverseName)
	if reverseInfo == nil {
		return errors.Errorf("unable to find information for edge type %s", info.ReverseName)
	}
	// Change the counts in both directions (follow count and followed_by count)
	err1 := s.updateCountForEdgeChange(ctx, from, edgeKind, info.CountTableName, info.Circuits.Count.Write, 1)
	err2 := s.updateCountForEdgeChange(ctx, to, info.ReverseName, reverseInfo.CountTableName, reverseInfo.Circuits.Count.Write, 1)
	return service_common.ConsolidateErrors([]error{err1, err2})
}

// repairEdgeCreation is eventually called when an edge is deleted
func (s *Storage) repairEdgeDeletion(ctx context.Context, from graphdbmodel.Node, edgeKind string, to graphdbmodel.Node) error {
	info := s.Lookup.LookupEdge(edgeKind)
	if info == nil {
		return errors.Errorf("unable to find information for edge type %s", edgeKind)
	}
	reverseInfo := s.Lookup.LookupEdge(info.ReverseName)
	if reverseInfo == nil {
		return errors.Errorf("unable to find information for edge type %s", info.ReverseName)
	}
	// Change the counts in both directions (follow count and followed_by count)
	err1 := s.updateCountForEdgeChange(ctx, from, edgeKind, info.CountTableName, info.Circuits.Count.Write, -1)
	err2 := s.updateCountForEdgeChange(ctx, to, info.ReverseName, reverseInfo.CountTableName, reverseInfo.Circuits.Count.Write, -1)
	return service_common.ConsolidateErrors([]error{err1, err2})
}
