package tablelookup

import (
	"context"
	"fmt"
	"sync"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/graphdbmodel"
	"code.justin.tv/feeds/log"
	"code.justin.tv/hygienic/errors"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/cep21/circuit"
)

const (
	// The node type of GraphDB nodes that store node information
	NodeRegistry = "_node_registry"
	// The node type of GraphDB nodes that store edge information
	EdgeRegistry = "_edge_registry"
)

// NodeStorageClient is where we store node and edge registry information
type NodeStorageClient interface {
	NodeList(ctx context.Context, nodeType string, page graphdbmodel.PagedRequest) (*graphdbmodel.ListNodeResult, error)
	// Note: Create is only used in initializeEdgeRegistry, when a registry didn't already exist
	NodeCreate(ctx context.Context, node graphdbmodel.Node, data *graphdbmodel.DataBag, creationTime time.Time) (*graphdbmodel.LoadedNode, error)
}

type Config struct {
	RefreshDuration        *distconf.Duration
	NodeRegistryTable      *distconf.Str
	NodeRegistryCountTable *distconf.Str
}

func (c *Config) Load(d *distconf.Distconf) error {
	c.RefreshDuration = d.Duration("graphdb.lookup_refresh", time.Minute*10)
	c.NodeRegistryTable = d.Str("graphdb.node_registry_table", "")
	if c.NodeRegistryTable.Get() == "" {
		return errors.New("unable to find node registry table")
	}
	c.NodeRegistryCountTable = d.Str("graphdb.node_registry_count_table", "")
	if c.NodeRegistryTable.Get() == "" {
		return errors.New("unable to find node registry counts table")
	}
	return nil
}

// Lookup contains/allows lookup of node and edge information.  This tells us which DynamoDB table to store data in,
// what the reverse type is, where to store count values, etc.  Each edge and node type supported by GraphDB should
// have an entry here.
//
// It is also slightly recursive since GraphDB uses GraphDB itself to store node and edge information.  I admit this
// can look somewhat complicated, but the overall goal is to store metadata about edge and node types inside GraphDB,
// rather than a JSON file.
type Lookup struct {
	// Client is only used for describing the tables that are stored so we can get the correct index keys.  NodeList
	// is used to actually read the registry data
	Client *dynamodb.DynamoDB
	Logger log.Logger
	// Circuit is used for all Client operations
	Circuit *circuit.Circuit
	// Manager is used to create circuits for each edge/node type.  We have different circuits for each type of edge
	// or node
	Manager     *circuit.Manager
	Config      Config
	Environment string

	UpdateClient NodeStorageClient
	// OnUpdate is called whenever table information is refreshed from DynamoDB
	OnUpdate func(ctx context.Context) error

	// In memory map of all the edge types GraphDB supports and information about each type
	edgeInfo map[string]*EdgeInfo
	// In memory map of all the node types GraphDB supports and information about each type
	nodeInfo map[string]*NodeInfo
	// nodeRegistry is required to bootstrap Lookup and is the NodeInfo data for the registry node type
	nodeRegistry NodeInfo
	// onClose is closed on Close() and tells the fresh loop to stop
	onClose chan struct{}
	once    sync.Once
	mu      sync.RWMutex
}

// setupNodeRegistryTable bootstraps nodeRegistry
func (l *Lookup) setupNodeRegistryTable() error {
	tableCache := cachedDescribeTableResponse{
		Client:  l.Client,
		Circuit: l.Circuit,
	}
	// bootstrap the NodeRegistry information.  From that we can bootstrap EdgeRegistry, then finally all the node and
	// edge types stored by GraphDB
	ret, err := l.newNodeInfo(context.Background(), l.Config.NodeRegistryTable.Get(), l.Config.NodeRegistryCountTable.Get(), NodeRegistry, &tableCache, "ca")
	if err != nil {
		return err
	}
	l.nodeRegistry = *ret
	return nil
}

func (l *Lookup) init() {
	l.once.Do(func() {
		l.onClose = make(chan struct{})
	})
}

// TableNames returns every DynamoDB table GraphDB talks to
func (l *Lookup) TableNames() []string {
	l.mu.Lock()
	defer l.mu.Unlock()
	retMap := make(map[string]struct{}, len(l.edgeInfo)+len(l.nodeInfo))
	retMap[l.Config.NodeRegistryTable.Get()] = struct{}{}
	for _, v := range l.edgeInfo {
		retMap[v.Table.Name] = struct{}{}
	}
	for _, v := range l.nodeInfo {
		retMap[v.Table.Name] = struct{}{}
	}
	ret := make([]string, 0, len(retMap))
	for k := range retMap {
		ret = append(ret, k)
	}
	return ret
}

// AllEdgeTypes are all edge types (including the reverse names) that GraphDB supports
func (l *Lookup) AllEdgeTypes() []string {
	l.mu.RLock()
	defer l.mu.RUnlock()
	ret := make([]string, 0, len(l.edgeInfo))
	for k := range l.edgeInfo {
		ret = append(ret, k)
	}
	return ret
}

// LookupEdge returns the EdgeInfo data for an edge.  It is a fast, in memory lookup and returns nil if we don't know
// the edge name.
func (l *Lookup) LookupEdge(edgeName string) *EdgeInfo {
	l.mu.RLock()
	defer l.mu.RUnlock()
	return l.edgeInfo[edgeName]
}

// LookupEdge returns the NodeInfo data for an edge.  It is a fast, in memory lookup and returns nil if we don't know
// the edge name.
func (l *Lookup) LookupNode(nodeName string) *NodeInfo {
	if nodeName == NodeRegistry {
		return &l.nodeRegistry
	}
	l.mu.RLock()
	defer l.mu.RUnlock()
	return l.nodeInfo[nodeName]
}

// cachedDescribeTableResponse returns dynamo's TableDescription information for tables, but caches previous
// DescribeTableRequest results so we don't have to fetch information for the same table twice.
type cachedDescribeTableResponse struct {
	Client  *dynamodb.DynamoDB
	Circuit *circuit.Circuit
	// cache of TableDescription requests
	cache map[string]*dynamodb.TableDescription
}

// describeTable does a DescribeTableRequest on tableName, but caches the result so we don't have to fetch for the
// same table twice.
func (c *cachedDescribeTableResponse) describeTable(ctx context.Context, tableName string) (*dynamodb.TableDescription, error) {
	if cachedResult, exists := c.cache[tableName]; exists {
		return cachedResult, nil
	}
	req, resp := c.Client.DescribeTableRequest(&dynamodb.DescribeTableInput{
		TableName: &tableName,
	})
	err := c.Circuit.Run(ctx, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	if err != nil {
		return nil, err
	}
	if c.cache == nil {
		c.cache = make(map[string]*dynamodb.TableDescription)
	}
	c.cache[tableName] = resp.Table
	return resp.Table, nil
}

// LoadTableInfo extracts a usable table description (using cache).  It is helpful for figuring out hash and sort keys
// of the table
func (c *cachedDescribeTableResponse) LoadTableInfo(ctx context.Context, tableName string) (TableInfo, error) {
	table, err := c.describeTable(ctx, tableName)
	if err != nil {
		return TableInfo{}, err
	}
	return newTableInfo(table), nil
}

// refreshNodeRegistry reloads all the node types that are supported by GraphDB.  This metadata is stored in the
// NodeRegistry node type.
func (l *Lookup) refreshNodeRegistry(ctx context.Context) error {
	// cache table description calls so we do not get throttled
	tableCache := cachedDescribeTableResponse{
		Client:  l.Client,
		Circuit: l.Circuit,
	}
	cursor := ""
	nodeInfo := make(map[string]*NodeInfo, 12)
	for {
		// Since all the node types supported by GraphDB are stored in GraphDB, we can do a list query and page it.
		// Note this creates a recursive dependency (since we already need to understand the `NodeRegistry` node type)
		// , that is why we bootstrap it inside setupNodeRegistryTable
		results, err := l.UpdateClient.NodeList(ctx, NodeRegistry, graphdbmodel.PagedRequest{
			Cursor: cursor,
		})
		if err != nil {
			return err
		}
		for _, result := range results.Nodes {
			// Load information about the node type from data on the node.  We expect all nodes to have a
			// `node_table` and `count_table` field.  The `sort_key` field is optional and almost always `ca` for
			// created_at
			ni, err := l.newNodeInfo(
				ctx,
				result.LoadedNode.Data.Data.Strings["node_table"],
				result.LoadedNode.Data.Data.Strings["count_table"],
				result.LoadedNode.Node.ID,
				&tableCache,
				result.LoadedNode.Data.Data.Strings["sort_key"])
			if err != nil {
				return err
			}
			if _, exists := nodeInfo[ni.Name]; exists {
				return errors.Errorf("node id exists twice: %s", result.LoadedNode.Node.ID)
			}
			nodeInfo[result.LoadedNode.Node.ID] = ni
		}
		cursor = results.Cursor
		if cursor == "" {
			break
		}
	}

	// This loop is called while the service is live so we need to lock nodeInfo before we mutate it
	l.mu.Lock()
	defer l.mu.Unlock()
	l.nodeInfo = nodeInfo
	return nil
}

func (l *Lookup) newNodeInfo(ctx context.Context, nodeTableName string, countTableName string, nodeName string, tableCache *cachedDescribeTableResponse, sortKey string) (*NodeInfo, error) {
	nodeTable, err := tableCache.LoadTableInfo(ctx, nodeTableName)
	if err != nil {
		return nil, err
	}
	ni := NodeInfo{
		CountedItemTable: CountedItemTable{
			Name:           nodeName,
			SortKey:        sortKey,
			Table:          nodeTable,
			Circuits:       newCircuits(l.Manager, nodeName),
			CountTableName: countTableName,
		},
	}
	// The default sort key for nodes is created at
	if ni.CountedItemTable.SortKey == "" {
		ni.CountedItemTable.SortKey = "ca"
	}
	return &ni, nil
}

// refreshEdgeRegistry reloads all the edge types GraphDB supports. This is called periodically while the service is
// live
func (l *Lookup) refreshEdgeRegistry(ctx context.Context) error {
	tableCache := cachedDescribeTableResponse{
		Client:  l.Client,
		Circuit: l.Circuit,
	}
	edgeInfo := make(map[string]*EdgeInfo, 12)
	cursor := ""
	for {
		// The types of edges supported by GraphDB are stored as nodes of the type `EdgeRegistry`.  Information about
		// node type `EdgeRegistry` is loaded inside refreshNodeRegistry, so refreshNodeRegistry must happen before
		// refreshEdgeRegistry
		results, err := l.UpdateClient.NodeList(ctx, EdgeRegistry, graphdbmodel.PagedRequest{
			Cursor: cursor,
		})
		if err != nil {
			return err
		}
		for _, result := range results.Nodes {
			// Each node in EdgeRegistry results in two nodes inside edgeInfo: the primary edge and reverse edge.
			primaryEdgeTable, err := tableCache.LoadTableInfo(ctx, result.LoadedNode.Data.Data.Strings["edge_table"])
			if err != nil {
				return err
			}
			reverseEdgeTable, err := tableCache.LoadTableInfo(ctx, result.LoadedNode.Data.Data.Strings["edge_table"])
			if err != nil {
				return err
			}
			primaryEdge := EdgeInfo{
				CountedItemTable: CountedItemTable{
					// `sort_key` is optional, but the other fields are required
					SortKey:        result.LoadedNode.Data.Data.Strings["sort_key"],
					Name:           result.LoadedNode.Node.ID,
					Table:          primaryEdgeTable,
					Circuits:       newCircuits(l.Manager, result.LoadedNode.Node.ID),
					CountTableName: result.LoadedNode.Data.Data.Strings["count_table"],
				},
				ReverseName:     result.LoadedNode.Data.Data.Strings["reverse_name"],
				PrimaryEdgeType: true,
			}
			reverseEdge := EdgeInfo{
				CountedItemTable: CountedItemTable{
					SortKey:        result.LoadedNode.Data.Data.Strings["sort_key"],
					Name:           result.LoadedNode.Data.Data.Strings["reverse_name"],
					Table:          reverseEdgeTable,
					Circuits:       newCircuits(l.Manager, result.LoadedNode.Data.Data.Strings["reverse_name"]),
					CountTableName: result.LoadedNode.Data.Data.Strings["reverse_count_table"],
				},
				// The reversed edge type of the reverse edge type, is the primary edge's type.
				ReverseName:     result.LoadedNode.Node.ID,
				PrimaryEdgeType: false,
			}
			// The default sort key for edges are created at
			if primaryEdge.CountedItemTable.SortKey == "" {
				primaryEdge.CountedItemTable.SortKey = "ca"
			}
			if reverseEdge.CountedItemTable.SortKey == "" {
				reverseEdge.CountedItemTable.SortKey = "ca"
			}

			// If no counts table is specified for the reverse edge, use the primary direction's count table
			if reverseEdge.CountedItemTable.CountTableName == "" {
				reverseEdge.CountedItemTable.CountTableName = primaryEdge.CountedItemTable.CountTableName
			}
			if _, exists := edgeInfo[primaryEdge.Name]; exists {
				return errors.Errorf("primary edge already exists: %s", primaryEdge.Name)
			}
			edgeInfo[primaryEdge.Name] = &primaryEdge
			if _, exists := edgeInfo[reverseEdge.Name]; exists {
				return errors.Errorf("reverse edge already exists: %s", reverseEdge.Name)
			}
			edgeInfo[reverseEdge.Name] = &reverseEdge
		}
		cursor = results.Cursor
		if cursor == "" {
			break
		}
	}
	// We have to lock to update edgeInfo since this loop is executed live
	l.mu.Lock()
	defer l.mu.Unlock()
	l.edgeInfo = edgeInfo
	return nil
}

func (l *Lookup) Setup() error {
	l.init()
	if err := l.setupNodeRegistryTable(); err != nil {
		return errors.Wrap(err, "unable to setup node registry")
	}
	ctx := context.Background()
	return l.Refresh(ctx)
}

func (l *Lookup) Start() error {
	l.init()
	for {
		// Refresh the stored edge/node information every RefreshDuration amount of time
		select {
		case <-l.onClose:
			return nil
		case <-time.After(l.Config.RefreshDuration.Get()):
			if err := l.Refresh(context.Background()); err != nil {
				l.Logger.Log("err", err, "unable to refresh tables")
			}
		}
	}
}

func (l *Lookup) Close() error {
	l.init()
	close(l.onClose)
	return nil
}

// Refresh the list of edge/node types supported by GraphDB
func (l *Lookup) Refresh(ctx context.Context) error {
	ctx, cancel := context.WithTimeout(ctx, time.Minute*3)
	defer cancel()
	if err := l.refreshNodeRegistry(ctx); err != nil {
		return errors.Wrap(err, "unable to refresh node registry")
	}
	var shouldRefreshNodeRegistry bool
	var err error
	shouldRefreshNodeRegistry, err = l.initializeEdgeRegistry(ctx)
	if err != nil {
		// If there was some error bootstrapping the registry, bail on the refresh so we do not get in a bad state
		return errors.Wrap(err, "unable to init edge registry")
	} else if shouldRefreshNodeRegistry {
		if err := l.refreshNodeRegistry(ctx); err != nil {
			return errors.Wrap(err, "unable to refresh node registry on second try")
		}
	}
	// Must refreshEdgeRegistry after refreshing node registry, since it uses information populated by the node registry
	if err := l.refreshEdgeRegistry(ctx); err != nil {
		return errors.Wrap(err, "unable to refresh edge registry")
	}

	// OnUpdate is just a callback trigger for after refresh is called
	if err := l.OnUpdate(ctx); err != nil {
		return errors.Wrap(err, "unable to finish on-update call")
	}
	return nil
}

// initializeEdgeRegistry assumes we've bootstrapped the edge registry.  This only needs to execute on a fresh GraphDB
// database, and otherwise is not useful once executed.
func (l *Lookup) initializeEdgeRegistry(ctx context.Context) (bool, error) {
	nodeRegistry := l.LookupNode(NodeRegistry)
	if nodeRegistry == nil {
		return false, errors.New("unable to find node registry")
	}
	edgeRegistry := l.LookupNode(EdgeRegistry)
	if edgeRegistry != nil {
		// edgeRegistry already setup
		return true, nil
	}
	var db graphdbmodel.DataBag
	db.AddString("node_table", nodeRegistry.Table.Name)
	db.AddString("count_table", nodeRegistry.CountTableName)
	_, err := l.UpdateClient.NodeCreate(ctx, graphdbmodel.Node{
		Type: NodeRegistry,
		ID:   EdgeRegistry,
	}, &db, time.Now())
	if err != nil {
		return false, errors.Wrap(err, "unable to create initial edge registry node")
	}
	return false, err
}

// Circuits control breakers for common DynamoDB operations on tables
type Circuits struct {
	Table struct {
		Read  *circuit.Circuit
		Write *circuit.Circuit
	}
	Index struct {
		List *circuit.Circuit
	}
	Count struct {
		Read  *circuit.Circuit
		Write *circuit.Circuit
	}
}

func createOrGetCircuit(m *circuit.Manager, name string, defaultConfig ...circuit.Config) *circuit.Circuit {
	ret, err := m.CreateCircuit(name, defaultConfig...)
	if err == nil {
		return ret
	}
	return m.GetCircuit(name)
}

func newCircuits(m *circuit.Manager, name string) Circuits {
	ret := Circuits{}
	ret.Table.Read = createOrGetCircuit(m, fmt.Sprintf("%s-edge-read", name), circuit.Config{
		Execution: circuit.ExecutionConfig{
			MaxConcurrentRequests: 150,
			Timeout:               time.Second * 2,
		},
	})
	ret.Table.Write = createOrGetCircuit(m, fmt.Sprintf("%s-edge-write", name), circuit.Config{
		Execution: circuit.ExecutionConfig{
			// For batch creation
			MaxConcurrentRequests: 1000,
			Timeout:               time.Second * 8,
		},
	})
	ret.Count.Read = createOrGetCircuit(m, fmt.Sprintf("%s-count-read", name), circuit.Config{
		Execution: circuit.ExecutionConfig{
			MaxConcurrentRequests: 75,
			Timeout:               time.Second * 2,
		},
	})
	ret.Count.Write = createOrGetCircuit(m, fmt.Sprintf("%s-count-write", name), circuit.Config{
		Execution: circuit.ExecutionConfig{
			// For batch creation
			MaxConcurrentRequests: 1000,
			Timeout:               time.Second * 8,
		},
	})
	ret.Index.List = createOrGetCircuit(m, fmt.Sprintf("%s-edge-list", name), circuit.Config{
		Execution: circuit.ExecutionConfig{
			MaxConcurrentRequests: 50,
			Timeout:               time.Second * 4,
		},
	})
	return ret
}

// TableInfo tells us about a DynamoDB table
type TableInfo struct {
	Description *dynamodb.TableDescription
	Name        string
}

// DefaultSortKeyValue returns a sortkey we can use for a query.  Some DynamoDB operations require us to have a sort
// key inside them.  If none is given, we have to create a default sort key.  To create a default sort key, we need
// to know the attribute type of the sort key for that table.  DynamoDB supports three sort key types: string, number,
// and binary.  This returns a default `empty` sort key.
func (t *TableInfo) DefaultSortKeyValue(sortKeyName string) *dynamodb.AttributeValue {
	for _, d := range t.Description.AttributeDefinitions {
		if *d.AttributeName == sortKeyName {
			if *d.AttributeType == "S" {
				return &dynamodb.AttributeValue{
					S: aws.String(""),
				}
			}
			if *d.AttributeType == "N" {
				return &dynamodb.AttributeValue{
					N: aws.String("0"),
				}
			}
			if *d.AttributeType == "B" {
				return &dynamodb.AttributeValue{
					B: []byte(""),
				}
			}
			// This should never happen since sort keys are of only three types
			panic("Logic error: unknown attribute type: " + *d.AttributeType)
		}
	}
	// This should probably never happen, but means we couldn't find a sort key
	return nil
}

func newTableInfo(description *dynamodb.TableDescription) TableInfo {
	ret := TableInfo{
		Description: description,
		Name:        *description.TableName,
	}
	return ret
}

// EdgeInfo is metadata information we store about edges
type EdgeInfo struct {
	CountedItemTable
	ReverseName     string
	PrimaryEdgeType bool
}

// CountedItemTable is common metadata between nodes and edges
type CountedItemTable struct {
	Name           string
	Circuits       Circuits
	SortKey        string
	Table          TableInfo
	CountTableName string
}

// NodeInfo is all the metadata information we store about nodes
type NodeInfo struct {
	CountedItemTable
}
