package main

import (
	"context"
	"encoding/json"
	"expvar"
	"fmt"
	"io/ioutil"
	"math/rand"
	"net"
	"net/http"
	"os"
	"os/signal"
	"strings"
	"sync"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/accesslog"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/admin/repair"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/api/httpapi"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/api/twirpserver"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/async"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/cache"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/cacheinvalidation"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/edgechange"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/oldapi/datastore"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/oldapi/datastore/twirpdatastore"
	sns_client "code.justin.tv/feeds/graphdb/cmd/graphdb/internal/sns"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage/complexstorage"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/storage/tablelookup"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/twirphooks"
	"code.justin.tv/feeds/graphdb/proto/datastorerpc"
	"code.justin.tv/feeds/graphdb/proto/dynamoevent"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/graphdb/proto/graphdbadmin"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/hygienic/dynamocursor"
	"code.justin.tv/hygienic/elastimemcache"
	memcacheVerifier "code.justin.tv/hygienic/elastimemcache/memcache"
	"code.justin.tv/hygienic/gomemcache/memcache"
	"code.justin.tv/hygienic/messagebatch"
	"code.justin.tv/hygienic/messagebatch/ext/cwlogevent"
	"code.justin.tv/hygienic/objectcache"
	"code.justin.tv/hygienic/objectcache/gomemcache"
	"code.justin.tv/hygienic/rlimit"
	"code.justin.tv/hygienic/sqsprocessor"
	"code.justin.tv/hygienic/statsdsender"
	"code.justin.tv/hygienic/workerpool"
	"code.justin.tv/web/cohesion/associations"
	"github.com/aws/aws-sdk-go/service/cloudwatchlogs"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/cep21/circuit"
	"github.com/cep21/circuit/metrics/responsetimeslo"
	"golang.org/x/sync/semaphore"
)

const (
	teamName    = "feeds"
	serviceName = "graphdb"
)

// CodeVersion is set by build script
var CodeVersion string

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		SLOConfig: map[string]responsetimeslo.Config{
			"memcache": {
				MaximumHealthyTime: time.Millisecond * 50,
			},
			"async.queue.receive": {
				MaximumHealthyTime: time.Minute * 2,
			},
			"accesslog_circuit": {
				MaximumHealthyTime: time.Second * 30,
			},
			"follows.edge.list": {
				MaximumHealthyTime: time.Millisecond * 500,
			},
			"followed_by.edge.list": {
				MaximumHealthyTime: time.Millisecond * 500,
			},
			"cache_invalidation_queue.receive": {
				MaximumHealthyTime:  time.Second * 7,
			},
		},
		CodeVersion: CodeVersion,
	},
}

type service struct {
	osExit                  func(code int)
	onListen                func(net.Addr)
	schemaManager           *associations.Schema
	sigChan                 chan os.Signal
	serviceCommon           service_common.ServiceCommon
	CursorFactory           dynamocursor.Factory
	edgeChangeProcessor     *sqsprocessor.SQSProcessor
	lookup                  tablelookup.Lookup
	storageLayer            storage.Storage
	runner                  service_common.ServiceRunner
	twirpServer             twirpserver.Server
	nodeStorage             storage.NodeStorage
	httpServer              httpapi.Server
	defaultCache            cache.ObjCache
	moderationCache         cache.ObjCache
	accessLog               accesslog.AccessLog
	dynamoClient            *dynamodb.DynamoDB
	dynamoEventsTwirpServer twirpserver.DynamoEventsTwirpServer
	snsClient               sns_client.SNSClient
	configs                 struct {
		Config                  Config
		serverConfig            httpapi.Config
		cacheConfig             cache.Config
		cwconfig                cwlogevent.Config
		queueConfig             async.QueueConfig
		loaderConfig            tablelookup.Config
		processorConfig         async.ProcessorConfig
		edgechangeConfig        edgechange.Config
		snsConfig               sns_client.SNSConfig
		cacheInvalidationConfig cacheinvalidation.QueueConfig
	}
}

type Config struct {
	StorageFilename           *distconf.Str
	CacheFilename             *distconf.Str
	CountQueueSize            *distconf.Int
	CountWorkerRoutines       *distconf.Int
	CacheDirtyQueueSize       *distconf.Int
	CacheDirtyWorkerRoutines  *distconf.Int
	BlockForCount             *distconf.Bool
	MultiServerQueueSize      *distconf.Int
	MultiServerWorkerRoutines *distconf.Int
}

func (c *Config) Load(d *distconf.Distconf) error {
	c.StorageFilename = d.Str("graphdb.storage_config", "./cmd/graphdb/internal/storage/storage.json,./storage.json")
	c.CacheFilename = d.Str("graphdb.cache_config", "./cmd/graphdb/internal/cache/cache.json,./cache.json")

	c.CountQueueSize = d.Int("graphdb.count_queue_size", 4096)
	c.CountWorkerRoutines = d.Int("graphdb.count_worker_routines", 32)
	c.BlockForCount = d.Bool("graphdb.block_for_counts", false)

	c.CacheDirtyQueueSize = d.Int("graphdb.dirty.queue_size", 4096)
	c.CacheDirtyWorkerRoutines = d.Int("graphdb.dirty.worker_routines", 32)

	c.MultiServerQueueSize = d.Int("graphdb.multi.queue_size", 4096)
	c.MultiServerWorkerRoutines = d.Int("graphdb.multi.worker_routines", 32)
	return nil
}

// readConfigFiles accepts a list of files to check for config.
// readConfigFiles returns the contents of the first file location that can be read successfully.
func readConfigFiles(files []string) (string, error) {
	var bs []byte
	var err error
	for _, part := range files {
		bs, err = ioutil.ReadFile(part)
		if err == nil {
			return string(bs), nil
		}
	}
	return "", err
}

func (f *service) readStorageConfig() (string, error) {
	files := strings.Split(f.configs.Config.StorageFilename.Get(), ",")
	return readConfigFiles(files)
}

func (f *service) readCacheConfig() (string, error) {
	files := strings.Split(f.configs.Config.CacheFilename.Get(), ",")
	return readConfigFiles(files)
}

func mergeSchemas(names ...string) (*associations.Schema, error) {
	ret := associations.Schema{
		AssocSchema:  make(map[string]associations.AssocKind),
		EntitySchema: make(map[string]associations.EntityKind),
	}
	for _, name := range names {
		s, err := associations.NewSchemaManager(name)
		if err != nil {
			return nil, err
		}
		for k, as := range s.AssocSchema {
			if _, exists := ret.AssocSchema[k]; exists {
				return nil, errors.Errorf("already see schema name %s", k)
			}
			ret.AssocSchema[k] = as
		}
		for k, v := range s.EntitySchema {
			ret.EntitySchema[k] = v
		}
		ret.EmptyEntity = s.EmptyEntity
		ret.UnknownAssocKind = s.UnknownAssocKind
		ret.UnknownEntityKind = s.UnknownEntityKind
	}
	return &ret, nil
}

// To fix races during unit tests, and since we have to set a global variable for old version, use once during the set
var setSchemaManagerOnce sync.Once

// This hacky code is because of old cohesion's use of singletons
func setSchemas(extras ...string) {
	setSchemaManagerOnce.Do(func() {
		a := append(extras, []string{"friends", "follows", "chat"}...)
		schemaManager, err := mergeSchemas(a...)
		if err != nil {
			panic(err)
		}
		associations.SchemaManager = schemaManager
	})
}

func (f *service) setup() error {
	f.configs.cwconfig.Prefix = "graphdb.access_logs."
	if err := f.serviceCommon.Setup(); err != nil {
		return errors.Wrap(err, "unable to setup service common")
	}
	if err := service_common.LoadConfigs(
		f.serviceCommon.Config,
		&f.configs.Config,
		&f.configs.cacheConfig,
		&f.configs.cwconfig,
		&f.configs.queueConfig,
		&f.configs.processorConfig,
		&f.configs.edgechangeConfig,
		&f.configs.loaderConfig,
		&f.configs.serverConfig,
		&f.configs.snsConfig,
		&f.configs.cacheInvalidationConfig); err != nil {
		return errors.Wrap(err, "unable to load configs")
	}
	var err error

	// This hacky code is because of old cohesion's use of singletons
	setSchemas()
	f.schemaManager = associations.SchemaManager

	// Canary uses the production tables.  Fall back to those names.
	env := f.serviceCommon.Config.Str("consul.fallback_env", "").Get()
	if env == "" {
		env = f.serviceCommon.Environment
	}
	defaultCache, err := f.setupItemCache(f.configs.cacheConfig.DefaultConfigServer.Get(), "follow")
	if err != nil {
		return errors.Wrap(err, "unable to setup follow cache")
	}
	f.defaultCache = defaultCache

	moderationCache, err := f.setupItemCache(f.configs.cacheConfig.ModerationConfigServer.Get(), "mods")
	if err != nil {
		return errors.Wrap(err, "unable to setup mods cache")
	}
	f.moderationCache = moderationCache

	session, awsConf := service_common.CreateAWSSessionWithHttpClient(f.serviceCommon.Config, &http.Client{
		Transport: http.RoundTripper(&http.Transport{
			MaxIdleConns:        100,
			MaxIdleConnsPerHost: 100,
		}),
	})
	f.dynamoClient = dynamodb.New(session, awsConf...)

	snsBaseClient := sns.New(session, awsConf...)
	f.snsClient = sns_client.SNSClient{
		BaseClient:  snsBaseClient,
		SnsConfig:   f.configs.snsConfig,
		MarshalJSON: json.Marshal,
		Log:         f.serviceCommon.Log,
	}

	f.accessLog = accesslog.AccessLog{
		Rand: rand.New(rand.NewSource(time.Now().UnixNano())),
		CloudwatchLogBatcher: cwlogevent.CloudwatchLogBatcher{
			Batcher: messagebatch.Batcher{
				Log:    f.serviceCommon.Log,
				Events: make(chan interface{}, 15000),
			},
			Config: &f.configs.cwconfig,
			Client: cloudwatchlogs.New(session, awsConf...),
			Circuit: f.serviceCommon.Circuit.MustCreateCircuit(
				"accesslog_circuit",
				circuit.Config{
					Execution: circuit.ExecutionConfig{
						Timeout: time.Second * 9,
					},
				}),
		},
	}
	f.lookup = tablelookup.Lookup{
		Logger:      f.serviceCommon.Log,
		Client:      f.dynamoClient,
		Circuit:     f.serviceCommon.Circuit.MustCreateCircuit("registry-lookup"),
		Manager:     &f.serviceCommon.Circuit,
		Config:      f.configs.loaderConfig,
		Environment: env,
		OnUpdate:    f.CursorFactory.Refresh,
	}

	f.nodeStorage = storage.NodeStorage{
		Dynamo: f.dynamoClient,
		Lookup: &f.lookup,
		Logger: f.serviceCommon.Log,
		CountsPool: workerpool.Pool{
			FuturesChannel: make(chan *workerpool.Future, f.configs.Config.CountQueueSize.Get()),
		},
		CursorFactory:       &f.CursorFactory,
		BlockForCountRepair: f.configs.Config.BlockForCount,
	}

	f.CursorFactory.Client = f.dynamoClient
	f.storageLayer = storage.Storage{
		Stats: &statsdsender.ErrorlessStatSender{
			StatSender: f.serviceCommon.Statsd.NewSubStatter("edge.storage"),
		},
		CursorFactory: &f.CursorFactory,
		Logger:        f.serviceCommon.Log,
		Dynamo:        f.dynamoClient,
		Lookup:        &f.lookup,
		CountsPool: workerpool.Pool{
			FuturesChannel: make(chan *workerpool.Future, f.configs.Config.CountQueueSize.Get()),
		},
		BlockForCountRepair: f.configs.Config.BlockForCount,
	}
	f.storageLayer.CountsPool.SetNumberOfWorkers(int(f.configs.Config.CountWorkerRoutines.Get()))

	f.CursorFactory = dynamocursor.Factory{
		Client: f.dynamoClient,
		TableNames: func() []string {
			return f.lookup.TableNames()
		},
		Circuit: f.serviceCommon.Circuit.MustCreateCircuit("dynamocursor-factory", circuit.Config{
			Execution: circuit.ExecutionConfig{
				Timeout: time.Second * 10,
			},
		}),
		Log: f.serviceCommon.Log,
	}
	if err := f.CursorFactory.Refresh(context.Background()); err != nil {
		return errors.Wrap(err, "unable to refresh cursor factory")
	}

	f.lookup.UpdateClient = &f.nodeStorage
	if err := f.lookup.Setup(); err != nil {
		return errors.Wrap(err, "unable to setup lookup cache")
	}
	return nil
}

func (f *service) injectOldV1(complexLayer *complexstorage.Storage) {
	// TODO: Setup SchemaManager
	datastoreImpl := &datastore.Datastore{
		Storage: complexLayer,
		Lookup:  complexLayer.Lookup,
	}

	s := &twirpdatastore.Server{
		Store:   datastoreImpl,
		Statter: f.serviceCommon.Statsd.NewSubStatter("datastore"),
		Log:     f.serviceCommon.Log,
	}

	f.httpServer.DatastoreWriteTwirp = datastorerpc.NewWriterServer(s, twirphooks.TwirpHooks(f.serviceCommon.Hostname, f.serviceCommon.Statsd, &f.accessLog))
	f.httpServer.DynamoEvents = dynamoevent.NewDynamoUpdatesServer(&f.dynamoEventsTwirpServer, twirphooks.TwirpHooks(f.serviceCommon.Hostname, f.serviceCommon.Statsd, &f.accessLog))
	f.httpServer.DatastoreReadTwirp = datastorerpc.NewReaderServer(s, twirphooks.TwirpHooks(f.serviceCommon.Hostname, f.serviceCommon.Statsd, &f.accessLog))
}

func (f *service) setupItemCache(cfgServer string, itemType string) (*objectcache.Migration, error) {
	if cfgServer == "" {
		return nil, errors.New("unable to find memcache config_server for " + itemType)
	}
	cacheSelector := elastimemcache.Elasticache{
		CfgServer:     cfgServer,
		PollInterval:  time.Second * 30,
		UpdateTimeout: time.Second * 10,
		FetchErrorsCallback: func(err error) {
			f.serviceCommon.Log.Log("err", err, "unable to fetch elasticache servers")
		},
		HealthVerifier: &memcacheVerifier.Verifier{
			OnErr: func(err error) {
				f.serviceCommon.Log.Log("err", err, "problem verifying memcache server list")
			},
		},
	}
	if err := cacheSelector.Init(context.Background()); err != nil {
		return nil, err
	}
	memcacheClient := memcache.NewFromSelector(&cacheSelector)
	memcacheClient.MaxIdleConns = 100

	memcacheStats := &statsdsender.ErrorlessStatSender{
		StatSender: f.serviceCommon.Statsd.NewSubStatter("memcache_" + itemType),
	}
	keyTTL := func(string) time.Duration {
		return time.Hour * 96
	}
	memcacheCircuit := f.serviceCommon.Circuit.MustCreateCircuit("memcache_"+itemType, circuit.Config{
		Execution: circuit.ExecutionConfig{
			Timeout:               time.Millisecond * 600,
			MaxConcurrentRequests: 500,
		},
	})
	memcNew := &objectcache.ObjectCache{
		ClientPool: &gomemcache.CacheClientPool{
			Client:  memcacheClient,
			Circuit: memcacheCircuit,
		},
		ValueDecoder: objectcache.GZipDecoder(),
		Stats:        memcacheStats,
		KeyPrefix:    "c3:", // Move to key c3 now that we use gzip encoding
		KeyTTL:       keyTTL,
	}
	cacheKey := fmt.Sprintf("cachetest:%d", rand.Int())
	if err := memcNew.ForceCached(context.Background(), cacheKey, []byte("test")); err != nil {
		return nil, errors.Wrap(err, "unable to verify memcache client set")
	}
	var item []byte
	if err := memcNew.Cached(context.Background(), cacheKey, func() (interface{}, error) {
		return nil, errors.Errorf("%s: should be in the cache", cacheKey)
	}, &item); err != nil {
		return nil, errors.Wrap(err, "unable to verify memcache client get")
	}

	if string(item) != "test" {
		return nil, errors.New("unable to verify memcache client set-get")
	}

	f.serviceCommon.ExpvarHandler.Exported["elasticache_"+itemType] = expvar.Func(func() interface{} {
		return &cacheSelector
	})
	f.serviceCommon.ExpvarHandler.Exported["elastic_mc_client_"+itemType] = expvar.Func(func() interface{} {
		return memcacheClient.ClientStats()
	})

	memcOld := gomemcache.NewGomemcacheObjectPool(memcacheClient, keyTTL, "c2:", memcacheStats, memcacheCircuit)

	migrated := &objectcache.Migration{
		From: memcOld,
		To:   memcNew,
	}

	return migrated, nil
}

func (f *service) injectSQS(sqsClient *sqs.SQS, graphdb *twirpserver.Server) {
	stats := &statsdsender.ErrorlessStatSender{
		StatSender: f.serviceCommon.Statsd.NewSubStatter("edgechange"),
	}
	var circuits edgechange.QueueCircuits
	circuits.Circuits(&f.serviceCommon.Circuit)
	proc, sender := edgechange.New(sqsClient, f.serviceCommon.Log, circuits, f.configs.edgechangeConfig, graphdb, stats)
	graphdb.UpdateAllTypesMsgSender = sender
	f.edgeChangeProcessor = proc
}

func (f *service) inject() {
	session, awsConf := service_common.CreateAWSSession(f.serviceCommon.Config)
	sqsClient := sqs.New(session, awsConf...)
	f.CursorFactory.Client = f.dynamoClient

	memcacheLayer := &cache.MemcacheCache{
		DefaultCache: f.defaultCache,
		Caches: map[string]cache.ObjCache{
			"moderated_by": f.moderationCache,
			"moderates":    f.moderationCache,
			"vip_of":       f.moderationCache,
			"has_vip":      f.moderationCache,
		},
		Config: &f.configs.cacheConfig,
		Stats: &statsdsender.ErrorlessStatSender{
			StatSender: f.serviceCommon.Statsd.NewSubStatter("cache"),
		},
		Storage: &f.storageLayer,
		Lookup:  &f.lookup,
		AsyncDirtyPool: workerpool.Pool{
			FuturesChannel: make(chan *workerpool.Future, f.configs.Config.CacheDirtyQueueSize.Get()),
		},
		Log:           f.serviceCommon.Log,
		BlockForDirty: f.configs.Config.BlockForCount,
		SNSClient:     &f.snsClient,
	}
	memcacheLayer.AsyncDirtyPool.SetNumberOfWorkers(int(f.configs.Config.CacheDirtyWorkerRoutines.Get()))
	f.storageLayer.CountChange = memcacheLayer

	complexLayer := &complexstorage.Storage{
		Client: memcacheLayer,
		Lookup: f.storageLayer.Lookup,
		Log:    f.serviceCommon.Log,
		// Just hard code only 2 at once allowed
		Semaphore: semaphore.NewWeighted(2),
		Stats: &statsdsender.ErrorlessStatSender{
			StatSender: f.serviceCommon.Statsd.NewSubStatter("bulk"),
		},
	}

	f.twirpServer = twirpserver.Server{
		Storage:     complexLayer,
		NodeStorage: &f.nodeStorage,
		AsyncMultiPool: workerpool.Pool{
			FuturesChannel: make(chan *workerpool.Future, f.configs.Config.MultiServerQueueSize.Get()),
		},
		Stats: &statsdsender.ErrorlessStatSender{
			StatSender: f.serviceCommon.Statsd.NewSubStatter("graphdbapi"),
		},
		SNSClient: &f.snsClient,
	}
	f.injectSQS(sqsClient, &f.twirpServer)
	f.twirpServer.NodeStorage.CountsPool.SetNumberOfWorkers(int(f.configs.Config.CountWorkerRoutines.Get()))
	f.twirpServer.AsyncMultiPool.SetNumberOfWorkers(int(f.configs.Config.MultiServerWorkerRoutines.Get()))

	// TODO: Deeper access logs
	cohesionServer := graphdb.NewGraphDBServer(&f.twirpServer, twirphooks.TwirpHooks(f.serviceCommon.Hostname, f.serviceCommon.Statsd, &f.accessLog))
	p := async.Processor{
		Hostname: f.serviceCommon.ConfigCommon.Hostname,
		Queue: &async.Queue{
			Stats: &statsdsender.ErrorlessStatSender{
				StatSender: f.serviceCommon.Statsd.NewSubStatter("async_queue"),
			},
			SQS:         sqsClient,
			QueueConfig: &f.configs.queueConfig,
			Log:         f.serviceCommon.Log,
			Storage:     &f.twirpServer,
			Circuits: async.QueueCircuits{
				Receive: f.serviceCommon.Circuit.MustCreateCircuit("async.queue.receive", circuit.Config{
					Execution: circuit.ExecutionConfig{
						Timeout: time.Minute,
					},
				}),
				Delete: f.serviceCommon.Circuit.MustCreateCircuit("async.queue.delete"),
				Send: f.serviceCommon.Circuit.MustCreateCircuit("async.queue.send", circuit.Config{
					// These messages are very big and sometimes take a while to send
					Execution: circuit.ExecutionConfig{
						Timeout:               time.Second * 15,
						MaxConcurrentRequests: 40,
					},
				}),
			},
		},
		AccessLog:       &f.accessLog,
		ProcessorConfig: &f.configs.processorConfig,
		Log:             f.serviceCommon.Log,
		Repair:          &f.storageLayer,
		Client:          &f.twirpServer,
		Stats: statsdsender.ErrorlessStatSender{
			StatSender: f.serviceCommon.Statsd.NewSubStatter("async"),
		},
	}
	cache_invalidation := cacheinvalidation.Poller{
		Queue: &cacheinvalidation.Queue{
			SQS: sqsClient,
			Stats: &statsdsender.ErrorlessStatSender{
				StatSender: f.serviceCommon.Statsd.NewSubStatter("cache_invalidation_queue"),
			},
			Log: f.serviceCommon.Log,
			Circuits: cacheinvalidation.QueueCircuits{
				Receive: f.serviceCommon.Circuit.MustCreateCircuit("cache_invalidation_queue.receive", circuit.Config{
					Execution: circuit.ExecutionConfig{
						Timeout: time.Minute,
					},
				}),
				Delete: f.serviceCommon.Circuit.MustCreateCircuit("cache_invalidation_queue.delete"),
			},
			QueueConfig: &f.configs.cacheInvalidationConfig,
		},
		Log: f.serviceCommon.Log,
		Stats: &statsdsender.ErrorlessStatSender{
			StatSender: f.serviceCommon.Statsd.NewSubStatter("cache_invalidation"),
		},
		Cache: memcacheLayer,
	}
	f.storageLayer.BackoffAsyncRequests = &p
	f.twirpServer.Async = p.Queue
	f.storageLayer.AsyncQueue = p.Queue
	f.twirpServer.NodeStorage.AsyncQueue = p.Queue

	repairEdgeCounter := &repair.Repair{
		Log:          f.serviceCommon.Log,
		ExistingData: memcacheLayer,
		Stats: &statsdsender.ErrorlessStatSender{
			StatSender: f.serviceCommon.Statsd.NewSubStatter("repair"),
		},
	}

	admin := &twirpserver.Admin{
		RepairEdgeCounter: repairEdgeCounter,
	}

	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.serviceCommon, &f.storageLayer, &f.httpServer, &f.accessLog, &p, memcacheLayer, &f.CursorFactory, &f.lookup, f.edgeChangeProcessor, &cache_invalidation,
		},
		SigChan:      f.sigChan,
		SignalNotify: signal.Notify,
	}
	// Note: injectOldV1 must happen after this assignment
	f.httpServer = httpapi.Server{
		Log:        f.serviceCommon.Log,
		Config:     &f.configs.serverConfig,
		Twirp:      cohesionServer,
		AdminTwirp: graphdbadmin.NewGraphDBAdminServer(admin, twirphooks.TwirpHooks(f.serviceCommon.Hostname, f.serviceCommon.Statsd, &f.accessLog)),
		OnListen:   f.onListen,
	}
	f.injectOldV1(complexLayer)
	f.serviceCommon.ExpvarHandler.Exported["cohesion_schema"] = expvar.Func(func() interface{} {
		return associations.SchemaManager
	})
	f.serviceCommon.ExpvarHandler.Exported["async_dirty_pool"] = memcacheLayer.AsyncDirtyPool.Var()
	f.serviceCommon.ExpvarHandler.Exported["async_count_pool"] = memcacheLayer.AsyncDirtyPool.Var()
	f.serviceCommon.ExpvarHandler.Exported["rlimit"] = rlimit.Var()
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		f.osExit(1)
		return
	}
	f.inject()
	if err := f.runner.Execute(); err != nil {
		f.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
