// +build integration

package main

import (
	"context"
	"fmt"
	"math/rand"
	"net"
	"net/http"
	"os"
	"strconv"
	"sync/atomic"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/graphdb/cmd/graphdb/internal/oldapi/conversion"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	. "github.com/smartystreets/goconvey/convey"
)

func init() {
	setSchemas("test_edge")
}

var startRandomInt = rand.NewSource(time.Now().UnixNano()).Int63()

func randomUserEntity() *graphdb.Node {
	return &graphdb.Node{
		Type: "test_node",
		Id:   strconv.FormatInt(atomic.AddInt64(&startRandomInt, 1), 10),
	}
}

func TestIntegration_Nodes(t *testing.T) {
	//t.Parallel()
	fmt.Println("A")
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	fmt.Println("B")
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()
	Convey("create a node", t, func(c C) {
		idToMake := strconv.FormatInt(atomic.AddInt64(&startRandomInt, 1), 10)
		wasCreated := false
		c.Reset(func() {
			resp, err := client.NodeDelete(ctx, &graphdb.NodeDeleteRequest{
				Node: &graphdb.Node{
					Type: "test_node",
					Id:   idToMake,
				},
			})
			c.So(err, ShouldBeNil)
			if wasCreated {
				c.So(resp.Node, ShouldNotBeNil)
			} else {
				c.So(resp.Node, ShouldBeNil)
			}
		})
		resp, err := client.NodeCreate(ctx, &graphdb.NodeCreateRequest{
			Node: &graphdb.Node{
				Type: "test_node",
				Id:   idToMake,
			},
			Data: &graphdb.DataBag{
				Strings: map[string]string{
					"name": "john",
				},
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Node, ShouldNotBeNil)
		wasCreated = true
		c.So(resp.Node.Node.Type, ShouldEqual, "test_node")
		c.So(resp.Node.Data.Version, ShouldEqual, 0)

		// Then fetch it back
		resp2, err := client.NodeGet(ctx, &graphdb.NodeGetRequest{
			Node: &graphdb.Node{
				Type: "test_node",
				Id:   idToMake,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp2.Node, ShouldNotBeNil)
		c.So(resp2.Node.Node.Type, ShouldEqual, "test_node")
		c.So(resp2.Node.Data.Version, ShouldEqual, 0)

		// Should see a non zero count
		resp3, err := client.NodeCount(ctx, &graphdb.NodeCountRequest{
			NodeType: "test_node",
		})
		c.So(err, ShouldBeNil)
		c.So(resp3.Count, ShouldBeGreaterThan, 0)

		// Should be able to update
		resp4, err := client.NodeUpdate(ctx, &graphdb.NodeUpdateRequest{
			Node: &graphdb.Node{
				Type: "test_node",
				Id:   idToMake,
			},
			Data: &graphdb.DataBag{
				Strings: map[string]string{
					"name": "john",
				},
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp4.Node.Data.Version, ShouldEqual, 1)

		// Should see it in a list
		searchForNodeID(ctx, c, client, idToMake)
	})
}

func searchForNodeID(ctx context.Context, c C, client graphdb.GraphDB, expectedId string) {
	cursor := ""
	for {
		resp, err := client.NodeList(ctx, &graphdb.NodeListRequest{
			NodeType: "test_node",
			Page: &graphdb.PagedRequest{
				Cursor: cursor,
			},
		})
		c.So(err, ShouldBeNil)
		for _, i := range resp.Nodes {
			if i.Node.Node.Id == expectedId {
				return
			}
		}
		cursor := resp.Cursor
		if cursor == "" {
			panic("Expected to eventually find the item")
		}
	}
}

func TestIntegration_NotFound(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()
	Convey("Empty get", t, func(c C) {
		resp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: &graphdb.Edge{
				From: randomUserEntity(),
				To:   randomUserEntity(),
				Type: "test_edge",
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge, ShouldBeNil)
	})

	Convey("Empty count", t, func(c C) {
		resp, err := client.EdgeCount(ctx, &graphdb.EdgeCountRequest{
			From:     randomUserEntity(),
			EdgeType: "test_edge",
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Count, ShouldEqual, 0)
	})

	Convey("Empty list", t, func(c C) {
		resp, err := client.EdgeList(ctx, &graphdb.EdgeListRequest{
			From:     randomUserEntity(),
			EdgeType: "test_edge",
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Cursor, ShouldEqual, "")
		c.So(len(resp.Edges), ShouldEqual, 0)
	})

	Convey("Empty delete", t, func(c C) {
		resp, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
			Edge: &graphdb.Edge{
				From: randomUserEntity(),
				Type: "test_edge",
				To:   randomUserEntity(),
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge, ShouldBeNil)
	})

	Convey("Empty update", t, func(c C) {
		resp, err := client.EdgeUpdate(ctx, &graphdb.EdgeUpdateRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: randomUserEntity(),
				To:   randomUserEntity(),
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge, ShouldBeNil)
	})

	Convey("Empty change type", t, func(c C) {
		resp, err := client.EdgeUpdateType(ctx, &graphdb.EdgeUpdateTypeRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: randomUserEntity(),
				To:   randomUserEntity(),
			},
			NewType: "test_edge_reverse",
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge, ShouldBeNil)
	})

	Convey("Empty node get", t, func(c C) {
		resp, err := client.NodeGet(ctx, &graphdb.NodeGetRequest{
			Node: &graphdb.Node{
				Type: "test_node",
				Id:   "1",
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Node, ShouldBeNil)
	})

	Convey("Any node count", t, func(c C) {
		_, err := client.NodeCount(ctx, &graphdb.NodeCountRequest{
			NodeType: "test_node",
		})
		c.So(err, ShouldBeNil)
	})

	Convey("Any node list", t, func(c C) {
		_, err := client.NodeList(ctx, &graphdb.NodeListRequest{
			NodeType: "test_node",
		})
		c.So(err, ShouldBeNil)
	})

	Convey("Empty node delete", t, func(c C) {
		resp, err := client.NodeDelete(ctx, &graphdb.NodeDeleteRequest{
			Node: &graphdb.Node{
				Type: "test_node",
				Id:   "1",
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Node, ShouldBeNil)
	})

	Convey("Empty node update", t, func(c C) {
		resp, err := client.NodeUpdate(ctx, &graphdb.NodeUpdateRequest{
			Node: &graphdb.Node{
				Type: "test_node",
				Id:   "1",
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Node, ShouldBeNil)
	})
}

func TestIntegration_Moderates(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()
	Convey("Moderation creation works", t, func(c C) {
		from := randomUserEntity()
		to := randomUserEntity()
		resp, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: from,
				Type: "moderates",
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge.Edge.Type, ShouldEqual, "moderates")
		c.So(resp.Edge.Edge.To.Type, ShouldEqual, to.Type)
		c.So(resp.Edge.Edge.To.Id, ShouldEqual, to.Id)

		resp2, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: &graphdb.Edge{
				From: to,
				Type: "moderated_by",
				To:   from,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp2.Edge.Edge.Type, ShouldEqual, "moderated_by")
		c.So(resp2.Edge.Edge.To.Type, ShouldEqual, from.Type)
		c.So(resp2.Edge.Edge.To.Id, ShouldEqual, from.Id)
	})
}

func TestIntegration_Vip(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()
	Convey("vip creation works", t, func(c C) {
		from := randomUserEntity()
		to := randomUserEntity()
		resp, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: from,
				Type: "vip_of",
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge.Edge.Type, ShouldEqual, "vip_of")
		c.So(resp.Edge.Edge.To.Type, ShouldEqual, to.Type)
		c.So(resp.Edge.Edge.To.Id, ShouldEqual, to.Id)

		resp2, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: &graphdb.Edge{
				From: to,
				Type: "has_vip",
				To:   from,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp2.Edge.Edge.Type, ShouldEqual, "has_vip")
		c.So(resp2.Edge.Edge.To.Type, ShouldEqual, from.Type)
		c.So(resp2.Edge.Edge.To.Id, ShouldEqual, from.Id)
	})
}

func TestIntegration_CreateBackwards(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()
	Convey("double create works", t, func(c C) {
		from := randomUserEntity()
		to := randomUserEntity()
		resp, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge_reverse",
				From: to,
				To:   from,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge.Edge.Type, ShouldEqual, "test_edge_reverse")
		c.So(resp.Edge.Edge.To.Type, ShouldEqual, "test_node")
		c.So(resp.Edge.Edge.To.Id, ShouldEqual, from.Id)

		secondCreate, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: from,
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(secondCreate.Edge, ShouldBeNil)

		verifyReads(ctx, c, client, resp.Edge, "test_edge")

		deleteResp, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: from,
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(deleteResp.Edge.Edge.From.Id, ShouldEqual, from.Id)
		c.So(deleteResp.Edge.Edge.Type, ShouldEqual, "test_edge")

		secondDelete, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: from,
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(secondDelete.Edge, ShouldBeNil)
	})
}

func TestIntegration_FeedbackEdges(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()
	Convey("With two edges", t, func(c C) {
		user0 := randomUserEntity()
		user1 := randomUserEntity()
		user2 := randomUserEntity()
		c.Reset(func() {
			_, deleteErr := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: user1,
					Type: "feedback",
					To:   user0,
				},
			})
			c.So(deleteErr, ShouldBeNil)
			_, deleteErr = client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: user1,
					Type: "feedback",
					To:   user1,
				},
			})
			c.So(deleteErr, ShouldBeNil)
			_, deleteErr = client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: user1,
					Type: "feedback",
					To:   user2,
				},
			})
			c.So(deleteErr, ShouldBeNil)
		})
		_, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: user1,
				Type: "feedback",
				To:   user1,
			},
			Data: &graphdb.DataBag{
				Ints: map[string]int64{
					"last_update_time": 1,
				},
			},
		})
		c.So(err, ShouldBeNil)

		_, err = client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: user1,
				Type: "feedback",
				To:   user0,
			},
		})
		c.So(err, ShouldBeNil)

		_, err = client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: user1,
				Type: "feedback",
				To:   user2,
			},
			Data: &graphdb.DataBag{
				Ints: map[string]int64{
					"last_update_time": 3,
				},
			},
		})
		c.So(err, ShouldBeNil)

		// Items should return as user 0, 1, 2
		time.Sleep(time.Second)

		resp, listErr := client.EdgeList(ctx, &graphdb.EdgeListRequest{
			From:     user1,
			EdgeType: "feedback",
			Page: &graphdb.PagedRequest{
				Limit: 3,
			},
		})
		c.So(listErr, ShouldBeNil)
		c.So(len(resp.Edges), ShouldEqual, 3)
		c.So(resp.Edges[0].Edge.Edge.To.Id, ShouldEqual, user0.Id)
		c.So(resp.Edges[1].Edge.Edge.To.Id, ShouldEqual, user1.Id)
		c.So(resp.Edges[2].Edge.Edge.To.Id, ShouldEqual, user2.Id)

		// Change the middle creation time to something later, and watch it show up last now.

		_, err = client.EdgeUpdate(ctx, &graphdb.EdgeUpdateRequest{
			Edge: &graphdb.Edge{
				From: user1,
				Type: "feedback",
				To:   user0,
			},
			Data: &graphdb.DataBag{
				Ints: map[string]int64{
					"last_update_time": 4,
				},
			},
		})
		c.So(err, ShouldBeNil)

		resp2, listErr2 := client.EdgeList(ctx, &graphdb.EdgeListRequest{
			From:     user1,
			EdgeType: "feedback",
			Page: &graphdb.PagedRequest{
				Limit: 3,
			},
		})
		c.So(listErr2, ShouldBeNil)
		c.So(len(resp2.Edges), ShouldEqual, 3)
		c.So(resp2.Edges[0].Edge.Edge.To.Id, ShouldEqual, user1.Id)
		c.So(resp2.Edges[1].Edge.Edge.To.Id, ShouldEqual, user2.Id)
		c.So(resp2.Edges[2].Edge.Edge.To.Id, ShouldEqual, user0.Id)
	})
}

func TestIntegration_CreateForward(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	//conn, err := grpc.Dial(s.listenAddr, grpc.WithInsecure())
	//if err != nil {
	//	t.Fatal(err.Error())
	//}
	//client := graphdb.NewCohesion2Client(conn)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()
	Convey("double create works", t, func(c C) {
		from := randomUserEntity()
		to := randomUserEntity()
		resp, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: from,
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge.Edge.Type, ShouldEqual, "test_edge")
		c.So(resp.Edge.Edge.To.Type, ShouldEqual, "test_node")

		resp2, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: from,
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp2.Edge, ShouldBeNil)

		verifyReads(ctx, c, client, resp.Edge, "test_edge_reverse")

		deleteResp, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: from,
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(deleteResp.Edge.Edge.From.Id, ShouldEqual, from.Id)
		c.So(deleteResp.Edge.Edge.Type, ShouldEqual, "test_edge")

		secondDelete, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: from,
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(secondDelete.Edge, ShouldBeNil)
	})
}

func verifyReads(ctx context.Context, c C, client graphdb.GraphDB, expected *graphdb.LoadedEdge, reverseType string) {
	c.Convey("and read it back", func(c C) {
		resp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: expected.Edge,
		})
		c.So(err, ShouldBeNil)
		// Should be a time between now and 5 sec ago
		updatedAt := time.Unix(resp.Edge.Data.UpdatedAt.Seconds, int64(resp.Edge.Data.UpdatedAt.Nanos))
		c.So(time.Now().Sub(updatedAt), ShouldBeLessThan, time.Second*5)
		c.So(time.Now().Sub(updatedAt), ShouldBeGreaterThan, 0)
		c.So(resp.Edge.Data.Version, ShouldEqual, 0)

		c.So(resp.Edge.Edge.From.Id, ShouldEqual, expected.Edge.From.Id)
		c.So(resp.Edge.Edge.To.Id, ShouldEqual, expected.Edge.To.Id)
		c.So(resp.Edge.Data.Data.Ints["age"], ShouldEqual, expected.Data.Data.Ints["age"])
		c.So(resp.Edge.Data.Data.Strings["name"], ShouldEqual, expected.Data.Data.Strings["name"])
		c.So(resp.Edge.Data.Data.Bools["awesome"], ShouldEqual, expected.Data.Data.Bools["awesome"])
	})

	c.Convey("and read it backwards", func(c C) {
		resp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: &graphdb.Edge{
				From: expected.Edge.To,
				To:   expected.Edge.From,
				Type: reverseType,
			},
		})
		c.So(err, ShouldBeNil)

		c.So(resp.Edge.Edge.From.Id, ShouldEqual, expected.Edge.To.Id)
		c.So(resp.Edge.Edge.To.Id, ShouldEqual, expected.Edge.From.Id)

		// Should be a time between now and 5 sec ago
		updatedAt := time.Unix(resp.Edge.Data.UpdatedAt.Seconds, int64(resp.Edge.Data.UpdatedAt.Nanos))
		c.So(time.Now().Sub(updatedAt), ShouldBeLessThan, time.Second*5)
		c.So(time.Now().Sub(updatedAt), ShouldBeGreaterThan, 0)
		c.So(resp.Edge.Data.Version, ShouldEqual, 0)

		c.So(resp.Edge.Data.Data.Ints["age"], ShouldEqual, expected.Data.Data.Ints["age"])
		c.So(resp.Edge.Data.Data.Strings["name"], ShouldEqual, expected.Data.Data.Strings["name"])
		c.So(resp.Edge.Data.Data.Bools["awesome"], ShouldEqual, expected.Data.Data.Bools["awesome"])
	})

	c.Convey("and increment count", func(c C) {
		resp, err := client.EdgeCount(ctx, &graphdb.EdgeCountRequest{
			From:     expected.Edge.From,
			EdgeType: expected.Edge.Type,
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Count, ShouldEqual, 1)
	})

	c.Convey("and increment count backwards", func(c C) {
		resp, err := client.EdgeCount(ctx, &graphdb.EdgeCountRequest{
			From:     expected.Edge.To,
			EdgeType: reverseType,
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Count, ShouldEqual, 1)
	})

	c.Convey("and show up in the list", func(c C) {
		resp, err := client.EdgeList(ctx, &graphdb.EdgeListRequest{
			From:     expected.Edge.From,
			EdgeType: expected.Edge.Type,
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Cursor, ShouldEqual, "")
		c.So(len(resp.Edges), ShouldEqual, 1)
		c.So(resp.Edges[0].Edge.Edge.From.Id, ShouldEqual, expected.Edge.From.Id)
		c.So(resp.Edges[0].Edge.Edge.To.Id, ShouldEqual, expected.Edge.To.Id)
		c.So(resp.Edges[0].Edge.Edge.Type, ShouldEqual, expected.Edge.Type)
		c.So(resp.Edges[0].Cursor, ShouldEqual, "")
	})

	c.Convey("and show up in the list backwards", func(c C) {
		resp, err := client.EdgeList(ctx, &graphdb.EdgeListRequest{
			From:     expected.Edge.To,
			EdgeType: reverseType,
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Cursor, ShouldEqual, "")
		c.So(len(resp.Edges), ShouldEqual, 1)
		c.So(resp.Edges[0].Edge.Edge.From.Id, ShouldEqual, expected.Edge.To.Id)
		c.So(resp.Edges[0].Edge.Edge.To.Id, ShouldEqual, expected.Edge.From.Id)
		c.So(resp.Edges[0].Edge.Edge.Type, ShouldEqual, reverseType)
		c.So(resp.Edges[0].Cursor, ShouldEqual, "")
	})
}

func TestIntegration_Cursors(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()
	Convey("Create follower graph", t, func(c C) {
		// userA -> follows all other users (B, C, D)
		// userC -> followed by all other user (A, B, D)
		userA := randomUserEntity()
		userB := randomUserEntity()
		userC := randomUserEntity()
		userD := randomUserEntity()
		userE := randomUserEntity()
		resp, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: userA,
				To:   userB,
				Type: "test_edge",
			},
			Data: &graphdb.DataBag{
				Ints: map[string]int64{
					"age": 34,
				},
				Bools: map[string]bool{
					"awesome": true,
				},
				Strings: map[string]string{
					"name": "AB",
				},
			},
		})

		c.So(err, ShouldBeNil)
		c.So(resp, ShouldNotBeNil)

		resp, err = client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: userA,
				To:   userC,
				Type: "test_edge",
			},
			Data: &graphdb.DataBag{
				Ints: map[string]int64{
					"age": 27,
				},
				Bools: map[string]bool{
					"marvel": true,
				},
				Strings: map[string]string{
					"name": "AC",
				},
			},
		})

		c.So(err, ShouldBeNil)
		c.So(resp, ShouldNotBeNil)

		resp, err = client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: userA,
				To:   userD,
				Type: "test_edge",
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp, ShouldNotBeNil)
		resp, err = client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: userA,
				To:   userE,
				Type: "test_edge",
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp, ShouldNotBeNil)

		resp, err = client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: userC,
				To:   userB,
				Type: "test_edge_reverse",
			},
			Data: &graphdb.DataBag{
				Bools: map[string]bool{
					"batman": true,
				},
				Strings: map[string]string{
					"name": "BC",
				},
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp, ShouldNotBeNil)

		resp, err = client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: userC,
				To:   userD,
				Type: "test_edge_reverse",
			},
			Data: &graphdb.DataBag{
				Strings: map[string]string{
					"name": "DC",
				},
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp, ShouldNotBeNil)

		c.Reset(func() {
			_, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: userA,
					To:   userB,
					Type: "test_edge",
				},
			})
			c.So(err, ShouldBeNil)

			_, err = client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: userA,
					To:   userC,
					Type: "test_edge",
				},
			})
			c.So(err, ShouldBeNil)

			_, err = client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: userA,
					To:   userD,
					Type: "test_edge",
				},
			})
			c.So(err, ShouldBeNil)

			_, err = client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: userB,
					To:   userC,
					Type: "test_edge",
				},
			})
			c.So(err, ShouldBeNil)

			_, err = client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: userC,
					To:   userD,
					Type: "test_edge",
				},
			})
			c.So(err, ShouldBeNil)
		})

		c.Convey("fetch `followed_by` edges with a cursor in descending order", func(c C) {
			c.Convey("fetch first user with an empty cursor", func(c C) {
				lresp, err := client.EdgeList(ctx, &graphdb.EdgeListRequest{
					From:     userC,
					EdgeType: "test_edge_reverse",
					Page: &graphdb.PagedRequest{
						Limit:  1,
						Cursor: "",
						Order:  graphdb.PagedRequest_DESC,
					},
				})
				c.So(err, ShouldBeNil)
				c.So(lresp.Cursor, ShouldNotEqual, "")
				c.So(len(lresp.Edges), ShouldEqual, 1)
				c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userC.Id)
				c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userD.Id)
				c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge_reverse")
				c.So(lresp.Edges[0].Edge.Data.Data.Strings["name"], ShouldEqual, "DC")
				c.So(lresp.Edges[0].Cursor, ShouldNotEqual, "")
				c.So(lresp.Edges[0].Cursor, ShouldEqual, lresp.Cursor)

				cursor := lresp.Cursor
				c.Convey("querying wrong index with cursor should return an error", func(c C) {
					lresp, err = client.EdgeList(ctx, &graphdb.EdgeListRequest{
						From:     userC,
						EdgeType: "test_edge",
						Page: &graphdb.PagedRequest{
							Limit:  1,
							Cursor: cursor,
							Order:  graphdb.PagedRequest_DESC,
						},
					})
					c.So(err, ShouldNotBeNil)
				})

				c.Convey("fetch second user with RDS cursor", func(c C) {
					// override cursor with timestamp to test RDS cursors through graphdb.
					ca := lresp.Edges[0].Edge.Data.CreatedAt
					sec, nsec := ca.GetSeconds(), int64(ca.GetNanos())
					rdsCursor := strconv.Itoa(int(time.Unix(sec, nsec).UnixNano()))
					lresp, err = client.EdgeList(ctx, &graphdb.EdgeListRequest{
						From:     userC,
						EdgeType: "test_edge_reverse",
						Page: &graphdb.PagedRequest{
							Limit:  1,
							Cursor: rdsCursor,
							Order:  graphdb.PagedRequest_DESC,
						},
					})
					c.So(err, ShouldBeNil)
					c.So(lresp.Cursor, ShouldNotEqual, "")

					c.So(len(lresp.Edges), ShouldEqual, 1)
					c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userC.Id)
					c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userB.Id)
					c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge_reverse")
					c.So(lresp.Edges[0].Edge.Data.Data.Strings["name"], ShouldEqual, "BC")
					c.So(lresp.Edges[0].Edge.Data.Data.Bools["batman"], ShouldEqual, true)
					c.So(lresp.Edges[0].Cursor, ShouldNotEqual, "")
					c.So(lresp.Edges[0].Cursor, ShouldEqual, lresp.Cursor)
				})

				c.Convey("fetch second user with DynamoDB cursor", func(c C) {
					lresp, err = client.EdgeList(ctx, &graphdb.EdgeListRequest{
						From:     userC,
						EdgeType: "test_edge_reverse",
						Page: &graphdb.PagedRequest{
							Limit:  1,
							Cursor: cursor,
							Order:  graphdb.PagedRequest_DESC,
						},
					})
					c.So(err, ShouldBeNil)
					c.So(lresp.Cursor, ShouldNotEqual, "")

					c.So(len(lresp.Edges), ShouldEqual, 1)
					c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userC.Id)
					c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userB.Id)
					c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge_reverse")
					c.So(lresp.Edges[0].Edge.Data.Data.Strings["name"], ShouldEqual, "BC")
					c.So(lresp.Edges[0].Edge.Data.Data.Bools["batman"], ShouldEqual, true)
					c.So(lresp.Edges[0].Cursor, ShouldNotEqual, "")
					c.So(lresp.Edges[0].Cursor, ShouldEqual, lresp.Cursor)

					cursor = lresp.Cursor
					c.Convey("fetch last user but set limit to be 100", func(c C) {
						lresp, err = client.EdgeList(ctx, &graphdb.EdgeListRequest{
							From:     userC,
							EdgeType: "test_edge_reverse",
							Page: &graphdb.PagedRequest{
								Limit:  100,
								Cursor: cursor,
								Order:  graphdb.PagedRequest_DESC,
							},
						})
						c.So(err, ShouldBeNil)
						c.So(lresp.Cursor, ShouldEqual, "")
						c.So(len(lresp.Edges), ShouldEqual, 1)
						c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userC.Id)
						c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userA.Id)
						c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge_reverse")
						c.So(lresp.Edges[0].Edge.Data.Data.Ints["age"], ShouldEqual, 27)
						c.So(lresp.Edges[0].Edge.Data.Data.Strings["name"], ShouldEqual, "AC")
						c.So(lresp.Edges[0].Edge.Data.Data.Bools["marvel"], ShouldEqual, true)
						c.So(lresp.Edges[0].Cursor, ShouldEqual, lresp.Cursor)
						c.Convey("verify cursor is empty for last user", func(c C) {
							c.So(lresp.Edges[0].Cursor, ShouldEqual, "")
						})
					})
				})
			})
		})
		c.Convey("fetch `follow` edges with a cursor", func(c C) {
			c.Convey("fetch first user using empty cursor", func(c C) {
				lresp, err := client.EdgeList(ctx, &graphdb.EdgeListRequest{
					From:     userA,
					EdgeType: "test_edge",
					Page: &graphdb.PagedRequest{
						Limit:  1,
						Cursor: "",
					},
				})
				c.So(err, ShouldBeNil)
				c.So(lresp.Cursor, ShouldNotEqual, "")
				c.So(len(lresp.Edges), ShouldEqual, 1)
				c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userA.Id)
				c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userB.Id)
				c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge")
				c.So(lresp.Edges[0].Edge.Data.Data.Ints["age"], ShouldEqual, 34)
				c.So(lresp.Edges[0].Edge.Data.Data.Strings["name"], ShouldEqual, "AB")
				c.So(lresp.Edges[0].Edge.Data.Data.Bools["awesome"], ShouldEqual, true)
				c.So(lresp.Edges[0].Cursor, ShouldNotEqual, "")
				c.So(lresp.Edges[0].Cursor, ShouldEqual, lresp.Cursor)

				cursor := lresp.Cursor
				c.Convey("fetch second followed user using RDS cursor", func(c C) {
					// override cursor with timestamp to test RDS cursors through graphdb.
					ca := lresp.Edges[0].Edge.Data.CreatedAt
					sec, nsec := ca.GetSeconds(), int64(ca.GetNanos())
					rdsCursor := strconv.Itoa(int(time.Unix(sec, nsec).UnixNano()))
					lresp, err = client.EdgeList(ctx, &graphdb.EdgeListRequest{
						From:     userA,
						EdgeType: "test_edge",
						Page: &graphdb.PagedRequest{
							Limit:  1,
							Cursor: rdsCursor,
						},
					})
					c.So(err, ShouldBeNil)
					c.So(lresp.Cursor, ShouldNotEqual, "")
					c.So(len(lresp.Edges), ShouldEqual, 1)
					c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userA.Id)
					c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userC.Id)
					c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge")
					c.So(lresp.Edges[0].Edge.Data.Data.Ints["age"], ShouldEqual, 27)
					c.So(lresp.Edges[0].Edge.Data.Data.Strings["name"], ShouldEqual, "AC")
					c.So(lresp.Edges[0].Edge.Data.Data.Bools["marvel"], ShouldEqual, true)
					c.So(lresp.Edges[0].Cursor, ShouldNotEqual, "")
				})

				c.Convey("fetch second followed user using DynamoDB cursor", func(c C) {
					// Note: This should not hit storage (instead it should go to cache)
					lresp, err = client.EdgeList(ctx, &graphdb.EdgeListRequest{
						From:     userA,
						EdgeType: "test_edge",
						Page: &graphdb.PagedRequest{
							Limit:  1,
							Cursor: cursor,
						},
					})
					c.So(err, ShouldBeNil)
					c.So(lresp.Cursor, ShouldNotEqual, "")
					c.So(len(lresp.Edges), ShouldEqual, 1)
					c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userA.Id)
					c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userC.Id)
					c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge")
					c.So(lresp.Edges[0].Edge.Data.Data.Ints["age"], ShouldEqual, 27)
					c.So(lresp.Edges[0].Edge.Data.Data.Strings["name"], ShouldEqual, "AC")
					c.So(lresp.Edges[0].Edge.Data.Data.Bools["marvel"], ShouldEqual, true)
					c.So(lresp.Edges[0].Cursor, ShouldNotEqual, "")

					cursor = lresp.Cursor
					c.Convey("fetch second to last user but set limit to be 1", func(c C) {
						lresp, err = client.EdgeList(ctx, &graphdb.EdgeListRequest{
							From:     userA,
							EdgeType: "test_edge",
							Page: &graphdb.PagedRequest{
								Limit:  1,
								Cursor: cursor,
							},
						})
						c.So(err, ShouldBeNil)
						c.So(lresp.Cursor, ShouldNotEqual, "")
						c.So(len(lresp.Edges), ShouldEqual, 1)
						c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userA.Id)
						c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userD.Id)
						c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge")
						cursor = lresp.Cursor
						c.Convey("fetch last user but set limit to be 1", func(c C) {
							lresp, err = client.EdgeList(ctx, &graphdb.EdgeListRequest{
								From:     userA,
								EdgeType: "test_edge",
								Page: &graphdb.PagedRequest{
									Limit:  1,
									Cursor: cursor,
								},
							})
							c.So(err, ShouldBeNil)
							c.So(len(lresp.Edges), ShouldEqual, 1)
							c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userA.Id)
							c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userE.Id)
							c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge")
							c.So(lresp.Cursor, ShouldEqual, "")
							c.So(lresp.Edges[0].Cursor, ShouldEqual, "")
						})
					})
					c.Convey("fetch last user but set limit to be 100", func(c C) {
						lresp, err = client.EdgeList(ctx, &graphdb.EdgeListRequest{
							From:     userA,
							EdgeType: "test_edge",
							Page: &graphdb.PagedRequest{
								Limit:  100,
								Cursor: cursor,
							},
						})
						c.So(err, ShouldBeNil)
						c.So(lresp.Cursor, ShouldEqual, "")
						c.So(len(lresp.Edges), ShouldEqual, 2)
						c.So(lresp.Edges[0].Edge.Edge.From.Id, ShouldEqual, userA.Id)
						c.So(lresp.Edges[0].Edge.Edge.To.Id, ShouldEqual, userD.Id)
						c.So(lresp.Edges[0].Edge.Edge.Type, ShouldEqual, "test_edge")
						c.So(lresp.Edges[1].Edge.Edge.From.Id, ShouldEqual, userA.Id)
						c.So(lresp.Edges[1].Edge.Edge.To.Id, ShouldEqual, userE.Id)
						c.So(lresp.Edges[1].Edge.Edge.Type, ShouldEqual, "test_edge")
						// No last item
						c.So(lresp.Edges[1].Cursor, ShouldEqual, "")
					})
				})
			})
		})
	})
}

func TestIntegration_Flow(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()
	Convey("Can make an edge with twirp", t, func(c C) {
		from := randomUserEntity()
		to := randomUserEntity()
		resp, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				From: from,
				To:   to,
				Type: "test_edge",
			},
			Data: &graphdb.DataBag{
				Ints: map[string]int64{
					"age": 34,
				},
				Bools: map[string]bool{
					"awesome": true,
				},
				Strings: map[string]string{
					"name": "jack",
				},
			},
		})

		c.So(err, ShouldBeNil)
		c.So(resp, ShouldNotBeNil)
		c.Reset(func() {
			_, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: from,
					To:   to,
					Type: "test_edge",
				},
			})
			c.So(err, ShouldBeNil)
		})
		flowAfterMadeEdge(ctx, c, client, from, to, resp)
	})
}

func TestIntegration_DeleteVersioned(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()

	Convey("Given an edge", t, func(c C) {
		//create the edge
		from := randomUserEntity()
		to := randomUserEntity()

		resp, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: from,
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge, ShouldNotBeNil)

		createdEdge := resp.Edge
		c.So(createdEdge.Edge.Type, ShouldEqual, "test_edge")
		c.So(createdEdge.Edge.To.Type, ShouldEqual, "test_node")
		c.So(createdEdge.Data.Version, ShouldEqual, 0)

		c.Reset(func() {
			_, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: from,
					To:   to,
					Type: "test_edge",
				},
			})
			c.So(err, ShouldBeNil)
		})

		c.Convey("When trying to delete while passing a proper version", func(c C) {
			// delete
			delResp, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: from,
					To:   to,
					Type: "test_edge",
				},
				Version: &graphdb.RequiredVersion{
					UpdatedAt: createdEdge.Data.UpdatedAt,
				},
			})
			c.So(err, ShouldBeNil)
			c.So(delResp.Edge, ShouldNotBeNil)

			c.Convey("Then the edge should be deleted from database", func(c C) {
				// fetch from db and it should be nil.
				getResp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
					Edge: &graphdb.Edge{
						From: from,
						To:   to,
						Type: "test_edge",
					},
				})
				c.So(err, ShouldBeNil)
				c.So(getResp.Edge, ShouldBeNil)
			})
		})

		c.Convey("When trying to delete and passing an incorrect version", func(c C) {
			// delete
			delResp, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					From: from,
					To:   to,
					Type: "test_edge",
				},
				Version: &graphdb.RequiredVersion{
					UpdatedAt: conversion.ToProtoTime(time.Now()),
				},
			})
			c.So(err, ShouldBeNil)
			c.So(delResp.Edge, ShouldBeNil)

			c.Convey("Then the edge should still exist in database", func(c C) {
				getResp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
					Edge: &graphdb.Edge{
						From: from,
						To:   to,
						Type: "test_edge",
					},
				})
				c.So(err, ShouldBeNil)
				c.So(getResp.Edge, ShouldNotBeNil)
				c.So(createdEdge.Data.Version, ShouldEqual, getResp.Edge.Data.Version)
			})
		})
	})
}

func TestIntegration_UpdateVersioned(t *testing.T) {
	//t.Parallel()
	s, onClose := startServer(t)
	if s == nil {
		return
	}
	defer onClose(time.Second * 30)
	client := graphdb.NewGraphDBProtobufClient(s.listenAddr, &http.Client{})
	ctx := context.Background()

	Convey("Given an edge to update", t, func(c C) {
		from := randomUserEntity()
		to := randomUserEntity()

		resp, err := client.EdgeCreate(ctx, &graphdb.EdgeCreateRequest{
			Edge: &graphdb.Edge{
				Type: "test_edge",
				From: from,
				To:   to,
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge.Edge.Type, ShouldEqual, "test_edge")
		c.So(resp.Edge.Edge.To.Type, ShouldEqual, "test_node")
		c.So(resp.Edge.Data.Version, ShouldEqual, 0)
		createdEdge := resp.Edge

		c.Reset(func() {
			deleteResp, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
				Edge: &graphdb.Edge{
					Type: "test_edge",
					From: from,
					To:   to,
				},
			})
			c.So(err, ShouldBeNil)
			c.So(deleteResp.Edge.Edge.From.Id, ShouldEqual, from.Id)
			c.So(deleteResp.Edge.Edge.Type, ShouldEqual, "test_edge")
		})

		c.Convey("When you pass the proper version", func(c C) {
			// update the edge with the UA from the creation step.

			updateRes, err := client.EdgeUpdate(ctx, &graphdb.EdgeUpdateRequest{
				Edge: &graphdb.Edge{
					From: from,
					To:   to,
					Type: "test_edge",
				},
				Data: &graphdb.DataBag{
					Ints: map[string]int64{
						"age": 30,
					},
				},
				Version: &graphdb.RequiredVersion{
					UpdatedAt: createdEdge.Data.UpdatedAt,
				},
			})

			c.So(err, ShouldBeNil)
			c.So(updateRes.Edge, ShouldNotBeNil)
			c.So(updateRes.Edge.Data.Data.Ints["age"], ShouldEqual, 30)
			c.So(updateRes.Edge.Data.Version, ShouldEqual, 1)

			c.Convey("The edge should be updated in database", func(c C) {
				// get the edge and assert the data.

				getResp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
					Edge: &graphdb.Edge{
						From: from,
						To:   to,
						Type: "test_edge",
					},
				})
				c.So(err, ShouldBeNil)
				c.So(getResp.Edge, ShouldNotBeNil)
				c.So(getResp.Edge.Data.Data.Ints["age"], ShouldEqual, 30)
				c.So(getResp.Edge.Data.Version, ShouldEqual, 1)
			})
		})

		c.Convey("When you pass an incorrect version", func(c C) {
			// update the edge with the UA equal to now.

			resp, err := client.EdgeUpdate(ctx, &graphdb.EdgeUpdateRequest{
				Edge: &graphdb.Edge{
					From: from,
					To:   to,
					Type: "test_edge",
				},
				Data: &graphdb.DataBag{
					Ints: map[string]int64{
						"age": 99,
					},
				},
				Version: &graphdb.RequiredVersion{
					UpdatedAt: conversion.ToProtoTime(time.Now()),
				},
			})
			// This is scenario gets the same treatment as edges that do not exist.
			c.So(err, ShouldBeNil)
			// In which case, the Edge in the response is nil.
			c.So(resp.Edge, ShouldBeNil)

			c.Convey("The edge should not be updated", func(c C) {
				// get the edge and assert the data.

				getResp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
					Edge: &graphdb.Edge{
						From: from,
						To:   to,
						Type: "test_edge",
					},
				})
				c.So(err, ShouldBeNil)
				c.So(getResp.Edge, ShouldNotBeNil)
				c.So(getResp.Edge.Data.Version, ShouldEqual, 0)
			})
		})
	})
}

func flowAfterMadeEdge(ctx context.Context, c C, client graphdb.GraphDB, from *graphdb.Node, to *graphdb.Node, resp *graphdb.EdgeCreateResponse) {
	c.Convey("and explicitly remove", func(c C) {
		resp, err := client.EdgeDelete(ctx, &graphdb.EdgeDeleteRequest{
			Edge: &graphdb.Edge{
				From: from,
				To:   to,
				Type: "test_edge",
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge, ShouldNotBeNil)
		c.Convey("and never see again", func(c C) {
			resp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
				Edge: &graphdb.Edge{
					From: from,
					To:   to,
					Type: "test_edge",
				},
			})
			c.So(err, ShouldBeNil)
			c.So(resp.Edge, ShouldBeNil)

			resp2, err := client.EdgeCount(ctx, &graphdb.EdgeCountRequest{
				From:     from,
				EdgeType: "test_edge",
			})
			c.So(err, ShouldBeNil)
			c.So(resp2.Count, ShouldEqual, 0)
		})
	})

	verifyReads(ctx, c, client, resp.Edge, "test_edge_reverse")

	c.Convey("and updated it", func(c C) {
		updateRes, err := client.EdgeUpdate(ctx, &graphdb.EdgeUpdateRequest{
			Edge: &graphdb.Edge{
				From: from,
				To:   to,
				Type: "test_edge",
			},
			Data: &graphdb.DataBag{
				Ints: map[string]int64{
					"age": 30,
				},
			},
		})
		c.So(err, ShouldBeNil)
		c.So(updateRes.Edge.Edge.From.Id, ShouldEqual, from.Id)
		c.So(updateRes.Edge.Data.Data.Ints["age"], ShouldEqual, 30)
		c.So(updateRes.Edge.Data.Version, ShouldBeGreaterThan, 0)

		resp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: &graphdb.Edge{
				From: from,
				To:   to,
				Type: "test_edge",
			},
		})
		c.So(err, ShouldBeNil)
		// Should be a time between now and 5 sec ago
		updatedAt := time.Unix(resp.Edge.Data.UpdatedAt.Seconds, int64(resp.Edge.Data.UpdatedAt.Nanos))
		c.So(time.Now().Sub(updatedAt), ShouldBeLessThan, time.Second*5)
		c.So(time.Now().Sub(updatedAt), ShouldBeGreaterThan, 0)
		c.So(resp.Edge.Data.Version, ShouldEqual, 1)

		c.So(resp.Edge.Data.Data.Ints["age"], ShouldEqual, 30)
		c.So(resp.Edge.Data.Data.Strings["name"], ShouldEqual, "jack")
		c.So(resp.Edge.Data.Data.Bools["awesome"], ShouldEqual, true)
	})

	c.Convey("and updated it in reverse", func(c C) {
		updateRes, err := client.EdgeUpdate(ctx, &graphdb.EdgeUpdateRequest{
			Edge: &graphdb.Edge{
				From: to,
				To:   from,
				Type: "test_edge_reverse",
			},
			Data: &graphdb.DataBag{
				Ints: map[string]int64{
					"age": 30,
				},
			},
		})
		c.So(err, ShouldBeNil)
		c.So(updateRes.Edge.Edge.Type, ShouldEqual, "test_edge_reverse")
		c.So(updateRes.Edge.Edge.From.Id, ShouldEqual, to.Id)
		c.So(updateRes.Edge.Data.Data.Ints["age"], ShouldEqual, 30)
		c.So(updateRes.Edge.Data.Version, ShouldBeGreaterThan, 0)

		resp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: &graphdb.Edge{
				From: from,
				To:   to,
				Type: "test_edge",
			},
		})
		c.So(err, ShouldBeNil)
		// Should be a time between now and 5 sec ago
		updatedAt := time.Unix(resp.Edge.Data.UpdatedAt.Seconds, int64(resp.Edge.Data.UpdatedAt.Nanos))
		c.So(time.Now().Sub(updatedAt), ShouldBeLessThan, time.Second*5)
		c.So(time.Now().Sub(updatedAt), ShouldBeGreaterThan, 0)
		c.So(resp.Edge.Data.Version, ShouldEqual, 1)

		c.So(resp.Edge.Data.Data.Ints["age"], ShouldEqual, 30)
		c.So(resp.Edge.Data.Data.Strings["name"], ShouldEqual, "jack")
		c.So(resp.Edge.Data.Data.Bools["awesome"], ShouldEqual, true)
	})

	c.Convey("and put it", func(c C) {
		_, err := client.EdgeUpdate(ctx, &graphdb.EdgeUpdateRequest{
			Edge: &graphdb.Edge{
				From: from,
				To:   to,
				Type: "test_edge",
			},
			Data: &graphdb.DataBag{
				Ints: map[string]int64{
					"age": 30,
				},
			},
			OverwriteData: true,
		})
		c.So(err, ShouldBeNil)

		resp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: &graphdb.Edge{
				From: from,
				To:   to,
				Type: "test_edge",
			},
		})
		c.So(err, ShouldBeNil)
		// Should be a time between now and 5 sec ago
		updatedAt := time.Unix(resp.Edge.Data.UpdatedAt.Seconds, int64(resp.Edge.Data.UpdatedAt.Nanos))
		c.So(time.Now().Sub(updatedAt), ShouldBeLessThan, time.Second*5)
		c.So(time.Now().Sub(updatedAt), ShouldBeGreaterThan, 0)
		c.So(resp.Edge.Data.Version, ShouldEqual, 1)

		c.So(resp.Edge.Data.Data.Ints["age"], ShouldEqual, 30)
		c.So(resp.Edge.Data.Data.Strings["name"], ShouldEqual, "")
		c.So(resp.Edge.Data.Data.Bools["awesome"], ShouldEqual, false)
	})

	c.Convey("and change type", func(c C) {
		// Reverses the edge
		createdEdge, err := client.EdgeUpdateType(ctx, &graphdb.EdgeUpdateTypeRequest{
			Edge: &graphdb.Edge{
				From: from,
				To:   to,
				Type: "test_edge",
			},
			NewType: "test_edge_reverse",
		})
		c.So(err, ShouldBeNil)
		c.So(createdEdge.Edge, ShouldNotBeNil)
		c.So(createdEdge.Edge.Edge.Type, ShouldEqual, "test_edge_reverse")
		c.So(createdEdge.Edge.Edge.From.Id, ShouldEqual, from.Id)
		c.So(createdEdge.Edge.Edge.To.Id, ShouldEqual, to.Id)

		resp, err := client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: &graphdb.Edge{
				From: from,
				To:   to,
				Type: "test_edge",
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge, ShouldBeNil)

		resp, err = client.EdgeGet(ctx, &graphdb.EdgeGetRequest{
			Edge: &graphdb.Edge{
				From: to,
				To:   from,
				Type: "test_edge",
			},
		})
		c.So(err, ShouldBeNil)
		c.So(resp.Edge, ShouldNotBeNil)
		// Should be a time between now and 5 sec ago
		updatedAt := time.Unix(resp.Edge.Data.UpdatedAt.Seconds, int64(resp.Edge.Data.UpdatedAt.Nanos))
		c.So(time.Now().Sub(updatedAt), ShouldBeLessThan, time.Second*5)
		c.So(time.Now().Sub(updatedAt), ShouldBeGreaterThan, 0)
		// Edge got reset with the changed type
		c.So(resp.Edge.Data.Version, ShouldEqual, 0)

		c.So(resp.Edge.Data.Data.Strings["name"], ShouldEqual, "jack")
		c.So(resp.Edge.Data.Data.Bools["awesome"], ShouldEqual, true)
		c.So(resp.Edge.Data.Data.Ints["age"], ShouldEqual, 34)
	})
}

type testServerInfo struct {
	listenAddr string
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

func startServer(t *testing.T) (*testServerInfo, func(time.Duration)) {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"rollbar.access_token":     []byte(""),
		"statsd.hostport":          []byte(""),
		"debug.addr":               []byte(":0"),
		"graphdb.listen_addr":      []byte(":0"),
		"graphdb.storage_config":   []byte("./internal/storage/storage.json"),
		"graphdb.cache_config":     []byte("./internal/cache/cache.json"),
		"graphdb.cache.list_limit": []byte("2"),
		"graphdb.block_for_counts": []byte("true"),
		"logging.to_stdout":        []byte("false"),
		"consul.fallback_env":      []byte("integration"),
		"logging.to_stderr":        []byte("true"),
	})
	if err != nil {
		t.Error(err)
		return nil, nil
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				ElevateLogKey: elevateKey,
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: log.LoggerFunc(func(vs ...interface{}) {
						fmt.Println(vs...)
					}),
				},
				DebugLog: log.ContextLogger{
					Logger: log.LoggerFunc(func(vs ...interface{}) {
						fmt.Println(vs...)
					}),
				},
				LogToDebug: func(_ ...interface{}) bool {
					return true
				},
			},
			PanicLogger: panicPanic{},
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			t.Log("Listening address accepted", listeningAddr.String())
			fmt.Println("Listen accepted")
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions

	go func() {
		fmt.Println("Calling main")
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	fmt.Println("Main select?")
	select {
	case <-exitCalled:
		return nil, nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Minute * 5):
		t.Error("Took to long to start service")
		return nil, nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return &testServerInfo{
		listenAddr: addressForIntegrationTests,
	}, onFinish
}
