package cohesion

import (
	"net/http"

	"code.justin.tv/feeds/graphdb/proto/datastorerpc"
	v2cohesion "code.justin.tv/web/cohesion/api/v2"
	"code.justin.tv/web/cohesion/backend"
	v2 "code.justin.tv/web/cohesion/client/v2"
	"code.justin.tv/web/cohesion/clients"
	"code.justin.tv/web/cohesion/datastore/twirprpc"
	"code.justin.tv/web/cohesion/hitcounter"
	"code.justin.tv/web/cohesion/rpc"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
	"google.golang.org/grpc"
)

var _ hitcounter.HitCounter = &emptyHitCounter{}
var _ rpc.V2Client = &wrappedServer{}
var _ clients.ErrorLogger = &emptyErrorLogger{}

type emptyHitCounter struct{}

func (h *emptyHitCounter) Hit(client string) error {
	return nil
}
func (h *emptyHitCounter) HasQuota(client string) bool {
	return false
}
func (h *emptyHitCounter) Stop() {}

type emptyErrorLogger struct{}

func (e *emptyErrorLogger) RequestError(*http.Request, error)       {}
func (e *emptyErrorLogger) RequestPanic(*http.Request, interface{}) {}
func (e *emptyErrorLogger) Error(error)                             {}
func (e *emptyErrorLogger) Panic(interface{})                       {}

// NewClientV2 returns a v2 cohesion compatible client that talks to graphdb
func NewClientV2(reader datastorerpc.Reader, writer datastorerpc.Writer, mirror bool) v2.Client {
	t := &twirprpc.TwirpRPC{
		Reader: reader,
		Writer: writer,
		Mirror: mirror,
	}
	// stats aren't actually used!
	stats := &statsd.NoopClient{}
	b := backend.New(stats)
	b.Reader = t
	b.Writer = t
	s := v2cohesion.NewCohesionServer(b, stats, &emptyErrorLogger{}, &emptyHitCounter{})
	w := &wrappedServer{
		wrapped: s,
	}
	return v2.NewFromConn(w, "graphdb")
}

type wrappedServer struct {
	wrapped *v2cohesion.CohesionServer
}

func (w *wrappedServer) Create(ctx context.Context, in *rpc.RequestV2, opts ...grpc.CallOption) (*rpc.EmptyResponseV2, error) {
	return w.wrapped.Create(ctx, in)
}
func (w *wrappedServer) Delete(ctx context.Context, in *rpc.RequestV2, opts ...grpc.CallOption) (*rpc.EmptyResponseV2, error) {
	return w.wrapped.Delete(ctx, in)
}
func (w *wrappedServer) Update(ctx context.Context, in *rpc.RequestV2, opts ...grpc.CallOption) (*rpc.EmptyResponseV2, error) {
	return w.wrapped.Update(ctx, in)
}
func (w *wrappedServer) Get(ctx context.Context, in *rpc.RequestV2, opts ...grpc.CallOption) (*rpc.ListResponseV2, error) {
	return w.wrapped.Get(ctx, in)
}
func (w *wrappedServer) Count(ctx context.Context, in *rpc.RequestV2, opts ...grpc.CallOption) (*rpc.CountResponseV2, error) {
	return w.wrapped.Count(ctx, in)
}
func (w *wrappedServer) Associations(ctx context.Context, in *rpc.RequestV2, opts ...grpc.CallOption) (*rpc.GetAllResponseV2, error) {
	return w.wrapped.Associations(ctx, in)
}
func (w *wrappedServer) GetHitCounts(ctx context.Context, in *rpc.HitCountRequestV2, opts ...grpc.CallOption) (*rpc.HitCountResponseV2, error) {
	return w.wrapped.GetHitCounts(ctx, in)
}
