package checkpoint

import (
	"context"
	"time"

	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/log"
)

type checkpointInput struct {
	cursor   string
	progress float64
}

type Checkpointer struct {
	Destination graphdb.GraphDB
	Key         string
	NodeType    string
	Log         log.Logger
	queue       chan checkpointInput
	onClose     chan struct{}
}

func (c *Checkpointer) Write(cursor string, progress float64) {
	select {
	case c.queue <- checkpointInput{
		cursor:   cursor,
		progress: progress,
	}:
	case <-c.onClose:
	default:
	}
}

func (c *Checkpointer) Read(ctx context.Context) (string, error) {
	resp, err := c.Destination.NodeGet(ctx, &graphdb.NodeGetRequest{
		Node: &graphdb.Node{
			Type: c.NodeType,
			Id:   c.Key,
		},
	})
	if err != nil {
		return "", err
	}
	if resp.GetNode() == nil {
		return "", nil
	}
	if resp.GetNode().GetData().GetData().GetStrings() == nil {
		return "", nil
	}
	return resp.Node.GetData().GetData().GetStrings()["cursor"], nil
}

func (c *Checkpointer) UpdateCursor(ctx context.Context, cursor string, progress float64) error {
	_, err := c.Destination.NodeUpdate(ctx, &graphdb.NodeUpdateRequest{
		Node: &graphdb.Node{
			Type: c.NodeType,
			Id:   c.Key,
		},
		Data: &graphdb.DataBag{
			Strings: map[string]string{
				"cursor": cursor,
			},
			Doubles: map[string]float64{
				"progress": progress,
			},
		},
	})
	return err
}

func (c *Checkpointer) Setup() error {
	c.queue = make(chan checkpointInput, 1024)
	c.onClose = make(chan struct{})
	return nil
}

func (c *Checkpointer) Start() error {
	for {
		select {
		case <-c.onClose:
			return nil
		case in := <-c.queue:
			ctx, cancel := context.WithTimeout(context.Background(), time.Second*3)
			c.Log.Log("checkpoint_into", c.Key, "value", in.cursor)
			err := c.UpdateCursor(ctx, in.cursor, in.progress)
			cancel()
			if err != nil {
				c.Log.Log("err", err, "unable to checkpoint")
			}
		}
	}
}

func (c *Checkpointer) Close() error {
	close(c.onClose)
	return nil
}
