// Get the caller identity so we can parse AWS account number at runtime.
data "aws_caller_identity" "current" {}

// Create an SNS topic for error routing.
resource "aws_sns_topic" "error_topic" {
  name = "${var.name}-tahoe-v2-errors"
}

// Attach a policy to the SNS topic allowing publishes from CloudWatch Events.
resource "aws_sns_topic_policy" "error_policy" {
  arn    = aws_sns_topic.error_topic.arn
  policy = data.aws_iam_policy_document.error_policy_document.json
}

data "aws_iam_policy_document" "error_policy_document" {
  policy_id = "${var.name}-tahoe-v2-errors"

  statement {
    sid = "CWEToSNS"

    actions = [
      "sns:Publish",
    ]

    effect = "Allow"

    principals {
      type        = "Service"
      identifiers = ["events.amazonaws.com"]
    }

    resources = [
      aws_sns_topic.error_topic.arn,
    ]
  }
}

data "template_file" "table_config" {
  template = file("${path.module}/table_config.json.tpl")

  vars = {
    worker_count                 = var.worker_count
    dynamodb_read_capacity_ratio = var.dynamodb_read_capacity_ratio
    dynamodb_splits_count        = var.dynamodb_splits_count
    tahoe_view_name              = var.tahoe_view_name
    schema                       = var.schema
    schema_version               = var.schema_version
    output_fields                = jsonencode(var.output_fields)
  }
}

// Run the db-s3-glue Terraform module.
module "db_s3_glue" {
  source          = "git::ssh://git@git.xarth.tv/dp/db-s3-glue.git?ref=926a2ddddee44e1d528d7ac241c5dec1f1cddeca"
  job_name        = "${var.name}${var.is_qa ? "-qa" : ""}-tahoe-v2"
  qa_for_job_name = var.is_qa ? var.name : ""

  account_number        = data.aws_caller_identity.current.account_id
  cluster_name          = var.name
  s3_output_kms_key_arn = var.s3_output_kms_key_arn
  s3_output_bucket      = "${replace("${var.team}-${var.name}", "_", "-")}-exports-data"
  s3_output_key         = "production"
  s3_script_bucket      = "${replace("${var.team}-${var.name}", "_", "-")}-exports-scripts"
  database_type         = "dynamodb"
  error_sns_topic_name  = aws_sns_topic.error_topic.name
  trigger_schedule      = var.trigger_schedule
  dynamodb_splits_count = var.dynamodb_splits_count

  api_key_kms_key_id     = var.api_key_kms_key_id
  api_key_parameter_name = var.api_key_parameter_name


  // These parameters are set to dummy values for the first run and should be
  // populated after registering your producer.
  tahoe_producer_name = var.tahoe_producer_name

  tahoe_producer_role_arn = var.tahoe_producer_role_arn

  table_config = { "${var.name}" = data.template_file.table_config.rendered }

  spark_cleaning_code = var.cleaning_code

  // These parameters are not required for DynamoDB tables, but they must be
  // passed into the module to avoid runtime errors.

  availability_zone          = "NONE"
  cluster_username           = "NONE"
  db_password_key_id         = "NONE"
  db_password_parameter_name = "NONE"
  rds_subnet_group           = "NONE"
  subnet_id                  = "NONE"
  vpc_id                     = "NONE"

  create_s3_script_bucket = var.create_s3_script_bucket
  create_s3_output_bucket = var.create_s3_output_bucket
}

// Bubble up output variables.
output "s3_kms_key" {
  value = module.db_s3_glue.s3_kms_key
}

output "s3_output_bucket" {
  value = module.db_s3_glue.s3_output_bucket
}

output "s3_output_prefix" {
  value = module.db_s3_glue.s3_output_prefix
}

output "glue_role" {
  value = module.db_s3_glue.glue_role
}

output "data_infra_role" {
  value = module.db_s3_glue.data_infra_role
}
