locals {
  owner = "jlindamo@twitch.tv"
  team  = "feeds"

  // The two regions visage deploys to
  region1 = "us-west-2"
  region2 = "us-east-2"

  service = "graphdb"

  account_ids = {
    integration = "724951484461"
    staging     = "724951484461"
    production  = "914569885343"
  }

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  min_utilizations = {
    integration = 0.1
    staging     = 0.05
    production  = 0.1
  }

  // follows
  follow_read_capacities = {
    integration = 250
    staging     = 1000

    // NOTE: increasing to give Tahoe glue export job sufficient recources:
    // e.g. (80000 max RCU) * (0.1 min_utilization) = 8000 RCU * (0.5 read_ratio) = 4000 usable RCUs for job
    production = 80000
  }

  follow_write_capacities = {
    integration = 50
    staging     = 500
    production  = 14000
  }

  follow_count_read_capacities = {
    integration = 100
    staging     = 1000
    production  = 8000
  }

  follow_list_to_read_capacities = {
    integration = 250
    staging     = 1000
    production  = 16000
  }

  // follows end

  // moderates
  moderates_read_capacities = {
    integration = 100
    staging     = 450
    production  = 6000
  }
  moderates_write_capacities = {
    integration = 10
    staging     = 10
    production  = 100
  }
  moderates_count_read_capacities = {
    integration = 10
    staging     = 50
    production  = 100
  }
  moderates_list_to_read_capacities = {
    integration = 10
    staging     = 50
    production  = 450
  }

  // moderates end

  // vip (@igor)
  vip_read_capacities = {
    integration = 100
    staging     = 500
    production  = 10000
  }
  vip_write_capacities = {
    integration = 100
    staging     = 250
    production  = 1000
  }
  vip_count_read_capacities = {
    integration = 100
    staging     = 500
    production  = 1000
  }
  vip_list_to_read_capacities = {
    integration = 100
    staging     = 500
    production  = 10000
  }

  // vip end

  // feedback (@pwhite)
  feedback_read_capacities = {
    integration = 100
    staging     = 500
    production  = 500
  }
  feedback_write_capacities = {
    integration = 100
    staging     = 500
    production  = 500
  }
  feedback_count_read_capacities = {
    integration = 100
    staging     = 500
    production  = 500
  }
  feedback_list_to_read_capacities = {
    integration = 100
    staging     = 500
    production  = 500
  }

  // feedback end

  // game_follows (@tho)
  game_follows_read_capacities = {
    integration = 100
    staging     = 250
    production  = 3000
  }
  game_follows_write_capacities = {
    integration = 50
    staging     = 500
    production  = 2000
  }
  game_follows_count_read_capacities = {
    integration = 500
    staging     = 1000
    production  = 2500
  }
  game_follows_list_to_read_capacities = {
    integration = 500
    staging     = 500
    production  = 3000
  }

  // feedback end

  // general nodes start
  general_node_read_capacities = {
    integration = 200
    staging     = 1000
    production  = 1000
  }
  general_node_write_capacities = {
    integration = 200
    staging     = 1000
    production  = 1000
  }
  general_node_list_read_capacities = {
    integration = 200
    staging     = 1000
    production  = 1000
  }
  general_node_count_read_capacities = {
    integration = 200
    staging     = 1000
    production  = 1000
  }

  // general nodes end

  // Registry start
  registry_read_capacities = {
    integration = 200
    staging     = 200
    production  = 200
  }
  registry_write_capacities = {
    integration = 200
    staging     = 200
    production  = 200
  }
  registry_list_read_capacities = {
    integration = 200
    staging     = 200
    production  = 200
  }
  registry_count_read_capacities = {
    integration = 200
    staging     = 200
    production  = 200
  }

  // Registry end

  // general edges start
  general_edge_read_capacities = {
    integration = 200
    staging     = 200
    production  = 200
  }
  general_edge_write_capacities = {
    integration = 200
    staging     = 200
    production  = 200
  }
  general_edge_count_read_capacities = {
    integration = 200
    staging     = 200
    production  = 200
  }
  general_edge_list_to_read_capacities = {
    integration = 200
    staging     = 200
    production  = 200
  }

  // end general edges start

  alarm_threshold_percents = {
    integration = 70
    staging     = 70
    production  = 70
  }
  alarm_periods = {
    integration = 60
    staging     = 60
    production  = 60
  }
  dynamodb_autoscale_role_arns = {
    integration = "arn:aws:iam::724951484461:role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com/AWSServiceRoleForApplicationAutoScaling_DynamoDBTable"
    staging     = "arn:aws:iam::724951484461:role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com/AWSServiceRoleForApplicationAutoScaling_DynamoDBTable"
    production  = "arn:aws:iam::914569885343:role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com/AWSServiceRoleForApplicationAutoScaling_DynamoDBTable"
  }
  security_groups = {
    region1 = "${data.terraform_remote_state.account_west.twitch_subnets_sg}"

    //    region2 = "${data.terraform_remote_state.account_east.twitch_subnets_sg_id}"
  }
  subnets = {
    region1 = "${data.terraform_remote_state.account_west.private_subnets}"

    //    region2 = "${join(",", "${data.terraform_remote_state.account_east.private_subnets}")}"
  }
  lambda_arns = {
    region1 = "${data.terraform_remote_state.graphdb_lambda.lambda_arns["us-west-2"]}"

    //    region2 = "${data.terraform_remote_state.graphdb_lambda.lambda_arns["us-east-1"]}"
  }
  prekeys = {
    integration = "dev"
    staging     = "dev"
    production  = "prod"
  }
  prekey                             = "${lookup(local.prekeys, terraform.workspace)}"
  min_utilization                    = "${lookup(local.min_utilizations, terraform.workspace)}"
  follow_read_capacity               = "${lookup(local.follow_read_capacities, terraform.workspace)}"
  follow_write_capacity              = "${lookup(local.follow_write_capacities, terraform.workspace)}"
  follow_count_read_capacity         = "${lookup(local.follow_count_read_capacities, terraform.workspace)}"
  follow_list_to_read_capacity       = "${lookup(local.follow_list_to_read_capacities, terraform.workspace)}"
  moderates_read_capacity            = "${lookup(local.moderates_read_capacities, terraform.workspace)}"
  moderates_write_capacity           = "${lookup(local.moderates_write_capacities, terraform.workspace)}"
  moderates_count_read_capacity      = "${lookup(local.moderates_count_read_capacities, terraform.workspace)}"
  moderates_list_to_read_capacity    = "${lookup(local.moderates_list_to_read_capacities, terraform.workspace)}"
  game_follows_read_capacity         = "${lookup(local.game_follows_read_capacities, terraform.workspace)}"
  game_follows_write_capacity        = "${lookup(local.game_follows_write_capacities, terraform.workspace)}"
  game_follows_count_read_capacity   = "${lookup(local.game_follows_count_read_capacities, terraform.workspace)}"
  game_follows_list_to_read_capacity = "${lookup(local.game_follows_list_to_read_capacities, terraform.workspace)}"
  vip_read_capacity                  = "${lookup(local.vip_read_capacities, terraform.workspace)}"
  vip_write_capacity                 = "${lookup(local.vip_write_capacities, terraform.workspace)}"
  vip_count_read_capacity            = "${lookup(local.vip_count_read_capacities, terraform.workspace)}"
  vip_list_to_read_capacity          = "${lookup(local.vip_list_to_read_capacities, terraform.workspace)}"
  feedback_read_capacity             = "${lookup(local.feedback_read_capacities, terraform.workspace)}"
  feedback_write_capacity            = "${lookup(local.feedback_write_capacities, terraform.workspace)}"
  feedback_count_read_capacity       = "${lookup(local.feedback_count_read_capacities, terraform.workspace)}"
  feedback_list_to_read_capacity     = "${lookup(local.feedback_list_to_read_capacities, terraform.workspace)}"
  general_edge_read_capacity         = "${lookup(local.general_edge_read_capacities, terraform.workspace)}"
  general_edge_write_capacity        = "${lookup(local.general_edge_write_capacities, terraform.workspace)}"
  general_edge_count_read_capacity   = "${lookup(local.general_edge_count_read_capacities, terraform.workspace)}"
  general_edge_list_to_read_capacity = "${lookup(local.general_edge_list_to_read_capacities, terraform.workspace)}"
  general_node_read_capacity         = "${lookup(local.general_node_read_capacities, terraform.workspace)}"
  general_node_write_capacity        = "${lookup(local.general_node_write_capacities, terraform.workspace)}"
  general_node_list_read_capacity    = "${lookup(local.general_node_list_read_capacities, terraform.workspace)}"
  general_node_count_read_capacity   = "${lookup(local.general_node_count_read_capacities, terraform.workspace)}"
  registry_read_capacity             = "${lookup(local.registry_read_capacities, terraform.workspace)}"
  registry_write_capacity            = "${lookup(local.registry_write_capacities, terraform.workspace)}"
  registry_list_read_capacity        = "${lookup(local.registry_list_read_capacities, terraform.workspace)}"
  registry_count_read_capacity       = "${lookup(local.registry_count_read_capacities, terraform.workspace)}"
  alarm_threshold_percent            = "${lookup(local.alarm_threshold_percents, terraform.workspace)}"
  dynamodb_autoscale_role_arn        = "${lookup(local.dynamodb_autoscale_role_arns, terraform.workspace)}"
  alarm_period                       = "${lookup(local.alarm_periods, terraform.workspace)}"
  account_id                         = "${lookup(local.account_ids, terraform.workspace)}"
  profile                            = "${lookup(local.profiles, terraform.workspace)}"
}

// Default provider in first region
provider "aws" {
  profile             = "${local.profile}"
  allowed_account_ids = ["${local.account_id}"]
  region              = "${local.region1}"
}

provider "aws" {
  alias               = "${local.region1}"
  profile             = "${local.profile}"
  allowed_account_ids = ["${local.account_id}"]
  region              = "${local.region1}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/graphdb-dynamodb-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "graphdb_lambda" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/graphdb-lambda.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "account_west" {
  backend = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/feed/terraform/accounts/${local.profile}"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

data "terraform_remote_state" "account_east" {
  backend = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/systems/terraform/${local.prekey}/${local.profile}-east/terraform.tfstate"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

data "terraform_remote_state" "account_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-account.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "account_global" {
  backend   = "s3"
  workspace = "${local.profile}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-global.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "aws_iam_policy_document" "dynamodb_access" {
  statement {
    sid = "VisualEditor0"

    actions = [
      "dynamodb:BatchGetItem",
      "dynamodb:BatchWriteItem",
      "dynamodb:GetItem",
      "dynamodb:PutItem",
      "dynamodb:Query",
      "dynamodb:Scan",
      "dynamodb:UpdateItem",
      "dynamodb:DeleteItem",
      "dynamodb:DescribeTable",
    ]

    resources = [
      "arn:aws:dynamodb:*:${local.account_id}:table/${local.service}_${terraform.workspace}_*",
      "arn:aws:dynamodb:*:${local.account_id}:table/${local.service}_${terraform.workspace}_*/index/*",
    ]
  }
}

data "aws_region" "current" {}

resource "aws_iam_policy" "edge_policy" {
  name        = "table_policy_${local.service}_${terraform.workspace}"
  description = "Allows roles to read/write the edge tables for ${local.service} in ${terraform.workspace}"

  policy = "${data.aws_iam_policy_document.dynamodb_access.json}"
}

module follows_dynamodb {
  source    = "../../../modules/simpleassocdynamodb"
  edge_name = "follows"

  edge_read_capacity    = "${local.follow_read_capacity}"
  edge_write_capacity   = "${local.follow_write_capacity}"
  count_read_capacity   = "${local.follow_count_read_capacity}"
  list_to_read_capacity = "${local.follow_list_to_read_capacity}"

  autoscale_role_arn = "${local.dynamodb_autoscale_role_arn}"
  region1            = "${local.region1}"

  min_utilization = "${local.min_utilization}"

  region2 = "${local.region2}"

  providers = {
    "aws.region1" = "aws.${local.region1}"

    //    "aws.region2" = "aws.${local.region2}"
  }

  stream_lambda_arns = "${local.lambda_arns}"
}

module moderates_dynamodb {
  source    = "../../../modules/simpleassocdynamodb"
  edge_name = "moderates"

  edge_read_capacity    = "${local.moderates_read_capacity}"
  edge_write_capacity   = "${local.moderates_write_capacity}"
  count_read_capacity   = "${local.moderates_count_read_capacity}"
  list_to_read_capacity = "${local.moderates_list_to_read_capacity}"

  autoscale_role_arn = "${local.dynamodb_autoscale_role_arn}"
  region1            = "${local.region1}"
  region2            = "${local.region2}"

  // We special case min_utilization for this table to help run a Redshift load job. Autoscaling makes it
  // difficult to use enough read capacity to complete the job in a reasonable amount of time. In addition, we
  // need additional capacity since the dynamo partitions are heavily skewed because of non-uniform partition keys.
  // See https://jira.twitch.com/browse/SCIENG-2515 for further information.
  min_utilization = "0.33"

  providers = {
    "aws.region1" = "aws.${local.region1}"
  }

  stream_lambda_arns = "${local.lambda_arns}"
}

module "moderates_tahoe_export" {
  source                       = "../../../modules/dynamototahoe"
  team                         = "communications"
  name                         = "${module.moderates_dynamodb.table_name}"
  schema_version               = 3
  dynamodb_read_capacity_ratio = 0.2
  api_key_parameter_name       = "moderatesdbexport-producer-api-key"
  api_key_kms_key_id           = "d896877f-fd04-4c4f-92c8-85baef26524c"
  tahoe_producer_name          = "moderatesdbexport"
  tahoe_producer_role_arn      = "arn:aws:iam::331582574546:role/producer-moderatesdbexport"

  tahoe_view_name = "moderators"

  schema = <<EOF
[
  {
    "name": "fk",
    "type": "string",
    "sensitivity": "userid"
  },
  {
    "name": "to",
    "type": "string",
    "sensitivity": "userid"
  },
  {
    "name": "tk",
    "type": "string",
    "sensitivity": "userid"
  },
  {
    "name": "sk",
    "type": "int",
    "sensitivity": "userid"
  },
  {
    "name": "channel_id",
    "type": "int",
    "is_derived": "true",
    "sensitivity": "userid"
  },
  {
    "name": "moderator_id",
    "type": "int",
    "is_derived": "true",
    "sensitivity": "userid"
  },
  {
    "name": "created_at_utc",
    "type": "timestamp",
    "is_derived": "true"
  }
]
EOF

  output_fields = ["channel_id", "moderator_id", "created_at_utc"]

  // This Python snippet transforms the following schema:
  //   fk             string        user:145671775:moderates
  //   to             string        channel:49717001
  //   tk             string        channel:49717001:moderated_by
  //   sk             float         1553561495803116957
  // To:
  //   channel_id     int           49717001
  //   moderator_id   int           145671775
  //   created_at_utc timestamp     2015-11-14T02:53:45.000
  cleaning_code = <<CODE
if table_name == 'graphdb_production_moderates':
  def reassign(rec):
    channel_prefix = 'channel:'
    user_prefix = 'user:'
    moderates_suffix = ':moderates'

    if rec['to'].startswith(channel_prefix):
      channel_id = rec['to'][len(channel_prefix):]
      rec['channel_id'] = int(channel_id)
    else:
      # Unknown error
      rec['channel_id'] = None

    if rec['fk'].startswith(user_prefix) and rec['fk'].endswith(moderates_suffix):
      moderator_id = rec['fk'][len(user_prefix):-len(moderates_suffix)]
      rec['moderator_id'] = int(moderator_id)
    else:
      # Unknown error
      rec['moderator_id'] = None

    # Truncate timestamp from nanoseconds to seconds.
    # Python doesn't cleanly convert nanosecond timestamps.
    try:
      unix_ts_seconds = rec['sk'] / 1000000000
      rec['created_at_utc'] = datetime.datetime.utcfromtimestamp(unix_ts_seconds)
    except Exception as e:
      rec['created_at_utc'] = None

    return rec
  return Map.apply(frame=frame, f=reassign)
else:
  return frame
CODE
}

output moderates_tahoe_s3_kms_key {
  value = "${module.moderates_tahoe_export.s3_kms_key}"
}

output moderates_tahoe_s3_output_bucket {
  value = "${module.moderates_tahoe_export.s3_output_bucket}"
}

output moderates_tahoe_s3_output_prefix {
  value = "${module.moderates_tahoe_export.s3_output_prefix}"
}

output moderates_tahoe_glue_role {
  value = "${module.moderates_tahoe_export.glue_role}"
}

output moderates_tahoe_data_infra_role {
  value = "${module.moderates_tahoe_export.data_infra_role}"
}

module follows_game_dynamodb {
  source    = "../../../modules/simpleassocdynamodb"
  edge_name = "follows_game"

  edge_read_capacity    = "${local.game_follows_read_capacity}"
  edge_write_capacity   = "${local.game_follows_write_capacity}"
  count_read_capacity   = "${local.game_follows_count_read_capacity}"
  list_to_read_capacity = "${local.game_follows_list_to_read_capacity}"

  autoscale_role_arn = "${local.dynamodb_autoscale_role_arn}"
  region1            = "${local.region1}"
  region2            = "${local.region2}"
  min_utilization    = "${local.min_utilization}"

  providers = {
    "aws.region1" = "aws.${local.region1}"
  }

  stream_lambda_arns = "${local.lambda_arns}"
}

module vip_dynamodb {
  source    = "../../../modules/simpleassocdynamodb"
  edge_name = "vip"

  edge_read_capacity    = "${local.vip_read_capacity}"
  edge_write_capacity   = "${local.vip_write_capacity}"
  count_read_capacity   = "${local.vip_count_read_capacity}"
  list_to_read_capacity = "${local.vip_list_to_read_capacity}"

  autoscale_role_arn = "${local.dynamodb_autoscale_role_arn}"
  region1            = "${local.region1}"
  region2            = "${local.region2}"
  min_utilization    = "${local.min_utilization}"

  providers = {
    "aws.region1" = "aws.${local.region1}"
  }

  stream_lambda_arns = "${local.lambda_arns}"
}

module feedback_dynamodb {
  source    = "../../../modules/simpleassocdynamodb"
  edge_name = "feedback"

  edge_read_capacity    = "${local.feedback_read_capacity}"
  edge_write_capacity   = "${local.feedback_write_capacity}"
  count_read_capacity   = "${local.feedback_count_read_capacity}"
  list_to_read_capacity = "${local.feedback_list_to_read_capacity}"

  autoscale_role_arn = "${local.dynamodb_autoscale_role_arn}"
  region1            = "${local.region1}"
  region2            = "${local.region2}"
  min_utilization    = "${local.min_utilization}"

  providers = {
    "aws.region1" = "aws.${local.region1}"
  }

  stream_lambda_arns = "${local.lambda_arns}"
}

module gen_edges_dynamodb {
  source    = "../../../modules/simpleassocdynamodb"
  edge_name = "gen"

  edge_read_capacity    = "${local.general_edge_read_capacity}"
  edge_write_capacity   = "${local.general_edge_write_capacity}"
  count_read_capacity   = "${local.general_edge_count_read_capacity}"
  list_to_read_capacity = "${local.general_edge_list_to_read_capacity}"
  min_utilization       = "${local.min_utilization}"

  autoscale_role_arn = "${local.dynamodb_autoscale_role_arn}"
  region1            = "${local.region1}"
  region2            = "${local.region2}"

  providers = {
    "aws.region1" = "aws.${local.region1}"

    //    "aws.region2" = "aws.${local.region2}"
  }

  stream_lambda_arns = "${local.lambda_arns}"
}

module generic_nodes_dynamodb {
  source    = "../../../modules/simplenodedynamodb"
  node_name = "gen"

  node_read_capacity  = "${local.general_node_read_capacity}"
  node_write_capacity = "${local.general_node_write_capacity}"
  list_read_capacity  = "${local.general_node_list_read_capacity}"
  count_read_capacity = "${local.general_node_count_read_capacity}"
  min_utilization     = "${local.min_utilization}"

  autoscale_role_arn = "${local.dynamodb_autoscale_role_arn}"
  region1            = "${local.region1}"
  region2            = "${local.region2}"

  providers = {
    "aws.region1" = "aws.${local.region1}"
  }

  stream_lambda_arns = "${local.lambda_arns}"
}

module registry_dynamodb {
  source    = "../../../modules/simplenodedynamodb"
  node_name = "_registry"

  node_read_capacity  = "${local.registry_read_capacity}"
  node_write_capacity = "${local.registry_write_capacity}"
  list_read_capacity  = "${local.registry_list_read_capacity}"
  count_read_capacity = "${local.registry_count_read_capacity}"
  min_utilization     = "${local.min_utilization}"

  autoscale_role_arn = "${local.dynamodb_autoscale_role_arn}"
  region1            = "${local.region1}"
  region2            = "${local.region2}"

  providers = {
    "aws.region1" = "aws.${local.region1}"
  }

  stream_lambda_arns = "${local.lambda_arns}"
}

output dynamodb_iam_policy_arn {
  value = "${aws_iam_policy.edge_policy.arn}"
}

module "consulconf_node_registry" {
  source      = "../../../modules/consulconf"
  team        = "${local.team}"
  service     = "${local.service}"
  environment = "${terraform.workspace}"
  key         = "node_registry_table"
  value       = "${module.registry_dynamodb.node_dynamodb_name}"
}

module "consulconf_node_registry_counts" {
  source      = "../../../modules/consulconf"
  team        = "${local.team}"
  service     = "${local.service}"
  environment = "${terraform.workspace}"
  key         = "node_registry_count_table"
  value       = "${module.registry_dynamodb.node_count_dynamodb_name}"
}
