locals {
  owner   = "logan@twitch.tv"
  team    = "feeds"
  region  = "us-west-2"
  service = "graphdb"

  account_ids = {
    integration = "724951484461"
    staging     = "724951484461"
    production  = "914569885343"
  }

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  min_counts = {
    integration = "0"
    staging     = "0"
    production  = "0"
  }

  log_retentions = {
    integration = "1"
    staging     = "1"
    production  = "30"
  }

  desired_counts = {
    integration = "0"
    staging     = "0"
    production  = "0"
  }

  task_cpus = {
    integration = "1024"
    staging     = "7168" # 1024 * 7
    production  = "14848" # 1024 * 14.5
  }

  task_mems = {
    integration = "512"
    staging     = "1024"
    production  = "4056"
  }

  max_counts = {
    integration = "0"
    staging     = "0"
    production  = "0"
  }

  auto_promotes = {
    integration = "true"
    staging     = "true"
    production  = "false"
  }

  promote_froms = {
    integration = "latest"
    staging     = "integration"
    production  = "canary"
  }

  follow_elasticache_node_types = {
    integration = "cache.t2.small"
    staging     = "cache.r4.4xlarge"
    production  = "cache.r4.8xlarge"
  }

  num_cache_nodes = {
    integration = "2"
    staging     = "2"
    production  = "6"
  }

  mod_elasticache_node_types = {
    integration = "cache.t2.small"
    staging     = "cache.r4.large"
    production  = "cache.r4.4xlarge"
  }

  fulton_account_ids = {
    integration = "468685715363" // twitch-vx-lit+graphdb-beta-us-west-2
    staging     = "468685715363" // twitch-vx-lit+graphdb-beta-us-west-2
    production  = "840080373576" // twitch-vx-lit+graphdb-prod-us-west-2
  }

  cache_invalidation_sns = {
    staging = "arn:aws:sns:us-west-2:468685715363:TwitchVXGraphDBECS-PDX-beta-CacheInvalidationSNS-13F511FF7J3UI"
    production = "arn:aws:sns:us-west-2:840080373576:TwitchVXGraphDBECS-PDX-prod-CacheInvalidationSNS-U7AZTHAG3B0J"
  }

  follow_elasticache_node_type = "${lookup(local.follow_elasticache_node_types, terraform.workspace)}"
  mod_elasticache_node_type    = "${lookup(local.mod_elasticache_node_types, terraform.workspace)}"
  task_cpu                     = "${lookup(local.task_cpus, terraform.workspace)}"
  task_mem                     = "${lookup(local.task_mems, terraform.workspace)}"
  num_cache_node               = "${lookup(local.num_cache_nodes, terraform.workspace)}"
  promote_from                 = "${lookup(local.promote_froms, terraform.workspace)}"
  log_retention                = "${lookup(local.log_retentions, terraform.workspace)}"
  auto_promote                 = "${lookup(local.auto_promotes, terraform.workspace)}"
  min_count                    = "${lookup(local.min_counts, terraform.workspace)}"
  max_count                    = "${lookup(local.max_counts, terraform.workspace)}"
  desired_count                = "${lookup(local.desired_counts, terraform.workspace)}"
  profile                      = "${lookup(local.profiles, terraform.workspace)}"
  sns_can_subscribe_accounts   = ["arn:aws:iam::${lookup(local.fulton_account_ids, terraform.workspace)}:root"]
}

provider "aws" {
  version             = "~> 1.23.0"                                           // require fix for https://github.com/terraform-providers/terraform-provider-aws/issues/2658
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/graphdb-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "api.us-west-2.prod.consul.live-video.a2z.com"
  datacenter = "us-west2"
}

data "terraform_remote_state" "account" {
  backend = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/feed/terraform/accounts/${local.profile}"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

data "terraform_remote_state" "account_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-account.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "region_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-region-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "graphdb_alb" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/graphdb-alb.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "service_dynamodb" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/graphdb-dynamodb-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

resource "aws_iam_role" "dynamodb-fulton-role" {
  name = "dynamodb-fulton-role"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "AWS": "arn:aws:iam::${lookup(local.fulton_account_ids, terraform.workspace)}:root"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

data "aws_iam_policy" "DynamoDBFullAccesswithDataPipeline" {
  arn = "arn:aws:iam::aws:policy/AmazonDynamoDBFullAccesswithDataPipeline"
}

resource "aws_iam_role_policy_attachment" "dynamodbfullaccesswithdatapipeline-role-policy-attach" {
  role       = "${aws_iam_role.dynamodb-fulton-role.name}"
  policy_arn = "${data.aws_iam_policy.DynamoDBFullAccesswithDataPipeline.arn}"
}
