locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  region  = "us-west-2"
  service = "graphdb"

  account_ids = {
    canary = "914569885343"
  }

  profiles = {
    canary = "twitch-feed-aws"
  }

  min_counts = {
    canary = "1"
  }

  desired_counts = {
    canary = "1"
  }

  max_counts = {
    canary = "1"
  }

  auto_promotes = {
    canary = "false"
  }

  log_retentions = {
    canary = "30"
  }

  task_cpus = {
    canary = "12288" # 1024 * 12
  }

  task_mems = {
    canary = "2048"
  }

  promote_froms = {
    canary = "staging"
  }

  shared_froms = {
    canary = "production"
  }

  promote_from  = "${lookup(local.promote_froms, terraform.workspace)}"
  auto_promote  = "${lookup(local.auto_promotes, terraform.workspace)}"
  min_count     = "${lookup(local.min_counts, terraform.workspace)}"
  max_count     = "${lookup(local.max_counts, terraform.workspace)}"
  desired_count = "${lookup(local.desired_counts, terraform.workspace)}"
  log_retention = "${lookup(local.log_retentions, terraform.workspace)}"
  task_cpu      = "${lookup(local.task_cpus, terraform.workspace)}"
  task_mem      = "${lookup(local.task_mems, terraform.workspace)}"
  shared_from   = "${lookup(local.shared_froms, terraform.workspace)}"
  profile       = "${lookup(local.profiles, terraform.workspace)}"
}

provider "aws" {
  version             = "~> 1.23.0"
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/graphdb-shared-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "api.us-west-2.prod.consul.live-video.a2z.com"
  datacenter = "us-west2"
}

data "terraform_remote_state" "account" {
  backend = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/feed/terraform/accounts/${local.profile}"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

data "terraform_remote_state" "account_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "${local.team}/core-account.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "region_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "${local.team}/core-region-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "shared_data" {
  backend   = "s3"
  workspace = "${local.shared_from}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "${local.team}/${local.service}-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "service_dynamodb" {
  backend   = "s3"
  workspace = "${local.shared_from}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "${local.team}/${local.service}-dynamodb-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

module "permissions" {
  source                      = "../../modules/permissions"
  environment                 = "${terraform.workspace}"
  iam_id                      = "${module.service_cluster.iam_id}"
  dynamodb_iam_policy_arn     = "${data.terraform_remote_state.service_dynamodb.dynamodb_iam_policy_arn}"
  access_logs_permissions_arn = "${module.access_logs.iam_policy_arn}"
  async_queue_permissions_arn = "${data.terraform_remote_state.shared_data.async_iam_policy_arn}"
}

module "access_logs" {
  source        = "../../modules/accesslogs"
  log_retention = "${local.log_retention}"
}

module "service_cluster" {
  source           = "../../modules/cluster"
  task_cpu         = "${local.task_cpu}"
  task_mem         = "${local.task_mem}"
  environment      = "${terraform.workspace}"
  target_group_arn = "${data.terraform_remote_state.shared_data.target_group_arn}"
  cluster_name     = "${data.terraform_remote_state.region_core.common_cluster_name}"

  min_size                  = "${local.min_count}"
  desired_count             = "${local.desired_count}"
  max_size                  = "${local.max_count}"
  service_iam_role          = "${data.terraform_remote_state.account_core.service_iam_role}"
  task_autoscale_iam_role   = "${data.terraform_remote_state.account_core.task_autoscale_iam_role}"
  profile                   = "${local.profile}"
  promote_from              = "${local.promote_from}"
  auto_promote              = "${local.auto_promote}"
  deploy_aws_role           = "${data.terraform_remote_state.account_core.build_role}"
  container_loggroup_region = "${data.terraform_remote_state.region_core.container_loggroup_region}"
  container_loggroup        = "${data.terraform_remote_state.region_core.container_loggroup}"
  aws_creds                 = "${data.terraform_remote_state.account_core.jenkins_build_secret}"
  testing_assume_role       = "${data.terraform_remote_state.account_core.build_user_arn}"
}
