locals {
  owner              = "jlindamo@twitch.tv"
  team               = "feeds"
  region             = "us-west-2"
  service            = "graphdb"
  short_service_name = "coh2"

  azs = ["${concat(data.aws_availability_zones.available.names, data.aws_availability_zones.available.names, data.aws_availability_zones.available.names, data.aws_availability_zones.available.names, data.aws_availability_zones.available.names)}"]
}

variable "edge_name" {
  type        = "string"
  description = "Which type of edge to create the cache for"
}

variable private_subnets {
  description = "Which subnets the cache cluster is inside"
}

variable security_group_ids {
  description = "The security groups that the cluster belongs to."
}

variable num_cache_nodes {
  description = "How many cache nodes to create"
}

variable "elasticache_node_type" {
  description = "Which instance types the nodes should belong to"
}

// Returns the list of az's in this datacenter
data "aws_availability_zones" "available" {}

resource "aws_elasticache_subnet_group" "primary_cache" {
  // Note: If the worspace length is less than 5 characters, this will fail.  We'll need to get fancier if that ever happens.
  name        = "${local.service}-${terraform.workspace}-${var.edge_name}"
  description = "Subnet group for memcached cluster for ${local.service} in ${terraform.workspace} for edge types ${var.edge_name}"
  subnet_ids  = ["${split(",", "${var.private_subnets}")}"]
}

resource "aws_elasticache_cluster" "cache_cluster_new" {
  cluster_id = "${local.short_service_name}-${substr(terraform.workspace, 0, 5)}-${var.edge_name}-n"

  engine                       = "memcached"
  engine_version               = "1.4.34"
  node_type                    = "${var.elasticache_node_type}"
  port                         = 11211
  num_cache_nodes              = "${var.num_cache_nodes}"
  parameter_group_name         = "default.memcached1.4"
  security_group_ids           = ["${var.security_group_ids}"]
  subnet_group_name            = "${aws_elasticache_subnet_group.primary_cache.name}"
  preferred_availability_zones = ["${slice(local.azs, 0, var.num_cache_nodes)}"]

  tags {
    Owner       = "${local.owner}"
    Team        = "${local.team}"
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
  }

  lifecycle {
    //    prevent_destroy = true
  }
}

module "consulconf_cache_url" {
  source      = "../consulconf"
  team        = "${local.team}"
  service     = "${local.service}"
  environment = "${terraform.workspace}"
  key         = "cache.${var.edge_name}.config"
  value       = "${aws_elasticache_cluster.cache_cluster_new.configuration_endpoint}"
}
