locals {
  service        = "graphdb"
  container_port = "8000"
  region         = "us-west-2"
  task_cpu       = "${var.task_cpu}"
  task_mem       = "${var.task_mem}"
  team           = "feeds"
}

variable "environment" {}

variable task_cpu {
  default     = "1024"
  description = "How much CPU to reserve for tasks"
}

variable task_mem {
  default     = "1024"
  description = "How much memory (MB) to reserve for tasks"
}

variable profile {}
variable service_iam_role {}
variable "cluster_name" {}
variable "auto_promote" {}
variable "desired_count" {}
variable "max_size" {}
variable "promote_from" {}
variable "min_size" {}
variable aws_creds {}
variable "container_loggroup" {}
variable "container_loggroup_region" {}
variable "target_group_arn" {}
variable "deploy_aws_role" {}
variable "task_autoscale_iam_role" {}
variable "testing_assume_role" {}

module "simpleservice" {
  source            = "../../modules/simpleservice"
  cluster_name     = "${var.cluster_name}"
  team             = "${local.team}"
  environment      = "${var.environment}"
  desired_count    = "${var.desired_count}"
  target_group_arn = "${var.target_group_arn}"
  task_role_arn    = "${aws_iam_role.service_task.arn}"
  container_port   = "${local.container_port}"
  service_iam_role = "${var.service_iam_role}"
  service          = "${local.service}"
}

module "taskcpuscale" {
  source                  = "../../modules/taskcpuscale"
  max_size                = "${var.max_size}"
  min_size                = "${var.min_size}"
  cluster_name            = "${var.cluster_name}"
  service                 = "${module.simpleservice.service_name}"
  environment             = "${var.environment}"
  task_autoscale_iam_role = "${var.task_autoscale_iam_role}"
  scale_up_cpu            = "50"
  scale_down_cpu          = "20"
}

module "pipeline" {
  source          = "../pipeline"
  team            = "${local.team}"
  environment     = "${var.environment}"
  aws_creds       = "${var.aws_creds}"
  profile         = "${var.profile}"
  promote_from    = "${var.promote_from}"
  auto_promote    = "${var.auto_promote}"
  service         = "${local.service}"
  deploy_aws_role = "${var.deploy_aws_role}"
  region          = "${local.region}"
  service_name    = "${module.simpleservice.service_name}"
  cluster_name    = "${var.cluster_name}"

  data = "${map(
    "task_role_arn", "${aws_iam_role.service_task.arn}",
    "container_port", "${local.container_port}",
    "task_name", "${local.service}",
    "task_cpu", "${local.task_cpu}",
    "task_mem", "${local.task_mem}",
    "container_loggroup", "${var.container_loggroup}",
    "container_loggroup_region", "${var.container_loggroup_region}",
    "task_family", "${module.simpleservice.task_family}"
  )}"
}

### ----- task level iam policies -----
resource "aws_iam_role" "service_task" {
  name               = "${local.team}-${local.service}-${var.environment}"
  assume_role_policy = "${data.aws_iam_policy_document.service_task.json}"
  description        = "Allows access to ${local.service} resources in ${var.environment}"
}

data "aws_iam_policy_document" "service_task" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }

  # This allows integration tests to run from the build user.
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "AWS"
      identifiers = ["${var.testing_assume_role}"]
    }
  }
}

output iam_id {
  value = "${aws_iam_role.service_task.id}"
}

output task_role_arn {
  value = "${aws_iam_role.service_task.arn}"
}

output task_family {
  value = "${module.simpleservice.task_family}"
}

output service_name {
  value = "${module.simpleservice.service_name}"
}
