locals {
  vartype      = "${var.index_name == "" ? "table" : "index"}"
  index_suffix = "${var.index_name == "" ? "" : "/index/${var.index_name}"}"
}

variable "autoscale_role_arn" {
  type        = "string"
  description = "Which ARN allows autoscale events"
}

variable "table_name" {
  type        = "string"
  description = "Which table name to scale"
}

variable "target_utilization" {
  type        = "string"
  description = "% of utilization to target scaling for"
  default     = "70"
}

variable "min_read_capacity" {
  description = "The least amount of read capacity to use"
}

variable "max_read_capacity" {
  description = "The most amount of read capacity to use"
}

variable "min_write_capacity" {
  description = "The least amount of write capacity to use"
}

variable "max_write_capacity" {
  description = "The most amount of write capacity to use"
}

variable "index_name" {
  description = "Optional. Index name to scale.  If empty, works on the table itself"
  default     = ""
}

// Autoscale reads
resource "aws_appautoscaling_target" "dynamodb_table_read_target" {
  max_capacity       = "${var.max_read_capacity}"
  min_capacity       = "${var.min_read_capacity}"
  resource_id        = "table/${var.table_name}${local.index_suffix}"
  role_arn           = "${var.autoscale_role_arn}"
  scalable_dimension = "dynamodb:${local.vartype}:ReadCapacityUnits"
  service_namespace  = "dynamodb"
}

resource "aws_appautoscaling_policy" "dynamodb_table_read_policy" {
  name               = "DynamoDBReadCapacityUtilization:${aws_appautoscaling_target.dynamodb_table_read_target.resource_id}"
  policy_type        = "TargetTrackingScaling"
  resource_id        = "${aws_appautoscaling_target.dynamodb_table_read_target.resource_id}"
  scalable_dimension = "${aws_appautoscaling_target.dynamodb_table_read_target.scalable_dimension}"
  service_namespace  = "${aws_appautoscaling_target.dynamodb_table_read_target.service_namespace}"

  target_tracking_scaling_policy_configuration {
    predefined_metric_specification {
      predefined_metric_type = "DynamoDBReadCapacityUtilization"
    }

    target_value = "${var.target_utilization}"
  }
}

// Autoscale writes
resource "aws_appautoscaling_target" "dynamodb_table_write_target" {
  max_capacity       = "${var.max_write_capacity}"
  min_capacity       = "${var.min_write_capacity}"
  resource_id        = "table/${var.table_name}${local.index_suffix}"
  role_arn           = "${var.autoscale_role_arn}"
  scalable_dimension = "dynamodb:${local.vartype}:WriteCapacityUnits"
  service_namespace  = "dynamodb"
}

resource "aws_appautoscaling_policy" "dynamodb_table_write_policy" {
  name               = "DynamoDBWriteCapacityUtilization:${aws_appautoscaling_target.dynamodb_table_write_target.resource_id}"
  policy_type        = "TargetTrackingScaling"
  resource_id        = "${aws_appautoscaling_target.dynamodb_table_write_target.resource_id}"
  scalable_dimension = "${aws_appautoscaling_target.dynamodb_table_write_target.scalable_dimension}"
  service_namespace  = "${aws_appautoscaling_target.dynamodb_table_write_target.service_namespace}"

  target_tracking_scaling_policy_configuration {
    predefined_metric_specification {
      predefined_metric_type = "DynamoDBWriteCapacityUtilization"
    }

    target_value = "${var.target_utilization}"
  }
}
